//*********************************************************************************
//**
//** Project.........: AD8307 based RF Power Meter
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-04-01, Loftur Jonasson, TF3LJ
//**
#define                VERSION "1.01"   // 2012-04-14
//**
//** History.........: Check the PM.c file
//**
//*********************************************************************************

#ifndef _TF3LJ_PM_H_
#define _TF3LJ_PM_H_ 1

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include <avr/power.h>
#include <rprintf.h>							// AVRLIB (not AVRLibc) functions
#include <lcd.h>	


//-----------------------------------------------------------------------------
// Features Selection
//-----------------------------------------------------------------------------  
// EEPROM settings Serial Number. Increment this number when firmware mods necessitate
// fresh "Factory Default Settings" to be forced into the EEPROM at first boot after
// an upgrade
#define COLDSTART_REF		0x03// When started, the firmware examines this "Serial Number
								// and enforce factory reset if there is a mismatch.
								// This is useful if the EEPROM structure has been modified
//-----------------------------------------------------------------------------
// LED Blink
//
// None, or only one of the four should be selected
#define FAST_LOOP_THRU_LED	0	// Blink PB2 LED every time, when going through the mainloop *OR*
#define	MS_LOOP_THRU_LED	1	// Blink PB2 LED every 1ms, when going through the mainloop *OR*
#define	MED_LOOP_THRU_LED	0	// Blink PB2 LED every 10ms, when going through the mainloop *OR*
#define SLOW_LOOP_THRU_LED	0	// Blink PB2 LED every 100ms, when going through the mainloop


//-----------------------------------------------------------------------------
//Defs for AD8307 Calibration (dBm *10)
#define	CAL1_NOR_VALUE		   0// Default for AD8717 at   0 dBm
#define	CAL1_RAW_DEFAULT	 537// Default for AD8717 at   0 dBm
#define	CAL2_NOR_VALUE		-400// Default for AD8717 at -40 dBm
#define	CAL2_RAW_DEFAULT	 285// Default for AD8717 at -40 dBm

//-----------------------------------------------------------------------------
//Defs for default Gain Settings (dB *10)
#define	GAINSET1			-400// 40 dB Attenuator
#define	GAINSET2			-700// 15 dB Attenuator
#define	GAINSET3			 180// 18.0 dB Gain (amplifier)

//-----------------------------------------------------------------------------
//DEFS for LED
#define LED_PORT			PORTD		// port for the LED
#define LED_DDR				DDRD		// port for the LED
#define LED					(1<<6)		// pin for LED


//-----------------------------------------------------------------------------
//DEFS for LCD Display
// The below #defines have been moved from the lcd.h file
#define LCD_PORT			PORTC		// port for the LCD lines
#define LCD_DATA0_PORT		LCD_PORT	// port for 4bit data bit 0
#define LCD_DATA1_PORT		LCD_PORT	// port for 4bit data bit 1
#define LCD_DATA2_PORT		LCD_PORT	// port for 4bit data bit 2
#define LCD_DATA3_PORT		LCD_PORT	// port for 4bit data bit 3
#define LCD_DATA0_PIN		4			// pin  for 4bit data bit 0
#define LCD_DATA1_PIN		5			// pin  for 4bit data bit 1
#define LCD_DATA2_PIN		6			// pin  for 4bit data bit 2
#define LCD_DATA3_PIN		7			// pin  for 4bit data bit 3
#define LCD_RS_PORT			PORTC		// port for RS line
#define LCD_RS_PIN			0			// pin  for RS line
#define LCD_RW_PORT			PORTC		// port for RW line
#define LCD_RW_PIN			1			// pin  for RW line
#define LCD_E_PORT			PORTC		// port for Enable line
#define LCD_E_PIN			2			// pin  for Enable line


//-----------------------------------------------------------------------------
// A couple of #defines moved from lcd.h:
#define LCD_LINES			2			// number of visible lines of the display 
#define LCD_DISP_LENGTH		16			// visibles characters per line of the display


//-----------------------------------------------------------------------------
// DEFS for the Rotary Encoder VFO function
// Configuration of the two input pins, Phase A and Phase B
// They can be set up to use any pin on two separate input ports
#define ENC_A_PORT		PORTD				// PhaseA port register
#define ENC_A_DDR		DDRD				// PhaseA port direction register
#define ENC_A_PORTIN	PIND				// PhaseA port input register
#define ENC_A_PIN		(1 << 7)			// PhaseA port pin
#define ENC_B_PORT		PORTD				// PhaseB port register
#define ENC_B_DDR		DDRD				// PhaseB port direction register
#define ENC_B_PORTIN	PIND				// PhaseB port input register
#define ENC_B_PIN		(1 << 5)			// PhaseB port pin

#define	ENC_RES_DIVIDE		1				// Default reduction of the Encoder Resolution
#define ENCODER_DIR_REVERSE	0	// Reverse the direction of the Rotary Encoder


//-----------------------------------------------------------------------------
// Definitions for the Pushbutton Encoder functionality
#define ENC_PUSHB_PORT		PORTD
#define ENC_PUSHB_DDR		DDRD
#define	ENC_PUSHB_INPORT	PIND
#define	ENC_PUSHB_PIN		(1 << 4)		// PD4

#define ENC_PUSHB_MIN		1				// Min pushdown for valid push (x 10ms)
#define	ENC_PUSHB_MAX		10				// Min pushdown for memory save (x 10ms)


//-----------------------------------------------------------------------------
// Bargraph bounds
#define BAR_MIN_VALUE		800	// -80dB
#define BAR_FULL_RANGE		1000// 100dB full range
#define	BAR_FINE_RES		160	// 16dB for full scale variation if in fine resolution mode
#define BAR_LENGTH			16	// 16 characters long


//-----------------------------------------------------------------------------
// Select Bargraph display style
#define BARGRAPH_STYLE_1	1	// Used if LCD bargraph alternatives.  N8LP LP-100 look alike bargraph         *OR*
#define BARGRAPH_STYLE_2	0	// Used if LCD bargraph alternatives.  Bargraph with level indicators          *OR*
#define BARGRAPH_STYLE_3	0	// Used if LCD bargraph alternatives.  Another bargraph with level indicators  *OR*
#define BARGRAPH_STYLE_4	0	// Used if LCD bargraph alternatives.  Original bargraph, Empty space enframed *OR*
#define BARGRAPH_STYLE_5	0	// Used if LCD bargraph alternatives.  True bargraph, Empty space is empty


//
//-----------------------------------------------------------------------------
// Miscellaneous software defines, functions and variables
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Flags

// DEFS for all kinds of Flags
extern uint8_t			Status;
#define ENC_CHANGE		(1 << 0)			// Indicates that Encoder value has been modified
#define SHORT_PUSH		(1 << 1)			// Short Push Button Action
#define	LONG_PUSH		(1 << 2)			// Long Push Button Action
#define BARGRAPH_CAL	(1 << 3)			// 16dB Bargraph has been recentred
#define MODE_CHANGE		(1 << 4)			// Display mode has changed
#define MODE_DISPLAY	(1 << 5)			// Display mode has changed

// Operation Mode Flags
extern	uint8_t			Menu_Mode;			// Which Menu Mode is active
#define POWER_DB		(1 << 0)
#define	POWER_W			(1 << 1)
#define VOLTAGE			(1 << 2)
#define BARGRAPH_FULL	(1 << 3)
#define BARGRAPH_16dB	(1 << 4)
#define MIXED			(1 << 5)
#define	CONFIG			(1 << 6)
//
#define	DEFAULT_MODE 	(1 << 0)			// Default Menu Mode

// Bool stuff
#define	TRUE		1
#define FALSE		0


//-----------------------------------------------------------------------------
// Structures and Unions

typedef struct {
	int16_t	db10m;							// Calibrate, value in dBm x 10
	int16_t	ad;								// corresponding A/D value
} cal_t;

typedef struct 
{
		uint8_t		EEPROM_init_check;		// If value mismatch,
											// then update EEPROM to factory settings
		int16_t		encoderRes;				// (initial) Encoder resolution
		uint8_t		which_gainset;			// which gain setting is selected?
		int16_t		gainset[4];				// first one always zero, other 3: -70 to +70dB
		cal_t		calibrate[2];			// 2 Calibration points
} var_t;


//-----------------------------------------------------------------------------
// Global variables
extern	EEMEM 		var_t E;				// Default Variables in EEPROM
extern				var_t R;				// Runtime Variables in Ram

extern 	int16_t		ad8307_ad;				// Measured A/D value from the AD8307
extern 	double 		ad8307_real;			// Calculated power measured by the AD8307
extern	double		power_db;				// Calculated power in dBm
extern	double		power_db_pep;			// Calculated PEP power in dBm
extern	double		power_mw;				// Calculated power in mW
extern	double		power_mw_pep;			// Calculated PEP power in mW
extern	int8_t		modulation_index;		// AM modulation index in %
extern	int16_t		power_snapshot_db10;	// A pushbutton snapshot of measured power, used for bargraph_16db

extern	int16_t		encOutput;				// Output From Encoder


//-----------------------------------------------------------------------------
// Prototypes for functions
// PM.c

// PM_Encoder.c
extern void 		encoder_Init(void);				// Initialise the Rotary Encoder
extern void			encoder_Scan(void);				// Scan the Rotary Encoder

// Push Button and Rotary Encoder Menu functions
extern void			PushButtonMenu(void);

// PM_Display_Functions.c
extern void			lcd_display_power_db(void);
extern void			lcd_display_power_w(void);
extern void			lcd_display_voltage(void);
extern void			lcd_display_bargraph_full(void);
extern void			lcd_display_bargraph_16db(void);
extern void			lcd_display_mixed(void);
extern void			usb_send_data(void);			// Print stuff on USB bus

// LCD Bargraph stuff
extern void 		lcdProgressBar(uint16_t progress, uint16_t maxprogress, uint8_t length);
extern void			lcd_bargraph_init(void);

#endif
