//*********************************************************************************
//**
//** Project.........: AD8307 based RF Power Meter
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-04-01, Loftur Jonasson, TF3LJ
//**
//**
//** History.........: Check the PM.c file
//**
//*********************************************************************************

//
// This code makes use of the LUFA library for USB connectivity
//
/*
             LUFA Library
     Copyright (C) Dean Camera, 2012.

  dean [at] fourwalledcubicle [dot] com
           www.lufa-lib.org
*/

/*
  Copyright 2012  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

#include 	<math.h>
#include	"PM.h"
#include 	"USB/USBtoSerial.h"				// LUFA include



//
//-----------------------------------------------------------------------------------------
// 			Throw a character onto the USB bus queue (128 char ringbuffer)
//-----------------------------------------------------------------------------------------
//
void usb_send_char(uint8_t input)
{
	// LUFA USB library function, used with the USBtoSerial stuff
	RingBuffer_Insert(&USARTtoUSB_Buffer, input);
}


//
//-----------------------------------------------------------------------------
//			Print dBm, accepts 10x dBm input value
//-----------------------------------------------------------------------------
//
void print_dbm(int16_t db10m)
{
	// Add Gain in the selected External Gain Preset (Attenuator is negative)
	db10m = db10m - R.gainset[R.which_gainset];

	int16_t pwrdb_tenths = db10m;

	if (pwrdb_tenths < 0) pwrdb_tenths *= -1;
	int16_t pwrdb = pwrdb_tenths / 10;
	pwrdb_tenths = pwrdb_tenths % 10;

	if (db10m <= -100)
	{
		rprintf("-%2u.%1udBm",pwrdb,pwrdb_tenths);
	}
	else if (db10m < 0)
	{
		rprintf(" -%1u.%1udBm",pwrdb,pwrdb_tenths);
	}
	else rprintf("%3u.%1udBm",pwrdb,pwrdb_tenths);
}


//
//-----------------------------------------------------------------------------
//			Print Power, input value is in milliWatts
//-----------------------------------------------------------------------------
//
void print_p_mw(double mw)
{
	uint32_t p_calc;
	uint16_t power_sub, power;
	int16_t gain;


	// Add Gain in the selected External Gain Preset (Attenuator is negative)
	gain = R.gainset[R.which_gainset];

	mw = mw * pow(10,-gain/100.0);

	if(mw >= 10000.0)
	{
		p_calc = mw;
		power = p_calc / 1000;
		power_sub = ((p_calc % 1000)+50)/100;
		rprintf("%2u.%01uW",power, power_sub);
	}
	else if(mw >= 1000.0)
	{
		p_calc = mw;
		power = p_calc / 1000;
		power_sub = ((p_calc % 1000)+5)/10;
		rprintf("%1u.%02uW",power, power_sub);
	}
	else if(mw >= 100.0)
	{
		rprintf("%4umW",(uint16_t)mw);
	}
	else if(mw >= 10.0)
	{
		p_calc = mw * 10;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01umW",power, power_sub);
	}
	else if(mw >= 1.0)
	{
		p_calc = mw * 100;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02umW",power, power_sub);
	}
	else if(mw >= 0.1)
	{
		power = mw * 1000;
		rprintf("%4uuW",power);
	}
	else if(mw >= 0.01)
	{
		p_calc = mw * 10000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01uuW",power, power_sub);
	}
	else if(mw >= 0.001)
	{
		p_calc = mw * 100000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02uuW",power, power_sub);
	}
	else if(mw >= 0.0001)
	{
		power = mw * 1000000;
		rprintf("%4unW",power);
	}
	else if(mw >= 0.00001)
	{
		p_calc = mw * 10000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01unW",power, power_sub);
	}
	else if(mw >= 0.000001)
	{
		p_calc = mw * 100000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02unW",power, power_sub);
	}
	else if(mw >= 0.0000001)
	{
		power = mw * 1000000000;
		rprintf("%4upW",power);
	}
	else if(mw >= 0.00000001)
	{
		p_calc = mw * 10000000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01upW",power, power_sub);
	}
	else if(mw >= 0.000000001)
	{
		p_calc = mw * 100000000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02upW",power, power_sub);
	}
	else if(mw >= 0.0000000001)
	{
		power = mw * 1000000000000;
		rprintf("%4ufW",power);
	}
	else if(mw >= 0.00000000001)
	{
		p_calc = mw * 10000000000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01ufW",power, power_sub);
	}
	else
	{
		p_calc = mw * 100000000000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02ufW",power, power_sub);
	}
}


//
//-----------------------------------------------------------------------------
//			Print Voltage, input value in volts
//-----------------------------------------------------------------------------
//
void print_v(double voltage)
{
	uint16_t power_sub, power;
	int16_t gain = 0;

	// Add Gain in the selected External Gain Preset (Attenuator is negative)
	gain = R.gainset[R.which_gainset];

	voltage = voltage * pow(10,-gain/200.0);

	if(voltage >= 10.0)
	{
		power_sub = voltage * 10;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01uV",power, power_sub);
	}	
	else if(voltage >= 1.0)
	{
		power_sub = voltage * 100;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%2u.%02uV",power, power_sub);
	}
	else if(voltage >= 0.1)
	{
		power = voltage * 1000;
		rprintf("%4umV",power);
	}
	else if(voltage >= 0.01)
	{
		power_sub = voltage * 10000;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01umV",power, power_sub);
	}
	else if(voltage >= 0.001)
	{
		power_sub = voltage * 100000;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%1u.%02umV",power, power_sub);
	}
	else if(voltage >= 0.0001)
	{
		power = voltage * 1000000;
		rprintf("%4uuV",power);
	}
	else if(voltage >= 0.00001)
	{
		power_sub = voltage * 10000000;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01uuV",power, power_sub);
	}
	else
	{
		power_sub = voltage * 100000000;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%1u.%02uuV",power, power_sub);
	}
}


//
//-----------------------------------------------------------------------------
//			Display Power in dBm
//-----------------------------------------------------------------------------
//
void lcd_display_power_db(void)
{
	static uint8_t mode_timer = 0;

	lcd_clrscr();

	//------------------------------------------
	// Display mode intro for one second
	if(Status & MODE_DISPLAY)
	{
		if(Status & MODE_CHANGE)
		{
			Status &= ~MODE_CHANGE;			// Clear display change mode
			mode_timer = 0;					// New mode, reset timer
		}
		lcd_gotoxy(0,0);
		rprintf("Display Power in");
		lcd_gotoxy(0,1);
		rprintf("dB milliWatts ->");
		mode_timer++;
		if (mode_timer == 10)
		{
			mode_timer = 0;
			Status &= ~MODE_DISPLAY;		// Clear display change mode
		}
	}
	else
	{
		//------------------------------------------
		// Power indication, dBm average
		lcd_gotoxy(0,0);
		print_dbm((int16_t) (power_db*10.0));
		rprintf(" avg");

		//------------------------------------------
		// Power indication, dBm pep
		lcd_gotoxy(0,1);
		print_dbm((int16_t) (power_db_pep*10.0));
		rprintf(" pep");

		//------------------------------------------
		// Amplitude Modulation Index
		lcd_gotoxy(14,0);
		rprintf("AM");
		lcd_gotoxy(12,1);
		rprintf("%3d",modulation_index);
		rprintfChar('%');
	}
}


//
//-----------------------------------------------------------------------------
//			Display Power in Watts
//-----------------------------------------------------------------------------
//
void lcd_display_power_w(void)
{
	static uint8_t mode_timer = 0;

	lcd_clrscr();

	//------------------------------------------
	// Display mode intro for one second
	if(Status & MODE_DISPLAY)
	{
		if(Status & MODE_CHANGE)
		{
			Status &= ~MODE_CHANGE;			// Clear display change mode
			mode_timer = 0;					// New mode, reset timer
		}
		lcd_gotoxy(0,0);
		rprintf("Display Power in");
		lcd_gotoxy(0,1);
		rprintf("Watts         ->");
		mode_timer++;
		if (mode_timer == 10)
		{
			mode_timer = 0;
			Status &= ~MODE_DISPLAY;		// Clear display change mode
		}
	}
	else
	{
		//------------------------------------------
		// Power indication, average
		lcd_gotoxy(0,0);
		// Wattage Printout)
		print_p_mw(power_mw);
		rprintf(" avg");

		//------------------------------------------
		// Power indication, 1 second PEP
		lcd_gotoxy(0,1);
		// Wattage Printout)
		print_p_mw(power_mw_pep);
		rprintf(" pep");

		//------------------------------------------
		// Amplitude Modulation Index
		lcd_gotoxy(14,0);
		rprintf("AM");
		lcd_gotoxy(12,1);
		rprintf("%3d",modulation_index);
		rprintfChar('%');
	}
}

//
//-----------------------------------------------------------------------------
//			Display Voltage (RMS and Peak-to-Peak) in 50 ohm
//-----------------------------------------------------------------------------
//
void lcd_display_voltage(void)
{
	double 	voltage;
	static uint8_t mode_timer = 0;

	lcd_clrscr();

	//------------------------------------------
	// Display mode intro for one second
	if(Status & MODE_DISPLAY)
	{
		if(Status & MODE_CHANGE)
		{
			Status &= ~MODE_CHANGE;			// Clear display change mode
			mode_timer = 0;					// New mode, reset timer
		}
		lcd_gotoxy(0,0);
		rprintf("Display average");
		lcd_gotoxy(0,1);
		rprintf("Voltage (50ohm)>");
		mode_timer++;
		if (mode_timer == 10)
		{
			mode_timer = 0;
			Status &= ~MODE_DISPLAY;		// Clear display change mode
		}
	}
	else
	{
		//------------------------------------------
		// Voltage (in 50 ohms) indication, RMS
		lcd_gotoxy(0,0);
		voltage = sqrt(power_mw*50/1000.0);
		print_v(voltage);
		rprintf(" rms");

		//------------------------------------------
		// Voltage (in 50 ohms) indication, Vpeak-to-peak
		lcd_gotoxy(0,1);
		voltage = sqrt(power_mw*50/1000.0) * 2 * sqrt(2);
		print_v(voltage);
		rprintf(" p-to-p");
	}
}


//
//-----------------------------------------------------------------------------
//			Display Bargraph meter
//-----------------------------------------------------------------------------
//
void lcd_display_bargraph_full(void)
{
	int16_t	db10;
	static uint8_t mode_timer = 0;

	db10 = (int16_t) (power_db*10.0);
	
	lcd_clrscr();

	//------------------------------------------
	// Display mode intro for one second
	if(Status & MODE_DISPLAY)
	{
		if(Status & MODE_CHANGE)
		{
			Status &= ~MODE_CHANGE;			// Clear display change mode
			mode_timer = 0;					// New mode, reset timer
		}
		lcd_gotoxy(0,0);
		rprintf("Bargraph Meter->");
		lcd_gotoxy(0,1);
		rprintf("(average power)");
		mode_timer++;
		if (mode_timer == 10)
		{
			mode_timer = 0;
			Status &= ~MODE_DISPLAY;		// Clear display change mode
		}
	}
	else
	{
		//------------------------------------------
		// Power indication, dBm average
		lcd_gotoxy(0,0);
		print_dbm(db10);

		//------------------------------------------
		// Power indication, average
		lcd_gotoxy(9,0);
		print_p_mw(power_mw);

		//------------------------------------------
		// Bargraph
		lcd_gotoxy(0,1);
		lcdProgressBar((db10 + BAR_MIN_VALUE), BAR_FULL_RANGE, BAR_LENGTH);
	}
}


//
//-----------------------------------------------------------------------------
//			Display Bargraph meter, +/- 8dB fullscale
//-----------------------------------------------------------------------------
//
void lcd_display_bargraph_16db(void)
{
	int16_t	db10;
	int16_t bar_input;
	static uint8_t mode_timer = 0;

	db10 = (int16_t) (power_db*10.0);
	
	lcd_clrscr();

	//------------------------------------------
	// Display mode intro for one second
	if(Status & MODE_DISPLAY)
	{
		if(Status & MODE_CHANGE)
		{
			Status &= ~MODE_CHANGE;			// Clear display change mode
			mode_timer = 0;					// New mode, reset timer
		}
		lcd_gotoxy(0,0);
		rprintf("+/- 8dB Bargraph");
		lcd_gotoxy(0,1);
		rprintf("Push to Center->");
		mode_timer++;
		if (mode_timer == 10)
		{
			mode_timer = 0;
			Status &= ~MODE_DISPLAY;		// Clear display change mode
		}
	}
	else
	{
		//------------------------------------------
		// Power indication, dBm average
		lcd_gotoxy(0,0);
		print_dbm(db10);

		//------------------------------------------
		// Power indication, average
		lcd_gotoxy(9,0);
		print_p_mw(power_mw);

		//------------------------------------------
		// Bargraph +/-8dB fullscale
		bar_input = db10 - power_snapshot_db10 + BAR_FINE_RES/2;	// Calculate bar length
		if (bar_input < 0) bar_input = 0;							// Set bounds
		if (bar_input > BAR_FINE_RES) bar_input = BAR_FINE_RES;

		lcd_gotoxy(0,1);											// Display the bargraph
		lcdProgressBar(bar_input, BAR_FINE_RES, BAR_LENGTH);
	}
}


//
//-----------------------------------------------------------------------------
//			Display a Mixed bag of irrelevant stuff
//-----------------------------------------------------------------------------
//
void lcd_display_mixed(void)
{
	double 	output_voltage;
	uint16_t power, power_sub;
	
	lcd_clrscr();

	lcd_gotoxy(0,0);
	rprintf("Dbg:");

	//------------------------------------------
	// AD8307 voltage indication
	lcd_gotoxy(4,0);
	output_voltage = ad8307_ad * 4 / 1000.0;
	power_sub = output_voltage * 1000;
	power = power_sub / 1000;
	power_sub = power_sub % 1000;
	rprintf("%u.%03uV %4u", power, power_sub, ad8307_ad);	

	//------------------------------------------
	// Calibrate 1
	lcd_gotoxy(0,1);
	//	print_dbm(R.calibrate[0].db10m);
	rprintf ("%3d",R.calibrate[0].db10m);
	rprintf ("%4u",R.calibrate[0].ad);

	//------------------------------------------
	// Calibrate 2
	//	print_dbm(R.calibrate[1].db10m);
	lcd_gotoxy(8
	,1);
	rprintf ("%4d",R.calibrate[1].db10m);
	rprintf ("%4u",R.calibrate[1].ad);
}


//
//-----------------------------------------------------------------------------------------
// 			Send AD8307 measurement data to the Computer
//-----------------------------------------------------------------------------------------
//
void usb_send_data(void)
{
//	uint8_t i = 0;

	// Init AVRLIB rprintf() against the USB
	rprintfInit(usb_send_char);

	//------------------------------------------
	// Power indication, dBm average (1 second window)
	print_dbm((int16_t) (power_db*10.0));
	rprintf("_AVG, ");

	//------------------------------------------
	// Power indication, dBm pep (1 second window)
	print_dbm((int16_t) (power_db_pep*10.0));
	rprintf("_PEP, ");

	//------------------------------------------
	// Power indication, average (1 second window)
	print_p_mw(power_mw);
	rprintf("_AVG, ");

	//------------------------------------------
	// Power indication, 1 second PEP
	print_p_mw(power_db_pep);
	rprintf("_PEP, ");

	//------------------------------------------
	// Amplitude Modulation Index
	rprintf("%3u",modulation_index);
	rprintfChar('%');
	rprintf("_AM\r\n");

	// Return the AVRLIB rprintf() to the LCD
	rprintfInit(lcd_data);
}

