//*********************************************************************************
//**
//** Project.........: AD8307 based RF Power Meter
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-04-01, Loftur Jonasson, TF3LJ
//**
//** History.........: ...
//**
//*********************************************************************************

#include "PM.h"
#include "lcd.h"
#include <stdio.h>


// First Level Menu Items
const uint8_t level0_menu_size = 6;
const char *level0_menu_items[] =
				{  "1-Inp Gain Sel",
				   "2-Gain Adjust",
				   "3-Calibrate",
				   "4-Disp Debug",
				 //"8-Adj Encoder Res",
				   "9-FactoryReset",
				   "0-Exit" };

// Flag for Debug Screen
#define DEBUG_MENU		1

// Flag for Reference 1 Menu Selection
//#define POW_MENU		2

// Flag for Set Gain values
#define GAIN_SEL_MENU		6
const uint8_t gain_sel_size = 6;
const char *gain_sel_items[] =
				{  "1-No Gain",
				   "2-GainPreset 1",
				   "3-GainPreset 2",
				   "4-GainPreset 3",
				   "9-Go Back",
				   "0-Exit"};

// Flag for Set Gain values
#define GAIN_MENU		7
const uint8_t gain_menu_size = 5;
const char *gain_menu_items[] =
				{  "1-GainPreset 1",
				   "2-GainPreset 2",
				   "3-GainPreset 3",
				   "9-Go Back",
				   "0-Exit"};

// Flags for Gain Submenu functions
#define GAIN_SET1_MENU	701
#define GAIN_SET2_MENU	702
#define GAIN_SET3_MENU	703

// Flag for Encoder Resolution Change
//#define ENCODER_MENU	8

// Flag for Calibrate menu
#define CAL_MENU		9
const uint8_t calibrate_menu_size = 4;
const char *calibrate_menu_items[] =
				{  "1-First (dBm)",
				   "2-Second (dBm)",
				   "9-Go Back",
				   "0-Exit"};

// Flags for Calibrate Submenu functions
#define CAL_SET1_MENU	901
#define CAL_SET2_MENU	902

// Flag for Factory Reset
#define FACTORY_MENU	10
// Factory Reset menu Items
const uint8_t factory_menu_size = 3;
const char *factory_menu_items[] =
				{  "1-Yes - Reset",
				   "9-No - Go back",
				   "0-No - Exit"};


uint16_t		menu_level = 0;						// Keep track of which menu we are in
uint8_t			menu_data = 0;						// Pass data to lower menu

int8_t			gain_selection;						// keep track of which GainPreset is currently selected


//----------------------------------------------------------------------
// Display a Menu of choices, one line at a time
//
// **menu refers to a pointer array containing the Menu to be printed
//
// menu_size indicates how many pointers (menu items) there are in the array
//
// current_choice indicates which item is currently up for selection if
// pushbutton is pushed
//
// begin row & begin_col are the coordinates for the upper lefthand corner
// of the three or four lines to be printed
//
//----------------------------------------------------------------------
void lcd_scroll_Menu(char **menu, uint8_t menu_size,
		uint8_t current_choice, uint8_t begin_row, uint8_t begin_col)
{
	uint8_t a;


	// Clear LCD from begin_col to end of line.
	lcd_gotoxy(begin_col, begin_row);
	for (a = begin_col; a < 16; a++)
		rprintfChar(' ');

	// Using Menu list pointed to by **menu, preformat for print:
	// First line contains previous choice, secon line contains
	// current choice preceded with a '->', and third line contains
	// next choice
	lcd_gotoxy(begin_col, begin_row);
	rprintf("->%s", *(menu + current_choice));
}


//----------------------------------------------------------------------
// Menu functions begin:
//----------------------------------------------------------------------


//--------------------------------------------------------------------
// Debug Screen, exit on push
//--------------------------------------------------------------------
void debug_menu(void)
{
	lcd_display_mixed();		// Display diverse debug stuff

	// Exit on Button Push
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;	// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Nothing Changed");
		_delay_ms(500);
		Menu_Mode |= CONFIG;	// We're NOT done, just backing off
		menu_level = 0;			// We are done with this menu level
	}
}



//--------------------------------------------------------------------
// Gain Preset Select Menu functions
//--------------------------------------------------------------------
void gain_select_menu(void)
{
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			gain_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			gain_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = gain_sel_size;
		while(gain_selection >= menu_size)
			gain_selection -= menu_size;
		while(gain_selection < 0)
			gain_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)gain_sel_items, menu_size, gain_selection,0, 0);

		// Indicate Currently selected Gain Preset
		lcd_gotoxy(0,1);
		rprintf("Select->   ");

		if (gain_selection < 4)
		{

			int16_t value=0;

			switch (gain_selection)
			{
				case 0:
					value = 0;	// No gain
					break;
				default:
					value = R.gainset[gain_selection];
					break;
			}

			// Format and print current value
			int16_t val_sub = value;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			if (value < 0)
			{
				val*=-1;
				val_sub*=-1;
				rprintf("-%1u.%01u",val, val_sub);
			}
			else
			{
				rprintf("%2u.%01u",val, val_sub);
			}
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				

		if (gain_selection < 4)			// Gain Preset was selected
		{
			R.which_gainset = gain_selection;
			eeprom_write_block(&R.which_gainset, &E.which_gainset, sizeof (R.which_gainset));
			rprintf("Value Stored");
		}
		else rprintf("Nothing Changed");

		_delay_ms(500);

		if (gain_selection == 5)		// Exit selected
		{
			Menu_Mode &= ~CONFIG;		// We're done, EXIT
		}
		else Menu_Mode |= CONFIG;		// We're NOT done, just backing off

		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}

//--------------------------------------------------------------------
// Gain Preset Submenu functions
//--------------------------------------------------------------------
void gain_menu_level2(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	if      (menu_level == GAIN_SET1_MENU) current_selection = R.gainset[1];
	else if (menu_level == GAIN_SET2_MENU) current_selection = R.gainset[2];
	else if (menu_level == GAIN_SET3_MENU) current_selection = R.gainset[3];


	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		int16_t max_value = 700;		// Highest Gain value in dB * 10
		int16_t min_value = -700;		// Lowest Gain value in dB * 10
		if(current_selection > max_value) current_selection = max_value;
		if(current_selection < min_value) current_selection = min_value;

		// Store Current value in running storage
		if      (menu_level == GAIN_SET1_MENU) R.gainset[1] = current_selection;
		else if (menu_level == GAIN_SET2_MENU) R.gainset[2] = current_selection;
		else if (menu_level == GAIN_SET3_MENU) R.gainset[3] = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	

		if (menu_level == GAIN_SET1_MENU)
		{
			rprintf("Gain Preset 1:");
		}
		else if (menu_level == GAIN_SET2_MENU)
		{
			rprintf("Gain Preset 2:");
		}
		else if (menu_level == GAIN_SET3_MENU)
		{
			rprintf("Gain Preset 3:");
		}
		lcd_gotoxy(0,1);
		rprintf("Adjust>  ");
		// Format and print current value
		int16_t val_sub = current_selection;
		int16_t val = val_sub / 10;
		val_sub = val_sub % 10;
		if (current_selection < 0)
		{
			val*=-1;
			val_sub*=-1;
			rprintf("-%1u.%01udB",val, val_sub);
		}
		else
		{
			rprintf(" %2u.%01udB",val, val_sub);
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		if (menu_level == GAIN_SET1_MENU)
		{
			eeprom_write_block(&R.gainset[1], &E.gainset[1], sizeof (R.gainset[1]));
		}
		else if (menu_level == GAIN_SET2_MENU)
		{
			eeprom_write_block(&R.gainset[2], &E.gainset[2], sizeof (R.gainset[2]));
		}
		else if (menu_level == GAIN_SET3_MENU)
		{
			eeprom_write_block(&R.gainset[3], &E.gainset[2], sizeof (R.gainset[3]));
		}

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = GAIN_MENU;			// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}



//--------------------------------------------------------------------
// Gain Menu functions
//--------------------------------------------------------------------
void gain_set_menu(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = gain_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)gain_menu_items, menu_size, current_selection,1, 0);

		// Indicate Current value stored under the currently selected Gain Preset
		// The "stored" value indication changes according to which Preset is currently selected.
		lcd_gotoxy(0,0);				
		rprintf("AdjustGain:");
		if (current_selection < 3)
		{
			int16_t value;

			value = R.gainset[current_selection+1];

			int16_t val_sub = value;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;

			// Print value of currently indicated gain Preset
			if (value < 0)
			{
				val*=-1;
				val_sub*=-1;
				rprintf("-%1u.%01u",val, val_sub);
			}
			else
			{
				rprintf(" %2u.%01u",val, val_sub);
			}
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0:
				menu_level = GAIN_SET1_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 1:
				menu_level = GAIN_SET2_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 2:
				menu_level = GAIN_SET3_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 3:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Gain");
				_delay_ms(500);
				Menu_Mode |= CONFIG;	// We're NOT done, just backing off
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Gain");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}

/*
//--------------------------------------------------------------------
// Rotary Encoder Resolution
//--------------------------------------------------------------------
void encoder_menu(void)
{

	uint8_t	current_selection;			// Keep track of current Encoder Resolution

	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.encoderRes;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection=current_selection<<1;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection=current_selection>>1;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > 128) current_selection = 128;
		if(current_selection < 1) current_selection = 1;

		// Store Current value in running storage
		R.encoderRes = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	
		rprintf("Encoder ResDivide:");

		lcd_gotoxy(0,1);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcd_gotoxy(0,3);
		rprintf("->");

		int16_t val = current_selection;
		rprintf("%3u",val);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.encoderRes, &E.encoderRes, sizeof (R.encoderRes));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}
*/

//--------------------------------------------------------------------
// Calibrate Submenu functions
//--------------------------------------------------------------------
void calibrate_menu_level2(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	if      (menu_level == CAL_SET1_MENU) current_selection = R.calibrate[0].db10m;
	else if (menu_level == CAL_SET2_MENU) current_selection = R.calibrate[1].db10m;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		int16_t max_value = 100;		// Highest permissible Calibration value in dB * 10
		int16_t min_value = -500;		// Lowest permissible Calibration value in dB * 10
		if(current_selection > max_value) current_selection = max_value;
		if(current_selection < min_value) current_selection = min_value;

		// Store Current value in running storage
		if      (menu_level == CAL_SET1_MENU) R.calibrate[0].db10m = current_selection;
		else if (menu_level == CAL_SET2_MENU) R.calibrate[1].db10m = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	

		if (menu_level == CAL_SET1_MENU)
		{
			rprintf("1st CalSetPoint:");
		}
		else if (menu_level == CAL_SET2_MENU)
		{
			rprintf("2nd CalSetPoint:");
		}

		lcd_gotoxy(0,1);
		rprintf("Adjust>");
		// Format and print current value
		int16_t val_sub = current_selection;
		int16_t val = val_sub / 10;
		val_sub = val_sub % 10;
		if (current_selection < 0)
		{
			val*=-1;
			val_sub*=-1;
			rprintf(" -%1u.%01udBm",val, val_sub);
		}
		else
		{
			rprintf(" %2u.%01udBm",val, val_sub);
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		if (menu_level == CAL_SET1_MENU)
		{
			R.calibrate[0].ad = ad8307_ad;
			eeprom_write_block(&R.calibrate[0].ad, &E.calibrate[0].ad, sizeof (R.calibrate[0].ad));
			eeprom_write_block(&R.calibrate[0].db10m, &E.calibrate[0].db10m, sizeof (R.calibrate[0].db10m));
		}
		else if (menu_level == CAL_SET2_MENU)
		{
			R.calibrate[1].ad = ad8307_ad;
			eeprom_write_block(&R.calibrate[1].ad, &E.calibrate[1].ad, sizeof (R.calibrate[1].ad));
			eeprom_write_block(&R.calibrate[1].db10m, &E.calibrate[1].db10m, sizeof (R.calibrate[1].db10m));
		}


		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = CAL_MENU;			// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}



//--------------------------------------------------------------------
// Calibrate Menu functions
//--------------------------------------------------------------------
void calibrate_menu(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = calibrate_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)calibrate_menu_items, menu_size, current_selection,1, 0);

		// Indicate Current value stored under the currently selected GainPreset
		// The "stored" value indication changes according to which GainPreset is currently selected.
		lcd_gotoxy(0,0);				
		rprintf("Calibrate");
		if (current_selection < 2)
		{
			int16_t value=0;

			switch (current_selection)
			{
				case 0:
					value = R.calibrate[0].db10m;
					break;
				case 1:
					value = R.calibrate[1].db10m;
					break;
			}
			int16_t val_sub = value;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;

			// Print value of currently indicated reference
			lcd_gotoxy(11,0);
			if (value < 0)
			{
				val*=-1;
				val_sub*=-1;
				rprintf("-%1u.%01u",val, val_sub);
			}
			else
			{
				rprintf("%2u.%01u",val, val_sub);
			}
		}
		else
		{
			lcd_gotoxy(11,0);
			rprintf(" --");
		}
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0:
				menu_level = CAL_SET1_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 1:
				menu_level = CAL_SET2_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 2:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Cal");
				_delay_ms(500);
				Menu_Mode |= CONFIG;	// We're NOT done, just backing off
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Cal");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//--------------------------------------------------------------------
// Factory Reset with all default values
//--------------------------------------------------------------------
void factory_menu(void)
{
	static int8_t	current_selection;
	static uint8_t	LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
		}
		else if (encOutput < 0)
		{
			current_selection--;
		}
  	  	// Reset data from Encoder
		Status &=  ~ENC_CHANGE;
		encOutput = 0;

		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = factory_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("All to default?");

		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)factory_menu_items, menu_size, current_selection, 1, 0);
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Factory Reset
				// Force an EEPROM update:
				eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
				lcd_clrscr();
				lcd_gotoxy(0,0);				
				rprintf("Factory Reset");
				lcd_gotoxy(0,1);
				rprintf("All default");
				while (1);			// Bye bye, Death by Watchdog
			case 1:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode |= CONFIG;// We're NOT done, just backing off
				menu_level = 0;		// We are done with this menu level
				LCD_upd = FALSE;	// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//
//--------------------------------------------------------------------
// Manage the first level of Menus
//--------------------------------------------------------------------
//
void menu_level0(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t	LCD_upd = FALSE;	// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	if (LCD_upd == FALSE)				// Need to update LCD
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = level0_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("Config Menu:");

		// Print the Menu
		lcd_scroll_Menu((char**)level0_menu_items, menu_size, current_selection,1, 0);
	}

	if (Status & SHORT_PUSH)
	{
		Status &= ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Select gain Preset (attenuator is negative)
				menu_level = GAIN_SEL_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				// Jump into the sub-menu indicating the currently selected gain Preset
				gain_selection = R.which_gainset; 
				break;

			case 1: // Adjust Gain compensation
				menu_level = GAIN_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 2: // Calibrate
				menu_level = CAL_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

//			case 3:// Encoder Resolution
//				menu_level = ENCODER_MENU;
//				LCD_upd = FALSE;		// force LCD reprint
//				break;

			case 3: // Display Debug stuff
				menu_level = DEBUG_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 4: // Factory Reset
				menu_level = FACTORY_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			default:
				// Exit
				lcd_clrscr();
				lcd_gotoxy(1,1);
		   		rprintf("Return from Menu");
				Menu_Mode &=  ~CONFIG;	// We're done
				LCD_upd = FALSE;		// Make ready for next time
		}
	}
}


//
//--------------------------------------------------------------------
// Scan the Configuraton Menu Status and delegate tasks accordingly
//--------------------------------------------------------------------
//
void PushButtonMenu(void)
{
	// Select which menu level to manage
	if (menu_level == 0) menu_level0();

	else if (menu_level == GAIN_SEL_MENU) gain_select_menu();

	else if (menu_level == GAIN_MENU) gain_set_menu();
	else if (menu_level == GAIN_SET1_MENU) gain_menu_level2();
	else if (menu_level == GAIN_SET2_MENU) gain_menu_level2();
	else if (menu_level == GAIN_SET3_MENU) gain_menu_level2();

	else if (menu_level == CAL_MENU) calibrate_menu();
	else if (menu_level == CAL_SET1_MENU) calibrate_menu_level2();
	else if (menu_level == CAL_SET2_MENU) calibrate_menu_level2();

	//else if (menu_level == ENCODER_MENU) encoder_menu();

	else if (menu_level == DEBUG_MENU) debug_menu();

	else if (menu_level == FACTORY_MENU) factory_menu();
}
