//*********************************************************************************
//**
//** Project.........: AD8307 based RF Power Meter
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-04-01, Loftur Jonasson, TF3LJ / VE2LJX
//**
//** Current version.: See PM.h
//**
//** History.........: ...
//**
//*********************************************************************************

//
// This code makes use of the LUFA library for USB connectivity
//
/*
             LUFA Library
     Copyright (C) Dean Camera, 2012.

  dean [at] fourwalledcubicle [dot] com
           www.lufa-lib.org
*/

/*
  Copyright 2012  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/


//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Microcontroller Pin assignment (defined in the PM.h file):
//
// PC0 = LCD control RS
// PC1 = LCD control RW
// PC2 = LCD control E
// PC3
// PC4 = LCD D4
// PC5 = LCD D5
// PC6 = LCD D6
// PC7 = LCD D7
//
// PF0 = AD MUX0, input from AD8307
//
// PD4 = Push Button Selector input
// PD5 = Rotary Encoder A input
// PD6 = LED output 
// PD7 = Rotary Encoder B input
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------


#include "PM.h"
#include "USB/USBtoSerial.h"				// LUFA include
#include "analog.h"
#include <math.h>

EEMEM		var_t	E;						// Variables in eeprom (user modifiable, stored)
			var_t	R						// Variables in ram/flash rom (default)
					=
					{
						COLDSTART_REF		// Update into eeprom if value mismatch
					, 	ENC_RES_DIVIDE		// Initial Encoder Resolution
					,	0					// Which GainPreset is selected (0,1,2,3) 0=0 or None
					, { 0 					// Gainset 0 always = 0
					, GAINSET1				// Three gain settings in dB x 10
					, GAINSET2              // Attenuators take a negative value
					, GAINSET3 }
					,{{ CAL1_NOR_VALUE
					   ,CAL1_RAW_DEFAULT}	// First Calibrate point, db*10 + AD value
					, { CAL2_NOR_VALUE
					   ,CAL2_RAW_DEFAULT}}	// Second Calibrate point, db*10 + AD value
					, USB_DATA				// Default USB Serial Data out setting
					};

uint8_t		Status = 0;						// Contains various status flags
uint8_t		Menu_Mode;						// Menu Mode Flags

int16_t		ad8307_ad;						// Measured A/D value from the AD8307 as a 10 bit value
											// Resolution is Vref/1024, or 4.0mV, when 
											// a reference of 4.096V is used.

double		power_db;						// Calculated power in dBm
double		power_db_pep;					// Calculated PEP power in dBm
double		power_mw;						// Calculated power in mW
double		power_mw_pep;					// Calculated PEP power in mW
int8_t		modulation_index;				// AM modulation index in %
double		ad8307_real;					// Measured AD8307 power in dBm
int16_t		power_snapshot_db10;			// A pushbutton snapshot of measured power, used for bargraph_16db

const		uint8_t aref = 0;				// External 4.096V reference is connected


//
//-----------------------------------------------------------------------------------------
// 			Read ADC Mux
//-----------------------------------------------------------------------------------------
//
int16_t adc_Read(uint8_t mux)
{
    uint8_t low;

    ADCSRA = (1<<ADEN) | ADC_PRESCALER;             // enable ADC
    ADCSRB = (1<<ADHSM) | (mux & 0x20);             // high speed mode
    ADMUX = aref | (mux & 0x1F);                    // configure mux input
    ADCSRA = (1<<ADEN) | ADC_PRESCALER | (1<<ADSC); // start the conversion
    while (ADCSRA & (1<<ADSC)) ;                    // wait for result
    low = ADCL;                                     // must read LSB first
    return (ADCH << 8) | low;                       // must read MSB only once!
}


//
//-----------------------------------------------------------------------------------------
// 			Convert Voltage Reading into Power
//-----------------------------------------------------------------------------------------
//
void measure_Power(void)
{
	double 	delta_db;
	int16_t	delta_ad;
	double	delta_ad1db;

	// Calculate the slope gradient between the two calibration points:
	//
	// (dB_Cal1 - dB_Cal2)/(V_Cal1 - V_Cal2) = slope_gradient
	//
	delta_db = (double)((R.calibrate[1].db10m - R.calibrate[0].db10m)/10.0);
	delta_ad = R.calibrate[1].ad - R.calibrate[0].ad;
	delta_ad1db = delta_db/delta_ad;

	//
	// measured current dB value is then: (V - V_Cal1) * slope_gradient + dB_Cal1
	//
	ad8307_real = (ad8307_ad - R.calibrate[0].ad) * delta_ad1db + R.calibrate[0].db10m/10.0;
}


//
//-----------------------------------------------------------------------------------------
// 			Calculate all kinds of power
//-----------------------------------------------------------------------------------------
//
void calc_Power(void)
{
	#define	BUFFER	200					// Max/Min/Average Buffer size

	// For measurement of peak and average power
	static int16_t p_avg[BUFFER];		// One second window
	static uint16_t i;

	int16_t	p_inst;						// Instantaneous power (dB * 100)
	int16_t	p_min, p_max;				// Keep track of Min and Max values
	double	v_min, v_max, v_avg;

	p_inst = 100*ad8307_real;

	// Find peaks and averages (done with voltages rather than dB)
	p_avg[i] = p_inst;					// Store dB value in ring buffer
	i++;
	if (i == BUFFER) i = 0;

	p_min =  32767;						// Maybe a bit excessive as max/min in dB*100 :)
	p_max = -32768;
	for (uint16_t j = 0; j < BUFFER; j++)// Retrieve the min/max values out of the measured window
	{
		if (p_min > p_avg[j]) p_min = p_avg[j];
		if (p_max < p_avg[j]) p_max = p_avg[j];
	}

	// Calculate max/min and average voltage
	v_min = pow(10,p_min/100.0/20.0);	// Normalize dB*100 and convert to Voltage
	v_max = pow(10,p_max/100.0/20.0);
	v_avg = (v_min + v_max) / 2.0;		// Average voltage in the presence of modulation

	// Average power
	power_db = 20 * log10(v_avg);

	// Average power, milliwatts
	power_mw = pow(10,power_db/10.0);			// Power in mW

	// PEP (1 second)
	power_db_pep = 20 * log10(v_max);
	power_mw_pep = pow(10,power_db_pep/10.0);	// Power in mW

	// Amplitude Modulation index
	modulation_index = (int8_t) (100.0 * (v_max-v_avg)/v_avg);
}


//
//-----------------------------------------------------------------------------------------
// Top level task
// runs in an endless loop
//-----------------------------------------------------------------------------------------
//
void maintask(void)
{
	// Now we can do all kinds of business, such as measuring the AD8307 voltage output, 
	// scanning Rotary Encoder, updating LCD etc...
	static uint16_t lastIteration, lastIteration1, lastIteration2;	// Counters to keep track of time
	uint16_t Timer1val, Timer1val1, Timer1val2;		// Timers used for 100ms and 10ms polls
	static uint8_t pushcount=0;						// Measure push button time (max 2.5s)
	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be run through as often as possible
	// currently measured to be approximately once every 25 - 50 us
	//-------------------------------------------------------------------------------
	encoder_Scan();									// Scan the Rotary Encoder

	#if FAST_LOOP_THRU_LED							// Blink a LED every time when going through the main loop 
	LED_PORT = LED_PORT ^ LED;						// Blink a led
	#endif
	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every approx 5ms
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val1 = TCNT1/328; // get current Timer1 value, changeable every ~5ms
	Timer1val1 = TCNT1/313;   // get current Timer1 value, changeable every ~5ms
	
	if (Timer1val1 != lastIteration1)				// Once every 5ms, do stuff
	{
		lastIteration1 = Timer1val1;				// Make ready for next iteration
		#if MS_LOOP_THRU_LED						// Blink LED every 5*2 ms, when going through the main loop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif

		ad8307_ad = adc_Read(0);					// Measure voltage from AD8307
		measure_Power();							// Convert to Power in dBm
		calc_Power();								// Calculate Power, includes a 1 second
													// sliding window of the last 200 samples
	}

	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/100th of a second (10ms)
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val2 = TCNT1/656; // get current Timer1 value, changeable every ~1/100th sec
	Timer1val2 = TCNT1/626;   // get current Timer1 value, changeable every ~1/100th sec
	if (Timer1val2 != lastIteration2)				// Once every 1/100th of a second, do stuff
	{
		lastIteration2 = Timer1val2;				// Make ready for next iteration

		#if MED_LOOP_THRU_LED						// Blink LED every 10ms, when going through the main loop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif
		
		// Nothing here
	}

	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/10th of a second
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val = TCNT1/6554; // get current Timer1 value, changeable every ~1/10th sec
	Timer1val = TCNT1/6253;   // get current Timer1 value, changeable every ~1/10th sec
	if (Timer1val != lastIteration)	// Once every 1/10th of a second, do stuff
	{
		lastIteration = Timer1val;					// Make ready for next iteration

		#if SLOW_LOOP_THRU_LED						// Blink LED every 100ms, when going through the main loop 
		LED_PORT = LED_PORT ^ LED;					// Blink a led
		#endif

		//-------------------------------------------------------------------
		// Read Encoder to cycle back and forth through modes
		//
		static int8_t current_mode = 0;			// Which display mode is active?
		#define MAX_MODES 4						// Number of available modes minus one

		// If the encoder was used while not in config mode:
		if ((!(Menu_Mode & CONFIG)) && (Status & ENC_CHANGE))
		{
			Status |= MODE_CHANGE + MODE_DISPLAY;// Used with LCD Display functions	
			
			// Mode switching travels only one click at a time, ignoring extra clicks
			if (encOutput > 0)
			{
				current_mode++;
				if (current_mode > MAX_MODES) current_mode = 0;
	    		// Reset data from Encoder
				Status &=  ~ENC_CHANGE;
				encOutput = 0;
			}
			else if (encOutput < 0)
			{
				current_mode--;
				if (current_mode < 0) current_mode = MAX_MODES;	
	  		  	// Reset data from Encoder
				Status &=  ~ENC_CHANGE;
				encOutput = 0;
			}
			switch (current_mode)
			{
					case 0:
						Menu_Mode = POWER_DB;
						break;
					case 1:
						Menu_Mode = POWER_W;
						break;
					case 2:
						Menu_Mode = VOLTAGE;
						break;
					case 3:
						Menu_Mode = BARGRAPH_FULL;
						break;
					case 4:
						Menu_Mode = BARGRAPH_16dB;
						break;
			}
		}
		
		//-------------------------------------------------------------------
		// Read Pushbutton state
		//
		// Enact Long Push (pushbutton has been held down for a certain length of time):
		if (pushcount >= ENC_PUSHB_MAX)				// "Long Push", goto Configuration Mode
		{
			Menu_Mode |= CONFIG;					// Switch into Configuration Menu, while
													// retaining memory of runtime function

			Status |= LONG_PUSH;					// Used with Configuration Menu functions	
			pushcount = 0;							// Initialize push counter for next time
		}
		// Enact Short Push (react on release if only short push)
		else if (ENC_PUSHB_INPORT & ENC_PUSHB_PIN) 	// Pin high = just released, or not pushed
		{
			// Do nothing if this is a release after Long Push
			if (Status & LONG_PUSH)					// Is this a release following a long push?
			{
					Status &= ~LONG_PUSH;			// Clear pushbutton status
			}
			// Do stuff on command
			else if (pushcount >= ENC_PUSHB_MIN)	// Check if this is more than a short spike
			{	
				if (Menu_Mode & CONFIG)
					Status |= SHORT_PUSH;			// Used with Configuration Menu functions	

				else
				{
					//
					// Various things to be done if short push... depending on which mode is active
					//
					// Only one thing enabled so far...
					power_snapshot_db10 = (int16_t) (power_db*10.0); // Used to center bargraph_16db
				} 
			}
			pushcount = 0;							// Initialize push counter for next time
		}
		else if (!(Status & LONG_PUSH))				// Button Pushed, count up the push timer
		{											// (unless this is tail end of a long push,
			pushcount++;							//  then do nothing)
		}

		//-------------------------------------------------------------------
		// Various Menu (rotary encoder) selectable display/function modes
		//
		if (Menu_Mode & CONFIG)					// Pushbutton Configuration Menu
		{
			PushButtonMenu();
		}	
		else if (Menu_Mode == POWER_DB)			// Power Meter in dBm
		{
			lcd_display_power_db();
		}
		else if (Menu_Mode == POWER_W)			// Power Meter in Watts
		{
			lcd_display_power_w();
		}
		else if (Menu_Mode == VOLTAGE)			// Power meter with Volt over 50 ohm
		{
			lcd_display_voltage();
		}
		else if (Menu_Mode == BARGRAPH_FULL)	// Bargraph meter
		{
			lcd_display_bargraph_full();
		}
		else if (Menu_Mode == BARGRAPH_16dB)	// Bargraph meter, 16dB full scale
		{
			lcd_display_bargraph_16db();
		}

		if (R.USB_data_out)						// Transmit data to serial port if USB Serial Data
		{
			usb_send_data();					// Send AD8307 measurement data to the Computer
		}		
	}
	wdt_reset();								// Whoops... must remember to reset that running watchdog
}


//
//-----------------------------------------------------------------------------------------
// 			Setup Ports, timers, start the works and never return, unless reset
//								by the watchdog timer
//						then - do everything, all over again
//-----------------------------------------------------------------------------------------
//
int main(void)
{
	MCUSR &= ~(1 << WDRF);							// Disable watchdog if enabled by bootloader/fuses
	wdt_disable();

	clock_prescale_set(clock_div_1); 				// with 16MHz crystal this means CLK=16000000

	//------------------------------------------
	// 16-bit Timer1 Initialization
	TCCR1A = 0; //start the timer
	TCCR1B = (1 << CS12); // prescale Timer1 by CLK/256
	// 16000000 Hz / 256 = 62500 ticks per second
	// 16-bit = 2^16 = 65536 maximum ticks for Timer1
	// 65536 / 62500 = ~1.05 seconds
	// so Timer1 will overflow back to 0 about every 1 seconds
	// Timer1val = TCNT1; // get current Timer1 value

	//------------------------------------------
	// Init and set output for LED
	LED_DDR = LED;
	LED_PORT = 0;

	//------------------------------------------
	// Init Pushbutton input
	ENC_PUSHB_DDR = ENC_PUSHB_DDR & ~ENC_PUSHB_PIN;	// Set pin for input
	ENC_PUSHB_PORT= ENC_PUSHB_PORT | ENC_PUSHB_PIN;	// Set pull up

	//------------------------------------------
	// Set run time parameters to Factory default under certain conditions
	//
	// Enforce "Factory default settings" when firmware is run for the very first time after
	// a fresh firmware installation with a new "serial number" in the COLDSTART_REF #define
	// This may be necessary if there is garbage in the EEPROM, preventing startup
	// To activate, roll "COLDSTART_REF" Serial Number in the PM.h file
	if (eeprom_read_byte(&E.EEPROM_init_check) != R.EEPROM_init_check)
	{
		eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
	}
	else
	{
		eeprom_read_block(&R, &E, sizeof(E));		// Load the persistent data from eeprom
	}

   	lcd_init(LCD_DISP_ON);							// Init the LCD

	rprintfInit(lcd_data);							// Init AVRLIB rprintf() against the LCD

	// Initialize the LCD bargraph, load the bargraph custom characters
	lcd_bargraph_init();

	//------------------------------------------
	// LCD Print Version information (5 seconds)
	lcd_clrscr();
	lcd_gotoxy(0,0);
	lcd_puts_P("AD8307 based");
	_delay_ms(500);
	lcd_gotoxy(2,1);
	lcd_puts_P("RF Power Meter");
	_delay_ms(1000);

	lcd_gotoxy(0,0);
	lcd_puts_P("Intelligent...");
	_delay_ms(500);
	lcd_gotoxy(2,1);
	lcd_puts_P("...Power Meter");
	_delay_ms(2000);

	lcd_clrscr();
	lcd_gotoxy(0,0);
	lcd_puts_P("TF3LJ / VE2LJX");
	lcd_gotoxy(11,1);
	rprintf("V%s", VERSION);
	_delay_ms(2000);

	if (R.USB_data_out)								// Enumerate USB serial port, if USB Serial Data enabled
	{
		lufa_init();								// Init USB-serial service
	}
	
	wdt_enable(WDTO_1S);							// Start the Watchdog Timer, 1 sec

	encoder_Init();									// Init Rotary encoder

	Menu_Mode = DEFAULT_MODE;						// Power Meter Mode is normal default

	// Start the works, we're in business
	while (1)
	{
		maintask();									// Do useful stuff
		lufa_manage_serial();						// Report to computer (if it is listening)
	}
}
