//*********************************************************************************
//**
//** Project.........: Read Hand Sent Morse Code (tolerant of considerable jitter)
//**
//** Copyright (c) 2016  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** The GNU General Public License is available at
//** http://www.gnu.org/licenses/
//**
//** Substantive portions of the methodology used here to decode Morse Code are found in:
//**
//** "MACHINE RECOGNITION OF HAND-SENT MORSE CODE USING THE PDP-12 COMPUTER"
//** by Joel Arthur Guenther, Air Force Institute of Technology,
//** Wright-Patterson Air Force Base, Ohio
//** December 1973
//** http://www.dtic.mil/dtic/tr/fulltext/u2/786492.pdf
//**
//** Platform........: Teensy 3.1 / 3.2 and the Teensy Audio Shield
//**                   & a 160x128 TFT LCD
//**
//** Initial version.: 0.00, 2016-01-25  Loftur Jonasson, TF3LJ / VE2LJX
//**
//*********************************************************************************

#include "CWReceive.h"
#include "CW_Key_Picture.h"
#include <Audio.h>
#include <SPI.h>
#include <Metro.h>
#include <TimerOne.h>

volatile bool        fft256sel    = FALSE;          // Normally 1024 bin fft selected
volatile int16_t     fftbuf[60];                    // FFT output buffer. Updated by SignalSampler() Interrupt function.
volatile int16_t     vol          = 0;              // Audio volume to Interrupt function
volatile double      agcvol       = 1.0;            // AGC adjusted volume, Max 1.0.  Updated by SignalSampler()
volatile int16_t     peakFrq      = 700;            // Audio peak tone frequency in Hz
volatile int16_t     thresh       = 1;              // Audio threshold level (0 - 40)
volatile bool        state;                         // Current decoded signal state
volatile sigbuf      sig[SIG_BUFSIZE];              // A circular buffer of decoded input levels and durations, input from
                                                    // SignalSampler().  Used by CW Decode functions
volatile int32_t     sig_lastrx   = 0;              // Circular buffer in pointer, updated by SignalSampler
         int32_t     sig_incount  = 0;              // Circular buffer in pointer, copy of sig_lastrx, used by CW Decode functions
         int32_t     sig_outcount = 0;              // Circular buffer out pointer, used by CW Decode functions
volatile int32_t     sig_timer    = 0;              // Elapsed time of current signal state, in units of the
                                                    // FFT conversion time. approx 2.9ms for FFT256 with no averaging
                                                    // or 11.6ms for FFT1024 with no averaging.  Updated by SignalSampler.
volatile int32_t      timer_stepsize=1;             // Step size of signal timer depends on FFT conversion time, FFT1024=4
int32_t               cur_time;                     // copy of sig_timer
int32_t               cur_outcount = 0;             // Basically same as sig_outcount, for Error Correction functionality
int32_t               last_outcount= 0;             // sig_outcount for previous character, used for Error Correction func
           
sigbuf                data[DATA_BUFSIZE];           // Buffer containing decoded dot/dash and time information
                                                    // for assembly into a character
uint8_t               data_len     = 0;             // Length of incoming character data
                                        
char                  code[DATA_BUFSIZE];           // Decoded dot/dash info in symbol form, e.g. ".-.."                                         

bflags                b;                            // Various Operational state flags

double                pulse_avg;                    // CW timing variables - pulse_avg is a composite value
double                dot_avg, dash_avg;            // Dot and Dash Space averages             
double                symspace_avg, cwspace_avg;    // Intra symbol Space and Character-Word Space
int32_t               w_space;                      // Last word space time

AudioInputI2S          audioInput;                  // audio shield: mic or line-in
AudioMixer4            mixer;
AudioSynthWaveformSine Sinewave;
AudioOutputI2S         audioOutput;                 // audio shield: headphones & line-out
                                    
AudioAnalyzeFFT1024    fft1024;                     // 1024 bin FFT for CW decode                         
AudioAnalyzeFFT256     fft256;                      // 256 bin FFT  for CW decode, 4x faster than fft1024

#if LOCAL_KEY
AudioSynthWaveformSine KeySine;                     // Input from locally attached CW key
#endif
#if LOCAL_NOISE
AudioSynthNoiseWhite   Noise;                       // Noise source for realism
#endif

AudioConnection patchCord1(audioInput, 0, mixer, 0);
AudioConnection patchCord2(audioInput, 1, mixer, 1);
AudioConnection patchCord3(mixer, 0, fft1024, 0);
AudioConnection patchCord4(mixer, 0, fft256, 0);
#if LOCAL_KEY                                       // Sine Wave input and Noise for a locally connected CW key
AudioConnection patchCord10(KeySine, 0, mixer, 2);
#endif
#if LOCAL_NOISE
AudioConnection patchCord11(Noise, 0, mixer, 3);
#endif
AudioConnection patchCord20(Sinewave, 0, audioOutput, 0);
AudioConnection patchCord21(Sinewave, 0, audioOutput, 1);

AudioControlSGTL5000 audioShield;

Metro loo_ms = Metro(100);                          // Set up a 100ms Metro


//------------------------------------------------------------------
//
// Start the Works
//
//------------------------------------------------------------------
void setup() 
{
  // Audio connections require memory to work.  For more
  // detailed information, see the MemoryAndCpuUsage example
  AudioMemory(12);

  // Enable the audio shield, set the FFT window algorithm and output volume
  audioShield.enable();
  audioShield.inputSelect(AUDIO_INPUT);
  
  // Play with window functions
  fft1024.windowFunction(AudioWindowBlackmanNuttall1024); // Good overall rejection characteristics
  fft256.windowFunction(AudioWindowBlackmanNuttall256);
  
  fft256.averageTogether(FFTAVERAGE);                     // Average for spike/noise canceling - does nothing with FFT1024

  audioShield.volume(0.9);
  Sinewave.amplitude(0);
  #if LOCAL_NOISE
  Noise.amplitude(NOISE_LEVEL);
  #endif
  #if LOCAL_KEY
  KeySine.frequency(LOCAL_KEYFRQ);                        // Key input frequency at 700 Hz
  KeySine.amplitude(0);
  pinMode(KEY_PIN, INPUT_PULLUP);
  #endif
  pinMode(LED_PIN, OUTPUT);                               // Set up LED pin
  pinMode(SPD_RESET_PIN, INPUT_PULLUP);                   // Set up CW Speed initialize (reset) Input pin
  pinMode(FFT_SEL_PIN, INPUT_PULLUP);                     // Set up FFT select pin (1024 - high or 256 - low)
  
  lcdInit();                                              // initialize the LCD display

  analogReadResolution(8);                                // Set Analog Resolution at 8 bits
  analogReadAveraging(16);                                // Set Analog Averaging to reduce noise
  
  Serial.begin(115200);                                   // Init Serial (USB) port  
  Timer1.initialize(INTERRUPT_TIMER);                     // Init the sample timer interrupt function
  Timer1.attachInterrupt(SignalSampler);
  
  b.initialized = FALSE;                                  // We start with no dot/dash rate information
}


//------------------------------------------------------------------
//
// Top Level Loop
//
//------------------------------------------------------------------
void loop() 
{
  static int32_t  spd;                              // CW speed indication
  double          spdcalc;
  static int16_t  oldthresh;                        // Used to trigger refresh of threshold in FFT Waterfall on LCD  

  //-----------------------------------
  // Select between 1024 and 256 bin FFT
  if (digitalRead(FFT_SEL_PIN) == LOW)              // 256 bin fft if FFT_SEL_PIN is pulled low
  {
    fft256sel = TRUE;
    timer_stepsize = FFTAVERAGE;                    // FFT averaging = 2 doubles sampling time etc...
  }                                                 // used for consistent timing btw fft256 and fft1024
  else
  {
    fft256sel = FALSE;
    timer_stepsize = 4;                             // FFT1024 takes quadruple the time of FFT256
  }
  
  //-----------------------------------
  // Force Mark/Space timing initialize upon request
  if (digitalRead(SPD_RESET_PIN) == LOW)
  {
    b.initialized = FALSE;                          // Force dot/dash rate initialize 
  }
   
  //-----------------------------------
  // Fetch variables from SignalSampler() Interrupt routine
  noInterrupts();
  sig_incount = sig_lastrx;                         // Current Incount pointer
  cur_time    = sig_timer;                          // Time since last signal state change
  interrupts();

  //-----------------------------------
  //-----------------------------------
  //-----------------------------------
  //-----------------------------------
  // Decode incoming Morse Code
  CW_Decode();                                      // Do all the heavy lifting
   
  //-----------------------------------
  // Display the FFT Waterfall
  lcdDisplayWaterfall(thresh);

  //-----------------------------------
  // Display Threshold level and Mark/Space state changes on LCD
  static bool stateout;                             // Keep track of state changes for visualisation
  if (stateout != state)                            // Visualize the signal received
  {
    lcdUpdWaterfallThresholdLevel(state, thresh);
    stateout = state;
  } 
  
  //-----------------------------------
  // Every 100 ms, do certain house keeping tasks, such as
  // adjust the volume, peak and threshold - as per AD input
  // and  calculate CW speed
  if (loo_ms.check() == 1)
  {
    //-----------------------------------
    // Read AD inputs for volume, peak frequency and threshold info   
    vol = analogRead(VOL_PIN) * 4;                  // 0 - 1024
    if (fft256sel)
    {
      peakFrq = ((analogRead(TONE_PIN)*15)/256) + 2;// bins we are interested in, reflecting the range 344 - 2752 Hz
      peakFrq = (44100/256) * peakFrq;              // BW_of_bin * number of bin
    }
    else
    {
      peakFrq = (analogRead(TONE_PIN) * 60)/256 + 5;// bins we are interested in, reflecting the range 215 - 2800 Hz
      peakFrq = (44100/1024) * peakFrq;             // BW_of_bin * number of bin
    }
    Sinewave.frequency(peakFrq);                    // Set output frequency for tone-modulated mark/space output
    
    thresh = 2 + analogRead(THRESH_PIN) / 7;        // Max 40
    if (oldthresh != thresh)                        // Update threshold indication on LCD
    { 
      lcdUpdWaterfallThresholdLevel(state, thresh);
      oldthresh = thresh;     
    }

    //-----------------------------------
    // Word time. Formula based on the word "PARIS"
    spdcalc = 10.0*dot_avg + 4.0*dash_avg + 9.0*symspace_avg + 5.0*cwspace_avg;
    spdcalc = spdcalc*1000.0/344.0;                 // Convert to Milliseconds per Word
    spd = (0.5 + 60000.0 / spdcalc);                // Convert to Words per Minute (WPM)
    lcdStatusPrint(spd);                            // Print status information on LCD
  }

  //-----------------------------------
  // Send CPU usage and CW rate information to serial port when timeout, if selected in CWReceive.h
  // Else newline upon timeout
  static bool oneshot;
  if (cur_time >= 344*TIMEOUT)
  {
    oneshot = TRUE;
  }
  else if (oneshot == TRUE)
  {
    oneshot = FALSE;
    Serial.println();
    #if RATE_TO_SERIAL
    //Serial.print("CPU: ");
    //Serial.print(AudioProcessorUsage());
    //Serial.print(" CPUmax: ");
    //Serial.print(AudioProcessorUsageMax());
    //Serial.print(", ");
    Serial.print("WPM: ");
    Serial.print(", WPM: ");
    Serial.print(spd);
    Serial.print(" pulse ");
    Serial.print(pulse_avg);
    Serial.print(" dot ");
    Serial.print(dot_avg);
    Serial.print(" dash ");
    Serial.print(dash_avg);
    Serial.print(" symspace ");
    Serial.print(symspace_avg);
    Serial.print(" cwspace ");
    Serial.println(cwspace_avg);
    #endif
  }
}


//------------------------------------------------------------------
//
// Signal Sampler and Change Detection Function
// Interrupt driven, polls the FFT function once every 200us
// to see whether it has new data.  Either 256 or 1024 FFT is
// used to identify tone,
//
// Output is a circular buffer, sig[SIG_BUFSIZE], containing
// timing information for High & Low states.
//
//------------------------------------------------------------------
void SignalSampler(void)
{
  static int16_t  siglevel;                         // FFT signal level
  int16_t         lvl=0;                              // Multiuse variable
  int16_t         pklvl;                            // Used for AGC calculations   
  int16_t         pk;                               // FFT bin containing peak level 
  static bool     prevstate;                        // Last recorded state of signal input (mark or space)
  static bool     toneout;                          // Keep track of state changes for tone out
    
  //----------------
  // Fast Fourier Transform to derive signal waterfall
  //
  if(fft_is_available())
  {
    //----------------
    // Automatic Gain Control (AGC) - using level at peakFrq as basis 
    if (fft256sel)                                  // FFT256 selected
    {
      pk = 4*peakFrq/172 - 1;                       // FFT256 frequency bin number of peak frequency
      pklvl = agcvol * vol * fft256.read(peakFrq/172); // Get level at peak frequency
    }
    else                                            // FFT1024 selected
    {
      pk = peakFrq/43 - 5;                          // FFT1024 frequency bin number of peak frequency
      pklvl = agcvol * vol * fft1024.read(peakFrq/43); // Get level at peak frequency
    }
    if (pklvl > 45) agcvol = agcvol * AGC_ATTACK;   // Decrease volume if above this level.
    if (pklvl < 40) agcvol = agcvol * AGC_DECAY;    // Increase volume if below this level.
    if (agcvol > 1.0) agcvol = 1.0;                 // Cap max at 1.0

    // Scan through all relevant FFT bins (215 - 2800 Hz)
    // collect the relevant ones around peakFrq - and plot all to LCD
    if (fft256sel)                                  // FFT256 scan through all bins
    {
      for (uint8_t x=0; x<60; x++)
      {
        // Get signal level at each frequency, start at 215 Hz (5 * 43 Hz)
        if (x%4==0) lvl= agcvol * vol *fft256.read((x+5)/4);  // Only read every fourth time
        if (lvl > 40) lvl=40;                       // Cap max at 40
        fftbuf[x] = lvl;
        siglevel = fftbuf[pk];                      // Fetch the signal level at peak frequency
      }
    }
    else                                            // FFT1024 scan through all bins
    {                
      for (uint8_t x=0; x<60; x++)
      {
        // Get signal level at each frequency, start at 215 Hz (5 * 43 Hz)
        lvl = agcvol * vol * fft1024.read(x+5);
        if (lvl > 40) lvl = 40;                     // Cap max at 40
        fftbuf[x] = lvl;     
        // Average the signal level in 1, 2 ... 5 FFT bins around the Peak frequency
        #if   FILTERBW == 5
        siglevel = (fftbuf[pk-2]+fftbuf[pk-1]+fftbuf[pk]+fftbuf[pk+1]+fftbuf[pk+2])/3;
        #elif FILTERBW == 4
        siglevel = (fftbuf[pk-1]+fftbuf[pk]+fftbuf[pk+1]+fftbuf[pk+2])/2;
        #elif FILTERBW == 3
        siglevel = (fftbuf[pk-1]+fftbuf[pk]+fftbuf[pk+1])/2;
        #elif FILTERBW == 2
        (FILTERBW==2) siglevel = (fftbuf[pk]+fftbuf[pk+1])/2;
        #else
        siglevel = fftbuf[pk];
        #endif               
      }
    }
    
    //----------------
    // Signal averaging (smoothing)
    static int16_t avg_win[SIGAVERAGE];             // Sliding window buffer for signal averaging, if used                           
    static uint8_t avg_cnt;                         // Sliding window counter
    avg_win[avg_cnt++] = siglevel;                  // Add value onto "sliding window" buffer  
    if (avg_cnt == SIGAVERAGE) avg_cnt = 0;         // and roll counter             
    lvl = 0;
    for (uint8_t x=0; x<SIGAVERAGE; x++)            // Average up all values within sliding window
    {
      lvl = lvl + avg_win[x];
    }
    siglevel = lvl/SIGAVERAGE;
    
    //----------------
    // Signal State sampling
    #if NOISECANCEL                                 // Noise Cancel function requires two consecutive
    static bool newstate, change;                   // reads to be the same to confirm a true change
    if (siglevel >= thresh) newstate = TRUE;
    else                    newstate = FALSE;
    if (change == TRUE)
    {
      state = newstate;
      change = FALSE;
    }
    else if (newstate != state) change = TRUE;
    #else                                           // No noise canceling
    if (siglevel >= thresh) state = TRUE;
    else                    state = FALSE;
    #endif

    //----------------
    // Record state changes and durations onto circular buffer
    if (state != prevstate)
    {
      // Enter the type and duration of the state change into the circular buffer
      sig[sig_lastrx].state  = prevstate;
      sig[sig_lastrx++].time = sig_timer;
      // Zero circular buffer when at max
      if (sig_lastrx == SIG_BUFSIZE) sig_lastrx = 0;
      sig_timer = 0;                                // Zero the signal timer.
      prevstate = state;                            // Update state
    }

    //----------------
    // Count signal state timer upwards based on which sampling rate is in effect
    sig_timer = sig_timer + timer_stepsize;
    if (sig_timer>=344*TIMEOUT) sig_timer = 344*TIMEOUT; // Impose a MAXTIME second boundary for overflow time 
  
    //static bool debug;
    //digitalWrite(LED_PIN, debug ^= 1);            // Debug - measure rate of FFT
    digitalWrite(LED_PIN, state);                   // Show current state on LED
  
    //----------------
    // Tone to Speaker when mark (key-down)
    if (toneout != state) 
    {
      if (state) Sinewave.amplitude(AUDIOOUT_LEVEL);
      else       Sinewave.amplitude(0);
      toneout = state;
    } 
  }

  //----------------
  // Key a Sinewave if LOCAL_KEY enabled 
  // (this does not necessarily belong in the Interrupt function)
  #if LOCAL_KEY                                     // Local Key input
  if (digitalRead(KEY_PIN) == LOW) KeySine.amplitude(KEY_LEVEL);
  else KeySine.amplitude(0);
  #endif
}


//------------------------------------------------------------------
//
// Returns TRUE if FFT function has new data
// (either fft256 or fft1024, as selected)
//
//------------------------------------------------------------------
bool fft_is_available(void)
{
  bool poll;
  if (fft256sel) poll = fft256.available();
  else           poll = fft1024.available();
  return poll;
}

