//*********************************************************************************
//**
//** Project.........: Read Hand Sent Morse Code (tolerant of considerable jitter)
//**
//** Copyright (c) 2016  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** The GNU General Public License is available at
//** http://www.gnu.org/licenses/
//**
//** Substantive portions of the methodology used here to decode Morse Code are found in:
//**
//** "MACHINE RECOGNITION OF HAND-SENT MORSE CODE USING THE PDP-12 COMPUTER"
//** by Joel Arthur Guenther, Air Force Institute of Technology,
//** Wright-Patterson Air Force Base, Ohio
//** December 1973
//** http://www.dtic.mil/dtic/tr/fulltext/u2/786492.pdf
//**
//** Platform........: Teensy 3.1 / 3.2 and the Teensy Audio Shield
//**                   & a 160x128 TFT LCD
//**
//** Initial version.: 0.00, 2016-01-25  Loftur Jonasson, TF3LJ / VE2LJX
//**
//*********************************************************************************

#include <Adafruit_GFX.h>                    // LCD Core graphics library
#include "FreeMono9pt_extended.h"            // 9pt font with non-ASCII characters

#if LCD_QDTECH
#include <Adafruit_QDTech.h>                 // 1.8" TFT Module using Samsung S6D02A1 chip
#else
#include <Adafruit_ST7735.h>                 // Standard Adafruit 1.8" 128x160 TFT LCD
#endif

#if LCD_QDTECH
Adafruit_QDTech tft = Adafruit_QDTech(CS_PIN, DC_PIN, RST_PIN);
#else
Adafruit_ST7735 tft = Adafruit_ST7735(CS_PIN, DC_PIN, MOSIPIN, SCLKPIN, RST_PIN);
#endif


//------------------------------------------------------------------
//
// Initialize LCD
//
//------------------------------------------------------------------
void lcdInit(void)
{
  #if LCD_QDTECH                             // QDTech = Samsung S6D02A1 chipset
  SPI.setMOSI(7);                            // Set up alternate HW SPI for LCD,
  SPI.setSCK(14);                            // audio shield uses the primary SPI
  tft.init();
  #else                                      // Standard Adafruit 1.8" 128x160 TFT LCD
  // Three different flavours of the Adafruit 1.8" 128x160 Adafruit TFT LCD
  #if ST7735_BLACKTAB
  tft.initR(INITR_BLACKTAB);                 // initialize a ST7735S chip, black tab
  #elif ST7735_REDTAB
  tft.initR(INITR_REDTAB);                   // initialize a ST7735R chip, red tab
  #elif ST7735_GREENTAB
  tft.initR(INITR_GREENTAB);                 // initialize a ST7735R chip, green tab
  #endif
  #endif

  tft.setRotation(1);
  displayPictureOnLCD(0,0);                  // Display a picture of a CW Key, no offset
  
  tft.setTextColor(0xF800);                  // Red
  tft.setTextWrap(false);
  //tft.setCursor(0, 85);
  tft.setCursor(0, 15);
  tft.print("   Teensy 3.2 CW Decoder");
  tft.setTextColor(0x001f);                  // Blue
  tft.setCursor(0, 105);
  tft.print(" Version: ");
  tft.print(VERSION);
  tft.print(" ");
  tft.print(DATE);
  tft.setCursor(0, 115);
  tft.print("       TF3LJ / VE2LJX");        // Feel free to replace with your own

  delay(5000);
  tft.fillScreen(0x000);                     // Black

  // Draw Box around Waterfall Display
  tft.drawFastHLine(78, 0,62, 0xfc00);       // Orange
  tft.drawFastHLine(78, 42,62, 0xfc00);
  tft.drawFastVLine(78, 0,42, 0xfc00);
  tft.drawFastVLine(139, 0,42, 0xfc00);
}


//------------------------------------------------------------------
//
// BMP fileformat Draw, reading from a  C include file (.h).
//
// The below function is a butchered subset extracted from an 
// Adafruit Example to read a BMP file from an SD card, hence
// the below Copyright notice:
//
/***************************************************
  This is an example sketch for the Adafruit 1.8" SPI display.
  This library works with the Adafruit 1.8" TFT Breakout w/SD card
  ----> http://www.adafruit.com/products/358
  as well as Adafruit raw 1.8" TFT display
  ----> http://www.adafruit.com/products/618

  Check out the links above for our tutorials and wiring diagrams
  These displays use SPI to communicate, 4 or 5 pins are required to
  interface (RST is optional)
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  MIT license, all text above must be included in any redistribution
 ****************************************************/  
 //------------------------------------------------------------------
void displayPictureOnLCD(uint8_t x, uint8_t y)
{

  int      bmpWidth, bmpHeight; // W+H in pixels
  uint32_t bmpImageoffset;      // Start of image data in file
  uint32_t rowSize;             // Not always = bmpWidth; may have padding
  boolean  flip = true;         // BMP is stored bottom-to-top
  int      w, h, row, col;
  uint8_t  r, g, b;
  uint32_t pos = 0;
  uint32_t *read32;             // Pointer to pull 32 bit little endian info out of the (uint8_t) picture[]

  if((x >= tft.width()) || (y >= tft.height())) return; // Outside of printable area
  
  read32 = (uint32_t*) &CWkey160_128[0x0a];
  bmpImageoffset = read32[0];                           // 0x0a - 0x0d, reference to start of image data
  bmpWidth = read32[2];                                 // 0x12 - 0x15
  bmpHeight = read32[3];                                // 0x16 - 0x19

  rowSize = (bmpWidth * 3 + 3) & ~3;                    // BMP rows are padded (if needed) to 4-byte boundary
  if(bmpHeight < 0)                                     // If bmpHeight is negative, image is in top-down order.
  {                                                     // This is not canon but has been observed in the wild.
    bmpHeight = -bmpHeight;
    flip      = false;
  }
  w = bmpWidth;                                         // Crop area to be loaded
  h = bmpHeight;
  if((x+w-1) >= tft.width())  w = tft.width()  - x;
  if((y+h-1) >= tft.height()) h = tft.height() - y;

  tft.setAddrWindow(x, y, x+w-1, y+h-1);                // Set TFT address window to clipped image bounds

  for (row=0; row<h; row++)                             // For each scanline...
  {
    if(flip)                                            // Bitmap is stored bottom-to-top order (normal BMP)
      pos = bmpImageoffset + (bmpHeight - 1 - row) * rowSize;
    else                                                // Bitmap is stored top-to-bottom
      pos = bmpImageoffset + row * rowSize;

    for (col=0; col<w; col++) {                         // For each pixel...
      #if !LCD_QDTECH && R_B_COLOUR_INVERT              // My 3rd party ST7735 display inverts the colours
      r = CWkey160_128[pos++];
      g = CWkey160_128[pos++];
      b = CWkey160_128[pos++];
      #else
      b = CWkey160_128[pos++];
      g = CWkey160_128[pos++];
      r = CWkey160_128[pos++];
      #endif
      tft.pushColor(tft.Color565(r,g,b));
    }
  }
}


//------------------------------------------------------------------
//
// Display the FFT Waterfall and update the Frequency Peak line
//
// Takes the Audio Level decision threshold (1 - 39) as argument 
//
//------------------------------------------------------------------
void lcdDisplayWaterfall(int16_t tpos)
{
  int16_t  lvl = 0;                                          // FFT graph: Normalized output from FFT
  uint16_t sig, nosig;
  uint8_t  pkpos;

  tpos = 41 - tpos;                                          // Position of threshold line on screen
  if (fft256sel) pkpos = peakFrq/43 - 3;                     // Center the frequency indication correctly
  else           pkpos = peakFrq/43 - 4;                     // based on whether fft256 or fft1024
  
  // Display FFT Waterfall
  for (uint8_t x = 0; x < 60; x++)
  {
    lvl = fftbuf[x];
        
    if (x == pkpos)                                          // Frequency Peak line colours
    {
      sig   = 0xffe0;                                        // Yellow
      nosig = 0x001f;                                        // Blue
    }
    else                                                     // FFT Waterfall colours
    {
      sig   = 0x07e0;                                        // Green
      nosig = 0x0000;                                        // Black
    }
    if (lvl >= thresh)      // level is higher than threshold, wrap green around tpos line
    {
      tft.drawFastVLine(79+x, 1, 40-lvl, nosig);            // Black (or Blue at peakFrq line)
      tft.drawFastVLine(79+x, 41-lvl, tpos-(41-lvl), sig);  // Green, Above tpos (or Yellow at peakFrq line)
      tft.drawFastVLine(79+x, tpos+1, 39-tpos, sig);        // Green, Below tpos
    }
    else                    // Level is lower than threshold. Wrap black around tpos line
    {
      tft.drawFastVLine(79+x, 1, tpos-1, nosig);            // Black, Above tpos
      tft.drawFastVLine(79+x, tpos+1, (41-lvl)-tpos, nosig);// BlackBelow tpos
      tft.drawFastVLine(79+x, 41-lvl,lvl, sig);             // Green
    }
  }
}


//------------------------------------------------------------------
//
// Update/Change Colour of Audio Threshold Level, when exceeded
//
// Takes the Audio Level decision threshold (1 - 39) as argument 
//
//------------------------------------------------------------------
void lcdUpdWaterfallThresholdLevel(bool state, int16_t tpos)
{
  tpos = 41 - tpos;                                        // Position of threshold line on screen
  if (state) tft.drawFastHLine(79, tpos,60, 0xffff);       // White
  else       tft.drawFastHLine(79, tpos,60, 0xf800);       // Red
}


//------------------------------------------------------------------
//
// LCD print and scroll.  Five lines of 14 characters each.
//
//------------------------------------------------------------------
void lcdLineScrollPrint(char in)
{
  static char     c1;                            // Last character buffer, for UTF-8 conversion
  static char     line1[15];                     // String storage for LCD lines 5 to 1
  static char     line2[15];
  static char     line3[15];
  static char     line4[15];
  static char     line5[15];
  static uint16_t curs;                          // Cursor position at bottom line
  
  static int8_t   x;                             // current X co-ordinate
  static uint8_t  last_space;                    // location of latest space character
  static bool     clearscreen=FALSE;             // Used to clear screen the very first time
  
  //-----------------------------------
  // Clear Screen on first run
  if (clearscreen == FALSE)
  {
    tft.fillRect(0, 50, 160, 78, 0x0000);        // Black, Clear block of all lines
    clearscreen = TRUE;
  }
  
  //-----------------------------------
  // Convert UTF-8 encoded non-ASCII to extended ASCII
  if ((in < 0x20) || (in >= 0xd0))               // Not a printable character, throw away garbage
  {
    c1 = 0;
    return;
  }
  if (in >= 0x80)                                // Extended Character, process
  {
    if (c1 == 0xc2)                              // No address translation necessary
    {
      if (in < 0xa0)                             // Whoops - unprintable char - throw away
      {
        c1 = 0;
        return;
      }
    }
    else if (c1 == 0xc3)                         // Translate address in accordance with UTF8
    {
      in |= 0xc0;
    }
    else if ((c1 > 0xc3) && (c1 <= 0xcf))        // Unprintable - goes beyond our font - throw away
    {
      c1 = 0;
      return;
    }
    else
    {
      c1 = in;
      return; 
    }
  }
  c1 = in;

  //-----------------------------------
  // Print and Scroll
  tft.setFont(&FreeMono9pt8b);
  if (in == ' ') last_space = x;                 // Keep track of word spaces in order to avoid breaking up words
  if (x == 14)                                   // End of line.  Scroll everything up one line
  {
    tft.setTextColor(0x0000);                    // Clear all lines by reprinting in BLACK
    tft.setCursor(0, 64);
    tft.print(line5);
    tft.setCursor(0, 79);
    tft.print(line4);
    tft.setCursor(0, 94);
    tft.print(line3);
    tft.setCursor(0, 109);
    tft.print(line2);
    tft.setCursor(0, 124);
    tft.print(line1);    

    memcpy(line5,line4,14);                      // Shift all lines up
    memcpy(line4,line3,14);
    memcpy(line3,line2,14);
    if (last_space == 0) last_space = 14;        // Special case, no space in line1
    memcpy(line2,line1,last_space);
    line2[last_space]='\0';                      // Terminate string

    tft.setTextColor(0xffe0);                    // Set Yellow as Text Colour
    tft.setCursor(0, 64);                        // and print data
    tft.print(line5);
    tft.setCursor(0, 79);
    tft.print(line4);
    tft.setCursor(0, 94);
    tft.print(line3);
    tft.setCursor(0, 109);
    tft.print(line2);
    tft.setCursor(0, 124);                       // Set cursor at beginning of first (lowest) line

    if (last_space == 14)                        // Special case, no space in line1, therefore nothing to print
    {
      line1[0] = '\0';
      x = 0;
      curs = 0;
    }
    else                                         // Move beginning of unfinished word to front of line
    {
      x = 14 - (last_space+1);                   // Get length of text we want to move to front 
      memmove(line1, &line1[last_space+1],x);
      line1[x] = '\0';                           // Terminate end of string
      tft.print(line1);                          // Print the moved word-fragment
      curs = x * 11 + 11;                        // and move cursor accordingly
      line1[x++] = in;                           // Print to string
      tft.print(in);                             // and print to LCD
    }
    last_space = 0;            
  }
  else                                           // Print new character to string and LCD, one char at a time
  {
    tft.setTextColor(0xffe0);                    // Set Yellow as Text Colour
    tft.setCursor(curs, 124);
    line1[x++] = in;
    tft.print(in);
    curs = curs + 11;
  }
}


//------------------------------------------------------------------
//
// Print Information on Left Hand Side of FFT Waterfall
//
//------------------------------------------------------------------
void lcdStatusPrint(int16_t cwspd)
{
  static int16_t oldSpd;
  static int16_t oldFrq;
  static bool    overloadstate = FALSE;
  char           string[14];
  static double  oldagc;
  
  if (cwspd < 0) cwspd = 0;                      // Impose sane bounds
  if (cwspd > 99) cwspd = 99;
  
  //-----------------------------------
  // Buffer Overload Indicatin
  if ((b.overload==TRUE)&&(overloadstate=FALSE)) // A Buffer Overload State just started
  {
    tft.setFont(NULL);
    tft.fillRect(0, 0, 75, 14, 0x0000);          // Black, Rather than keeping track of last state, just clear block
    tft.setTextColor(0xf800);                    // Change fonts and set Red as text colour
    tft.setCursor(0, 3);
    tft.print("BUF Overload");
    overloadstate = TRUE;
  }
  //-----------------------------------
  // Initialization Indication
  else if ((b.initialized==FALSE)&&(oldSpd!=100))// Initialize has just started
  {
    tft.setFont(NULL);
    tft.fillRect(0, 0, 75, 14, 0x0000);          // Black, Rather than keeping track of last state, just clear block
    tft.setTextColor(0xf800);                    // Change fonts and set Red as text colour
    tft.setCursor(0, 3);
    tft.print("Initializing");
    oldSpd = 100;                                // 100 indicates that "Initializing has already been printed
  }
  //-----------------------------------
  // Clear when either of the two is resolved
  else if (((b.initialized==TRUE)&&(oldSpd==100))// Oneshot: Remove "Initializing" or "BUF Overload"
        || ((b.overload==FALSE)&&(overloadstate==TRUE)))
  {
    tft.fillRect(0, 0, 75, 14, 0x0000);          // Black, Rather than keeping track of last state, just clear block
    oldSpd = 0;
    overloadstate = FALSE;
  }

  //-----------------------------------
  // Update CW speed indication
  if ((b.initialized==TRUE) && (oldSpd!=cwspd))
  {
    tft.setFont(&FreeMono9pt8b);
    tft.setTextColor(0x0000);                    // Overwrite old speed with black
    tft.setCursor(12, 10);
    tft.print(oldSpd);
    tft.print("WPM");     

    tft.setTextColor(0xfc00);                    // Orange
    tft.setCursor(12, 10);
    tft.print(cwspd);
    tft.print("WPM");     
    oldSpd = cwspd;
  }
   
  //-----------------------------------
  // Update Peak Frequency indication
  if (peakFrq != oldFrq)
  {
    tft.setFont(NULL);
    tft.setTextColor(0x0000);                    // Overwrite old text with black
    tft.setCursor(15, 20);
    tft.print(oldFrq);
    tft.print(" Hz"); 
    
    //tft.setTextColor(0x07ec);                  // Blue-Greenish
    tft.setTextColor(0x001f);                    // Blue
    tft.setCursor(15, 20);
    tft.print(peakFrq);
    tft.print(" Hz"); 
    oldFrq = peakFrq;
  }
  //-----------------------------------
  // Update AGC volume indication
  if (agcvol != oldagc)
  {
    tft.setFont(NULL);
    sprintf(string,"AGC: %1.02f",oldagc);
    tft.setTextColor(0x0000);                    // Overwrite old text with black
    tft.setCursor(15, 35);
    tft.print(string);
    
    sprintf(string,"AGC: %1.02f",agcvol);
    tft.setTextColor(0x07e0);                    // Green
    tft.setCursor(15, 35);
    tft.print(string);
    oldagc = agcvol;
  }
}

