//*********************************************************************************
//**
//** Project.........: AD8317 based Intelligent Attenuator
//**
//**
//**                   Modes of operation:
//**
//**                   1) Manually adjust a chain of 30 + 30 + 60 + 0.5-31.5dB
//**                      attenuators.  Range: 0 - 151.5dB, in 0.5dB steps.
//**                   2) Automatically adjust for a predetermined output value,
//**                      in either dBm or mV/uV.
//**                   3) RF power meter, -50dBm to +30dBm, 0 - 1000 MHz
//**
//** Platform........: AT90usb646 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//**
//**
//** Initial version.: 2010-08-05, Loftur Jonasson, TF3LJ
//**
//**
//** History.........: Check the Att.c file
//**
//*********************************************************************************

#include 	<math.h>
#include	"Att.h"


//
//-----------------------------------------------------------------------------
//			Print dBm, accepts 10x dBm input value
//-----------------------------------------------------------------------------
//
void print_dbm(int16_t db10m)
{
	// Add power dissipated in the External Front Attenuator
	db10m = db10m + R.front_Attenuator;

	int16_t pwrdb_tenths = db10m;

	if (pwrdb_tenths < 0) pwrdb_tenths *= -1;
	int16_t pwrdb = pwrdb_tenths / 10;
	pwrdb_tenths = pwrdb_tenths % 10;

	if (db10m <= -100)
	{
		rprintf("-%2u.%1udBm",pwrdb,pwrdb_tenths);
	}
	else if (db10m < 0)
	{
		rprintf(" -%1u.%1udBm",pwrdb,pwrdb_tenths);
	}
	else rprintf("%3u.%1udBm",pwrdb,pwrdb_tenths);
}


//
//-----------------------------------------------------------------------------
//			Print Power, input value is in milliWatts
//-----------------------------------------------------------------------------
//
void print_p_mw(double mw)
{
	uint32_t p_calc;
	uint16_t power_sub, power;

	// Add power dissipated in the External Front Attenuator
	mw = mw * pow(10,R.front_Attenuator/100.0);

	if(mw >= 10000.0)
	{
		p_calc = mw;
		power = p_calc / 1000;
		power_sub = ((p_calc % 1000)+50)/100;
		rprintf("%2u.%01uW",power, power_sub);
	}
	else if(mw >= 1000.0)
	{
		p_calc = mw;
		power = p_calc / 1000;
		power_sub = ((p_calc % 1000)+5)/10;
		rprintf("%1u.%02uW",power, power_sub);
	}
	else if(mw >= 100.0)
	{
		rprintf("%4umW",(uint16_t)mw);
	}
	else if(mw >= 10.0)
	{
		p_calc = mw * 10;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01umW",power, power_sub);
	}
	else if(mw >= 1.0)
	{
		p_calc = mw * 100;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02umW",power, power_sub);
	}
	else if(mw >= 0.1)
	{
		power = mw * 1000;
		rprintf("%4uuW",power);
	}
	else if(mw >= 0.01)
	{
		p_calc = mw * 10000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01uuW",power, power_sub);
	}
	else if(mw >= 0.001)
	{
		p_calc = mw * 100000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02uuW",power, power_sub);
	}
	else if(mw >= 0.0001)
	{
		power = mw * 1000000;
		rprintf("%4unW",power);
	}
	else if(mw >= 0.00001)
	{
		p_calc = mw * 10000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01unW",power, power_sub);
	}
	else if(power_out_mw >= 0.000001)
	{
		p_calc = mw * 100000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02unW",power, power_sub);
	}
	else if(power_out_mw >= 0.0000001)
	{
		power = power_out_mw * 1000000000;
		rprintf("%4upW",power);
	}
	else if(power_out_mw >= 0.00000001)
	{
		p_calc = power_out_mw * 10000000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01upW",power, power_sub);
	}
	else if(power_out_mw >= 0.000000001)
	{
		p_calc = power_out_mw * 100000000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02upW",power, power_sub);
	}
	else if(power_out_mw >= 0.0000000001)
	{
		power = power_out_mw * 1000000000000;
		rprintf("%4ufW",power);
	}
	else if(power_out_mw >= 0.00000000001)
	{
		p_calc = power_out_mw * 10000000000000;
		power = p_calc / 10;
		power_sub = p_calc % 10;
		rprintf("%2u.%01ufW",power, power_sub);
	}
	else
	{
		p_calc = power_out_mw * 100000000000000;
		power = p_calc / 100;
		power_sub = p_calc % 100;
		rprintf("%1u.%02ufW",power, power_sub);
	}
}


//
//-----------------------------------------------------------------------------
//			Print Voltage, input value in volts
//-----------------------------------------------------------------------------
//
void print_v(double voltage)
{
	uint16_t power_sub, power;

	// Add power dissipated in the External Front Attenuator
	voltage = voltage * pow(10,R.front_Attenuator/200.0);

	if(voltage >= 10.0)
	{
		power_sub = voltage * 10;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01uV",power, power_sub);
	}	
	else if(voltage >= 1.0)
	{
		power_sub = voltage * 100;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%2u.%02uV",power, power_sub);
	}
	else if(voltage >= 0.1)
	{
		power = voltage * 1000;
		rprintf("%4umV",power);
	}
	else if(voltage >= 0.01)
	{
		power_sub = voltage * 10000;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01umV",power, power_sub);
	}
	else if(voltage >= 0.001)
	{
		power_sub = voltage * 100000;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%1u.%02umV",power, power_sub);
	}
	else if(voltage >= 0.0001)
	{
		power = voltage * 1000000;
		rprintf("%4uuV",power);
	}
	else if(voltage >= 0.00001)
	{
		power_sub = voltage * 10000000;
		power = power_sub / 10;
		power_sub = power_sub % 10;
		rprintf("%2u.%01uuV",power, power_sub);
	}
	else
	{
		power_sub = voltage * 100000000;
		power = power_sub / 100;
		power_sub = power_sub % 100;
		rprintf("%1u.%02uuV",power, power_sub);
	}
}


//
//-----------------------------------------------------------------------------
//			Print Attenuation in dB, input value in dB * 10
//-----------------------------------------------------------------------------
//
void print_att(uint16_t attenuation)
{
	uint16_t att_tenths = attenuation;
	
	// Add attenuation in the External Front Attenuator
	att_tenths = attenuation + R.front_Attenuator;
	
	uint16_t att = att_tenths / 10;
	att_tenths = att_tenths % 10;
	rprintf("%3u.%1udB",att, att_tenths);
}


//
//-----------------------------------------------------------------------------
//			Display Power Meter
//-----------------------------------------------------------------------------
//
void lcd_display_power_meter(void)
{
	double 	input_voltage;
	
	lcd_clrscr();


	lcd_gotoxy(0,0);
	rprintf("Power Meter:");

	//------------------------------------------
	// Input power indication, dBm average
	lcd_gotoxy(0,1);
	//rprintf("P_in:");
	rprintf(" ");

	print_dbm((int16_t) (power_in_db*10.0));

	//------------------------------------------
	// Amplitude Modulation Index
	//lcd_gotoxy(0,3);
	//rprintf("AM index:   %3d",modulation_index);
	lcd_gotoxy(14,1);
	rprintf("AM%3d",modulation_index);
	rprintfChar('%');

	//------------------------------------------
	// Input power indication, average
	lcd_gotoxy(0,2);
	// Input wattage Printout)

	print_p_mw(power_in_mw);
	rprintf("rms");

	//------------------------------------------
	// Input power indication, 1 second PEP
	lcd_gotoxy(0,3);
	// Input wattage Printout)
	print_p_mw(power_in_mw_pep);
	rprintf("pep");

	//------------------------------------------
	// Input voltage (in 50 ohms) indication, RMS
	lcd_gotoxy(11,2);
	input_voltage = sqrt(power_in_mw*50/1000.0);
	print_v(input_voltage);
	rprintf("rms");

	//------------------------------------------
	// Input voltage (in 50 ohms) indication, Vpeak-to-Peak
	lcd_gotoxy(11,3);
	input_voltage = sqrt(power_in_mw*50/1000.0) * 2 * sqrt(2);
	print_v(input_voltage);
	rprintf("p-p");
}


//
//-----------------------------------------------------------------------------
//			Display Attenuator stuff 
//-----------------------------------------------------------------------------
//
void lcd_display_attenuator(void)
{
	double 	output_voltage;
	
	lcd_clrscr();

	lcd_gotoxy(0,0);
	rprintf("Attenuator:");

	//------------------------------------------
	// Attenuator Printout
	lcd_gotoxy(13,0);
	print_att(total_attenuation);

	//------------------------------------------
	// Input power indication, dBm average
	lcd_gotoxy(0,2);
	rprintf("P_in:");
	print_dbm((int16_t) (power_in_db*10.0));

	//------------------------------------------
	// Input wattage Printout
	lcd_gotoxy(14,2);
	// Input wattage Printout)
	print_p_mw(power_in_mw);

	//------------------------------------------
	// Output power indication, dBm average
	lcd_gotoxy(0,3);
	rprintf("P_o:");
	if((power_out_db - R.front_Attenuator)>=-100.0) rprintfChar(' ');

	// The print_dbm() function adds power according to the size of
	// the front_Attenuator.  Therefore it needs to be subtracted
	// to get the actual power out
	print_dbm((int16_t) (power_out_db*10.0 - R.front_Attenuator));

	//------------------------------------------
	// Output voltage Printout)
	lcd_gotoxy(14,3);

	output_voltage = sqrt(power_out_mw*50/1000.0);

	// The print_v() function adds voltage according to the size of
	// the front_Attenuator.  Therefore it needs to be subtracted
	// to get the actual voltage out
	print_v(output_voltage / pow(10,R.front_Attenuator/200.0));
}


//
//-----------------------------------------------------------------------------
//			Display Signal Generator Function
//-----------------------------------------------------------------------------
//
void lcd_display_siggen(void)
{
	double 	output_voltage;
	
	lcd_clrscr();

	lcd_gotoxy(0,0);
	rprintf("Signal Generator:");

	//------------------------------------------
	// Output power indication
	lcd_gotoxy(11,1);
	if((power_out_db + R.front_Attenuator) >=-100.0) rprintfChar(' ');

	// The print_dbm() function adds power according to the size of
	// the front_Attenuator.  Therefore it needs to be subtracted
	// to get the actual power out
	print_dbm((int16_t) (power_out_db*10.0 - R.front_Attenuator));

	//------------------------------------------
	// Output wattage Printout
	lcd_gotoxy(14,2);

	// The print_p_mw() function adds wattage according to the size of
	// the front_Attenuator.  Therefore it needs to be subtracted
	// to get the actual wattage out
	print_p_mw(power_out_mw / pow(10,R.front_Attenuator/100.0));

	//------------------------------------------
	// Output voltage indication
	lcd_gotoxy(14,3);
	output_voltage = sqrt(power_out_mw*50/1000.0);

	// The print_v() function adds voltage according to the size of
	// the front_Attenuator.  Therefore it needs to be subtracted
	// to get the actual voltage out
	print_v(output_voltage / pow(10,R.front_Attenuator/200.0));

	//------------------------------------------
	// Modulation Index
	//lcd_gotoxy(13,3);
	lcd_gotoxy(1,2);
	rprintf("AM:%3d",modulation_index);
	rprintfChar('%');

	//------------------------------------------
	// Attenuator Printout
	lcd_gotoxy(0,3);
	rprintf("Att:");
	print_att(total_attenuation);
}


//
//-----------------------------------------------------------------------------
//			Display a Mixed bag of irrelevant stuff
//-----------------------------------------------------------------------------
//
void lcd_display_mixed(void)
{
	double 	output_voltage;
	uint16_t att_tenths, att;
	uint16_t power, power_sub;
	
	lcd_clrscr();

	lcd_gotoxy(0,0);
	rprintf("Debug Stuff:");

	//------------------------------------------
	// Attenuator Printout
	lcd_gotoxy(15,0);
	att_tenths = total_attenuation + R.front_Attenuator;
	att = att_tenths / 10;
	att_tenths = att_tenths % 10;
	rprintf("%3u.%1u",att, att_tenths);

	//------------------------------------------
	// AD8317 input voltage indication
	lcd_gotoxy(0,1);
	output_voltage = ad8317_ad * 4 / 1000.0;
	power_sub = output_voltage * 1000;
	power = power_sub / 1000;
	power_sub = power_sub % 1000;
	rprintf("AD8317: %u.%03uV  %4u", power, power_sub, ad8317_ad);	

	//------------------------------------------
	// Calibrate 1
	lcd_gotoxy(0,2);
	rprintf("Cal1: ");
	print_dbm(R.calibrate[0].db10m);
	rprintf ("  %4u",R.calibrate[0].ad);

	//------------------------------------------
	// Calibrate 2
	lcd_gotoxy(0,3);
	rprintf("Cal2: ");
	print_dbm(R.calibrate[1].db10m);
	rprintf ("  %4u",R.calibrate[1].ad);
}
