//*********************************************************************************
//**
//** Project.........: AD8317 based Intelligent Attenuator
//**
//**                   PushButtonMenu.c
//**
//**                   This code is salvaged in part from the SDR-Widget Project 
//**                   (see groups.google.com/group/sdr-widget)
//**
//**                   One long push triggers the Menu functions, which utilize the Rotary
//**                   Encoder for menu choice selection, and use the push button for confirmation
//**                   of choice.  The Menu functions may be several levels deep, depending on
//**                   choice.
//**
//**                   See Att.c for further description
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2010-08-05, Loftur Jonasson, TF3LJ
//**
//** History.........: ...
//**
//*********************************************************************************

#include <stdio.h>
#include <math.h>

#include "Att.h"
#include "lcd.h"


// First Level Menu Items
const uint8_t level0_menu_size = 7;
const char *level0_menu_items[] =
				{  " 1-Display Debug",
				   " 2-Auto Calibrate",
				   " 3-AutoFrontAttSet",
				 //" 4-Manual Cal High",	// TODO
				 //" 5-Manual Cal Low",		// TODO
				 //" 6-Manual Att Set",		// TODO
				   " 7-Adj Attenuators",
				   " 8-Adj Encoder Res",
				   " 9-Factory Reset",
				   "10-Exit" };

// Flag for Debug Screen
#define DEBUG_MENU		1

// Flag for Reference 1 Menu Selection
#define POW_MENU		2

// Flag for AutoSet dBm for First Attenuator
#define AUTO_FIRST_ATT	3


// Flag for Set Attenuator values
#define ATT_MENU		7
const uint8_t attenuator_menu_size = 7;
const char *attenuator_menu_items[] =
				{  "1-Front Att",
				   "2-First Att",
				   "3-Second Att",
				   "4-Third Att",
   				   "5-Insert Loss",
				   "6-Go Back",
				   "7-Exit"};

// Flags for Attenuator Submenu functions
#define ATT_SET1_MENU	701
#define ATT_SET2_MENU	702
#define ATT_SET3_MENU	703
#define ATT_SET4_MENU	704
#define ATT_SET5_MENU	705

// Flag for Encoder Resolution Change
#define ENCODER_MENU	8

// Flag for Factory Reset
#define FACTORY_MENU	9
// Factory Reset menu Items
const uint8_t factory_menu_size = 3;
const char *factory_menu_items[] =
				{  "1-Yes  Reset",
				   "2-No - Go back",
				   "3-No - Exit"	};


uint16_t		menu_level = 0;						// Keep track of which menu we are in
uint8_t			menu_data = 0;						// Pass data to lower menu


//----------------------------------------------------------------------
// Display a 3 or 4 line Menu of choices
//
// **menu refers to a pointer array containing the Menu to be printed
//
// menu_size indicates how many pointers (menu items) there are in the array
//
// current_choice indicates which item is currently up for selection if
// pushbutton is pushed
//
// begin row & begin_col are the coordinates for the upper lefthand corner
// of the three or four lines to be printed
//
// lines can take the value of 3 or 4, indicating how many lines the scrolling
// menu contains
//----------------------------------------------------------------------
void lcd_scroll_Menu(char **menu, uint8_t menu_size,
		uint8_t current_choice, uint8_t begin_row, uint8_t begin_col, uint8_t lines)
{
	uint8_t a, x;


	// Clear LCD from begin_col to end of line.
	lcd_gotoxy(begin_col, begin_row);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
	lcd_gotoxy(begin_col, begin_row+1);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
	lcd_gotoxy(begin_col, begin_row+2);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');

	// Using Menu list pointed to by **menu, preformat for print:
	// First line contains previous choice, secon line contains
	// current choice preceded with a '->', and third line contains
	// next choice
	if (current_choice == 0) x = menu_size - 1;
	else x = current_choice - 1;
	lcd_gotoxy(begin_col + 2, begin_row);
	rprintf("%s", *(menu + x));
	lcd_gotoxy(begin_col, begin_row + 1);
	rprintf("->%s", *(menu + current_choice));
	if (current_choice == menu_size - 1) x = 0;
	else x = current_choice + 1;
	lcd_gotoxy(begin_col + 2, begin_row + 2);
	rprintf("%s", *(menu + x));

	// LCD print lines 1 to 3

	// 4 line display.  Preformat and print the fourth line as well
	if (lines == 4)
	{
		if (current_choice == menu_size-1) x = 1;
		else if (current_choice == menu_size - 2 ) x = 0;
		else x = current_choice + 2;
		lcd_gotoxy(begin_col, begin_row+3);
		for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
		lcd_gotoxy(begin_col + 2, begin_row + 3);
		rprintf("  %s", *(menu + x));
	}
}


//----------------------------------------------------------------------
// Menu functions begin:
//----------------------------------------------------------------------


//--------------------------------------------------------------------
// Debug Screen, exit on push
//--------------------------------------------------------------------
void debug_menu(void)
{

	set_Attenuators();			// Set the attenuators if Encoder input or if Power protect

	lcd_display_mixed();		// Display diverse debug stuff

	// Exit on Button Push
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;	// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Nothing Changed");
		_delay_ms(500);
		Menu_Mode |= CONFIG;	// We're NOT done, just backing off
		menu_level = 0;			// We are done with this menu level
	}
}


//--------------------------------------------------------------------
// Calibrate Power Meter based on known input
//--------------------------------------------------------------------
void pow_calibrate_menu(void)
{

	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements
	static uint8_t	Calibrating;		// Boolean for indicating calibrating in process
	static uint16_t	calTimer;			// Calibration timer

	// Is first attenuator force set due to high input power?
	if (Status & HI_INPUT)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Input level too high");
		_delay_ms(500);
		Menu_Mode &=  ~CONFIG;			// We're done
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
	
	else
	{
		if (!Calibrating)
		{
			//-------------------------------------------------------------------
			// Set current attenuation level and update attenuator outputs
			//
			R.Attenuators = ATT_SECOND | ATT_THIRD | 0xfc;	// Set all attenuators except for the first
			ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
			ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
			//ATT_PORT01 = (R.Attenuators & 0x003f) | (~R.Attenuators & 0x00c0);

			// Get Current value
			current_selection = R.calibrate[0].db10m;

			// Selection modified by encoder.  We remember last selection, even if exit and re-entry
			if (Status & ENC_CHANGE)
			{
				if (encOutput > 0)
				{
					current_selection++;
			    	// Reset data from Encoder
					Status &=  ~ENC_CHANGE;
					encOutput = 0;
				}
				else if (encOutput < 0)
				{
					current_selection--;
			  	  	// Reset data from Encoder
					Status &=  ~ENC_CHANGE;
					encOutput = 0;
				}
				// Indicate that an LCD update is needed
				LCD_upd = FALSE;
			}
		
			// If LCD update is needed
			if (LCD_upd == FALSE)
			{
				LCD_upd = TRUE;			// We have serviced LCD
		
				// Keep Encoder Selection Within Bounds of the Menu Size
				if(current_selection > CAL1_MAX_VALUE) current_selection = CAL1_MAX_VALUE;
				if(current_selection < CAL1_MIN_VALUE) current_selection = CAL1_MIN_VALUE;
		
				// Store Current value in running storage
				R.calibrate[0].db10m = current_selection;
		
				lcd_clrscr();
				lcd_gotoxy(0,0);	
				rprintf("Auto Calibrate:");
		
				lcd_gotoxy(0,1);
				rprintf("Rotate to Adjust");
				lcd_gotoxy(0,2);
				rprintf("Push to Save");
				//------------------------------------------
				// Format and print current value in dBm
				lcd_gotoxy(0,3);
				rprintf("->");
		
				int16_t val_sub = current_selection;
				if (val_sub < 0) val_sub *= -1;
				int16_t val = val_sub / 10;
				val_sub = val_sub % 10;
				if (current_selection < 0) rprintf("-");
				else rprintf(" ");
				rprintf("%2d.%01udBm",val, val_sub);

				//------------------------------------------
				// Format and print current value in Vp-p/50ohms
				lcd_gotoxy(11,3);
				double vpp = pow(10,(current_selection-300)/10.0/10.0);// Convert to Watts
				vpp = sqrt(vpp*50) * 2 * sqrt(2);	// Convert to Vp-p
				print_v(vpp);
				rprintf("p-p");
			}
		
			// Enact selection by saving in EEPROM
			if (Status & SHORT_PUSH)
			{
				Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		
				// Save modified value
				eeprom_write_block(&R.calibrate[0].db10m, &E.calibrate[0].db10m, sizeof (R.calibrate[0].db10m));
				R.calibrate[0].ad = ad8317_ad;
				eeprom_write_block(&R.calibrate[0].ad, &E.calibrate[0].ad, sizeof (R.calibrate[0].ad));
		
				Status &=  ~SHORT_PUSH;			// Clear pushbutton status
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Calibrating");
				Calibrating = 1;				// Calibration now in progress
			}
		}
		else
		{
			if(calTimer<20)						// Calibrate for 2 seconds
			{
				calTimer++;						// Increment by 0.1 seconds

				//-------------------------------------------------------------------
				// Set current attenuation level and update attenuator outputs
				//
				R.Attenuators = ATT_FIRST | ATT_SECOND | ATT_THIRD | 0xfc;// Set all attenuators
				ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
				ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
			}
			else
			{
				Calibrating = 0;				// Calibration done
				calTimer = 0;					// We are done waiting, set second value
				R.calibrate[1].db10m = R.calibrate[0].db10m - R.first_Attenuator;
				eeprom_write_block(&R.calibrate[1].db10m, &E.calibrate[1].db10m, sizeof(R.calibrate[1].db10m));
				R.calibrate[1].ad = ad8317_ad;
				eeprom_write_block(&R.calibrate[1].ad, &E.calibrate[1].ad, sizeof(R.calibrate[1].ad));
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Calibration Done");
				_delay_ms(500);
				Status &= ~SHORT_PUSH;			// Clear pushbutton status
				Menu_Mode &=  ~CONFIG;			// We're done
				menu_level = 0;					// We are done with this menu level
				LCD_upd = FALSE;				// Make ready for next time
			}
		}
	}
}


//--------------------------------------------------------------------
// AutoSet dBm for First Attenuator Menu functions
//--------------------------------------------------------------------
void first_db_menu(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.autoattenuate;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > AUTO_MAX_VALUE) current_selection = AUTO_MAX_VALUE;
		if(current_selection < AUTO_MIN_VALUE) current_selection = AUTO_MIN_VALUE;

		// Store Current value in running storage
		R.autoattenuate = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	
		rprintf("Atten Auto Engage:");

		lcd_gotoxy(0,1);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcd_gotoxy(0,3);
		rprintf("->");
		if (current_selection == AUTO_MAX_VALUE)
		{
			rprintf("Permanently -OFF-");
		}
		else if (current_selection == AUTO_MIN_VALUE)
		{
			rprintf("Permanently -ON-");
		}
		else
		{
			int16_t val_sub = current_selection;
			if (val_sub < 0) val_sub *= -1;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			if (current_selection < 0) rprintf("  -");
			else rprintf("   ");
			rprintf("%2d.%01udBm",val, val_sub);
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.autoattenuate, &E.autoattenuate, sizeof (R.autoattenuate));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}


//--------------------------------------------------------------------
// Attenuator Submenu functions
//--------------------------------------------------------------------
void attenuator_menu_level2(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	if      (menu_level == ATT_SET1_MENU)current_selection = R.front_Attenuator;
	else if (menu_level == ATT_SET2_MENU)current_selection = R.first_Attenuator;
	else if (menu_level == ATT_SET3_MENU)current_selection = R.second_Attenuator;
	else if (menu_level == ATT_SET4_MENU)current_selection = R.third_Attenuator;
	else if (menu_level == ATT_SET5_MENU)current_selection = R.fixed_Attenuation;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		int16_t max_value = 700;		// Highest Attenuation value in dB * 10
		int16_t min_value = 0;			// Lowest Attenuation value in dB * 10
		if(current_selection > max_value) current_selection = max_value;
		if(current_selection < min_value) current_selection = min_value;

		// Store Current value in running storage
		if      (menu_level == ATT_SET1_MENU)R.front_Attenuator = current_selection;
		else if (menu_level == ATT_SET2_MENU)R.first_Attenuator = current_selection;
		else if (menu_level == ATT_SET3_MENU)R.second_Attenuator = current_selection;
		else if (menu_level == ATT_SET4_MENU)R.third_Attenuator = current_selection;
		else if (menu_level == ATT_SET5_MENU)R.fixed_Attenuation = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	

		if (menu_level == ATT_SET1_MENU)
		{
			rprintf("FrontAttenuator Adj:");
		}
		else if (menu_level == ATT_SET2_MENU)
		{
			rprintf("Attenuator 1 Adjust:");
		}
		else if (menu_level == ATT_SET3_MENU)
		{
			rprintf("Attenuator 2 Adjust:");
		}
		else if (menu_level == ATT_SET4_MENU)
		{
			rprintf("Attenuator 3 Adjust:");
		}
		else if (menu_level == ATT_SET5_MENU)
		{
			rprintf("Insertion Loss Adj:");
		}
		lcd_gotoxy(0,2);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,3);
		rprintf("Push to Save->");
		// Format and print current value
		uint16_t val_sub = current_selection;
		uint16_t val = val_sub / 10;
		val_sub = val_sub % 10;
		rprintf("%2u.%01u",val, val_sub);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		if (menu_level == ATT_SET1_MENU)
		{
			eeprom_write_block(&R.front_Attenuator, &E.front_Attenuator, sizeof (R.front_Attenuator));
		}
		else if (menu_level == ATT_SET2_MENU)
		{
			eeprom_write_block(&R.first_Attenuator, &E.first_Attenuator, sizeof (R.first_Attenuator));
		}
		else if (menu_level == ATT_SET3_MENU)
		{
			eeprom_write_block(&R.second_Attenuator, &E.second_Attenuator, sizeof (R.second_Attenuator));
		}
		else if (menu_level == ATT_SET4_MENU)
		{
			eeprom_write_block(&R.third_Attenuator, &E.third_Attenuator, sizeof (R.third_Attenuator));
		}
		else if (menu_level == ATT_SET5_MENU)
		{
			eeprom_write_block(&R.fixed_Attenuation, &E.fixed_Attenuation, sizeof (R.fixed_Attenuation));
		}

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = ATT_MENU;			// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}



//--------------------------------------------------------------------
// Attenuator Menu functions
//--------------------------------------------------------------------
void attenuators_set_menu(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = attenuator_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)attenuator_menu_items, menu_size, current_selection,1, 5, 3);

		// Indicate Current Frequency and the Frequency stored under the currently selected Channel number
		// The "stored" Frequency always changes according to which channel is currently selected.
		lcd_gotoxy(0,0);				
		rprintf("Attenuators Adjust:");
		//lcd_gotoxy(0,2);
		//rprintf("Cur Val");
		lcd_gotoxy(0,3);
		if (current_selection < 5)
		{
			uint16_t value=0;

			switch (current_selection)
			{
				case 0:
					value = R.front_Attenuator;
					break;
				case 1:
					value = R.first_Attenuator;
					break;
				case 2:
					value = R.second_Attenuator;
					break;
				case 3:
					value = R.third_Attenuator;
					break;
				case 4:
					value = R.fixed_Attenuation;
					break;
			}
			uint16_t val_sub = value;
			uint16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			rprintf("%2u.%01udB",val, val_sub);
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0:
				menu_level = ATT_SET1_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 1:
				menu_level = ATT_SET2_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 2:
				menu_level = ATT_SET3_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 3:
				menu_level = ATT_SET4_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 4:
				menu_level = ATT_SET5_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 5:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Attenuators");
				_delay_ms(500);
				Menu_Mode |= CONFIG;	// We're NOT done, just backing off
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Attenuators");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//--------------------------------------------------------------------
// Rotary Encoder Resolution
//--------------------------------------------------------------------
void encoder_menu(void)
{

	uint8_t	current_selection;			// Keep track of current Encoder Resolution

	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.encoderRes;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection=current_selection<<1;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection=current_selection>>1;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > 128) current_selection = 128;
		if(current_selection < 1) current_selection = 1;

		// Store Current value in running storage
		R.encoderRes = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	
		rprintf("Encoder ResDivide:");

		lcd_gotoxy(0,1);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcd_gotoxy(0,3);
		rprintf("->");

		int16_t val = current_selection;
		rprintf("%3u",val);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.encoderRes, &E.encoderRes, sizeof (R.encoderRes));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}


//--------------------------------------------------------------------
// Factory Reset with all default values
//--------------------------------------------------------------------
void factory_menu(void)
{
	static int8_t	current_selection;
	static uint8_t	LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = factory_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("Set all to default:");

		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)factory_menu_items, menu_size, current_selection,
				1, /*column*/ 3, 6);
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Factory Reset
				// Force an EEPROM update:
				eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
				lcd_clrscr();
				lcd_gotoxy(0,0);				
				rprintf("Factory Reset was");
				lcd_gotoxy(0,1);
				rprintf("selected.");
				lcd_gotoxy(0,2);
				rprintf("All settings reset");
				lcd_gotoxy(0,3);
				rprintf("to default values.");
				while (1);			// Bye bye, Death by Watchdog
			case 1:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode |= CONFIG;// We're NOT done, just backing off
				menu_level = 0;		// We are done with this menu level
				LCD_upd = FALSE;	// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//
//--------------------------------------------------------------------
// Manage the first level of Menus
//--------------------------------------------------------------------
//
void menu_level0(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t	LCD_upd = FALSE;	// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	if (LCD_upd == FALSE)				// Need to update LCD
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = level0_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("Configuration Menu:");

		// Print the Menu
		lcd_scroll_Menu((char**)level0_menu_items, menu_size, current_selection,1, 0, 3);
	}

	if (Status & SHORT_PUSH)
	{
		Status &= ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Perform Power Calibration
				menu_level = DEBUG_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 1: // Perform Power Calibration
				menu_level = POW_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 2: // AutoSet dBm Switchover threshold for First Attenuator
				menu_level = AUTO_FIRST_ATT;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 3: // Adjust Attenuators
				menu_level = ATT_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 4:// Encoder Resolution
				menu_level = ENCODER_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 5: // Factory Reset
				menu_level = FACTORY_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;
			default:
				// Exit
				lcd_clrscr();
				lcd_gotoxy(1,1);
		   		rprintf("Return from Menu");
				Menu_Mode &=  ~CONFIG;	// We're done
				LCD_upd = FALSE;		// Make ready for next time
		}
	}
}


//
//--------------------------------------------------------------------
// Scan the Configuraton Menu Status and delegate tasks accordingly
//--------------------------------------------------------------------
//
void PushButtonMenu(void)
{
	// Select which menu level to manage
	if (menu_level == 0) menu_level0();

	else if (menu_level == DEBUG_MENU) debug_menu();

	else if (menu_level == POW_MENU) pow_calibrate_menu();

	else if (menu_level == AUTO_FIRST_ATT) first_db_menu();

	else if (menu_level == ATT_MENU) attenuators_set_menu();
	else if (menu_level == ATT_SET1_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET2_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET3_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET4_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET5_MENU) attenuator_menu_level2();

	else if (menu_level == ENCODER_MENU) encoder_menu();

	else if (menu_level == FACTORY_MENU) factory_menu();
}
