//*********************************************************************************
//**
//** Project.........: AD8317 based Intelligent Attenuator
//**
//**
//**                   Modes of operation:
//**
//**                   1) Manually adjust a chain of 30 + 30 + 60 + 0.5-31.5dB
//**                      attenuators (Mini-Circuits ZX76-31R5-PP+).
//**                      Range: 0 - 151.5dB, in 0.5dB steps.
//**                   2) Automatically adjust for a predetermined output value,
//**                      in either dBm or mV/uV.
//**                   3) RF power meter, -50dBm to +30dBm, 0 - 1000 MHz
//**
//** Platform........: AT90usb646 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2010-08-05, Loftur Jonasson, TF3LJ
//**
//** Current version.: See Att.h
//**
//** History.........: ...
//**
//*********************************************************************************


//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Pin assignment (defined in the Att.h file):

// PB0 = LCD control RS
// PB1 = LCD control RW
// PB2 = LCD control E
// PB4 = LCD D4
// PB5 = LCD D5
// PB6 = LCD D6
// PB7 = LCD D7

// PF0 = AD MUX0, input from AD8317

// PC0 = Second attenuator, ~30 dB, output inverted
// PC1 = Third  attenuator, ~60 dB, output inverted
// PC2 = ZX76-31R5-PP+ 0.5 dB bit, output inverted
// PC3 = ZX76-31R5-PP+   1 dB bit, output inverted
// PC4 = ZX76-31R5-PP+   2 dB bit, output inverted
// PC5 = ZX76-31R5-PP+   4 dB bit, output inverted
// PC6 = ZX76-31R5-PP+   8 dB bit, output inverted
// PC7 = ZX76-31R5-PP+  16 dB bit, output inverted

// PE0 = First  attenuator, ~30 dB, output inverted
// PE1 = Push Button Selector input

// PD5 = Rotary Encoder A input
// PD6 = LED output
// PD7 = Rotary Encoder B input
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------


#include "Att.h"
#include "analog.h"
#include <math.h>

EEMEM		var_t	E;						// Variables in eeprom (user modifiable, stored)
			var_t	R						// Variables in ram/flash rom (default)
					=
					{
						COLDSTART_REF		// Update into eeprom if value mismatch
					, 	ENC_RES_DIVIDE		// Initial Encoder Resolution
					,	0					// Which GainPreset is selected (0,1,2,3) 0=0 or None
					, { 0 					// Gainset 0 always = 0
					, GAINSET1				// Three gain settings in dB x 10
					, GAINSET2              // Attenuators take a negative value
					, GAINSET3 }
					,	0b011111100			// Initially all Attenuators are selected
											// First is  0b100000000, set = inverted
											// Second is 0b000000001, set = inverted
											// Third is  0b000000010, set = inverted
											// 6 bits in positions 0b011111100 are for a
											// Minicircuits ZX76-31R5-PP+
											// programmable 0 - 31.5dB attenuator
					,	300					// First attenuator in dB x 10
					,	300					// Second attenuator in dB x 10
					,	600					// Third attenuator in dB x 10
					,	18					// Fixed insertion loss, mainly through the
											// ZX76-31R5-PP+ step attenuator, dB x 10
					,{{ CAL1_NOR_VALUE
					   ,CAL1_RAW_DEFAULT}	// First Calibrate point, db*10 + AD value
					, { CAL2_NOR_VALUE
					   ,CAL2_RAW_DEFAULT}}	// Second Calibrate point, db*10 + AD value	
					,	AUTO_NOR_VALUE		// Max Power input before autoswitching the
											// first 30dB attenuator in (dBm x 10).	
					,	0					// Flag to compensate for ZX76-31R5 SWR at 0-30MHz
											// Takes values 0, 1, 2 or 3
											// (no comp, 0-30 MHz, 145 MHz or 435 MHz)
					};

uint8_t		Status = 0;						// Contains various status flags
uint8_t		Menu_Mode;						// Menu Mode Flags
int16_t		ad8317_ad;						// Measured A/D value from the AD8317
double		power_in_db;					// Calculated input power in dBm
double		power_in_db_pep;				// Calculated PEP input power in dBm
double		power_out_db;					// Calculated output power in dBm
double		power_in_mw;					// Calculated input power in mW
double		power_in_mw_pep;				// Calculated PEP input power in mW
double		power_out_mw;					// Calculated output power in mW
double		power_out_db_lock;				// Power Output reference for SigGen function 
int8_t		modulation_index;				// AM modulation index in %
double		ad8317_real;					// Measured AD8317 power in dBm
int16_t		total_attenuation;				// Current real attenuation in dB x 10

int8_t		pow_Tracker;					// Locks/tracks output power in Signal Generator mode

const		uint8_t aref = 0;				// External 4.096V reference is connected


//
//-----------------------------------------------------------------------------------------
// 			Read ADC Mux
//-----------------------------------------------------------------------------------------
//
int16_t adc_Read(uint8_t mux)
{
    uint8_t low;

    ADCSRA = (1<<ADEN) | ADC_PRESCALER;             // enable ADC
    ADCSRB = (1<<ADHSM) | (mux & 0x20);             // high speed mode
    ADMUX = aref | (mux & 0x1F);                    // configure mux input
    ADCSRA = (1<<ADEN) | ADC_PRESCALER | (1<<ADSC); // start the conversion
    while (ADCSRA & (1<<ADSC)) ;                    // wait for result
    low = ADCL;                                     // must read LSB first
    return (ADCH << 8) | low;                       // must read MSB only once!
}


//
//-----------------------------------------------------------------------------------------
// 			Convert Input Voltage Reading into Input Power
//-----------------------------------------------------------------------------------------
//
void measure_Input_Pwr(void)
{
	double 	delta_db;
	int 	delta_ad;
	double	delta_ad1db;
	double	ad8317_raw;

	delta_db = (double)((R.calibrate[1].db10m - R.calibrate[0].db10m)/10.0);
	delta_ad = R.calibrate[1].ad - R.calibrate[0].ad;
	delta_ad1db = delta_db/delta_ad;

	ad8317_raw = (ad8317_ad - R.calibrate[0].ad) * delta_ad1db + R.calibrate[0].db10m/10.0;

	// Compensate for known errors in the curve above -7dBm and below -52dBm, 
	// best fit when worse than 0.25dB
	// (See AD8317 datasheet (rev B), page 7, Typical Performance Characteristics,
	//  Figure 3. VOUT and Log Conformance vs. Input Amplitude at 900 MHz)
	if(ad8317_raw > -7.0)
	{
		ad8317_real = ad8317_raw + (ad8317_raw + 7.0) * 2/5.0;
	}
	if(ad8317_raw < -52.0)
	{
		ad8317_real = ad8317_raw - (-52.0 - ad8317_raw) * 2/5.0;
	}
	else ad8317_real = ad8317_raw;
}


//
//-----------------------------------------------------------------------------------------
// 			Ensure that first attenuator is set if input power is above a set value
//-----------------------------------------------------------------------------------------
//
void power_Protect(void)
{
	uint16_t	atten_add = 0;	// Value of first attenuator in dB*10, if selected
	static uint8_t wait;		// Patience timer, this routine is normally called
								// once every 100ms

	//
	// If input power is above what is specified in R.autoattenuate, then force set first attenuator
	//
	// Add, if 1st attenuator is selected
	if (R.Attenuators & ATT_FIRST)
	{
		atten_add = R.first_Attenuator;
	}

	// Fixed at Always OFF
	if (R.autoattenuate == AUTO_MAX_VALUE)
	{
		Status &= ~HI_INPUT;
	}
	// Fixed at Always ON
	else if (R.autoattenuate == AUTO_MIN_VALUE)
	{
		Status |= HI_INPUT;
	}

	// If input power is larger than set maximum, then set flag for 1st Attenuator to be set.
	else if ((int16_t)(ad8317_real*10 + atten_add) > R.autoattenuate)
	{
		Status |= HI_INPUT;
	}
	// Else, if power is less, then release 1st Attenuator after a set amount of time
	else
	{
		wait++;
		if (wait == 20)			// If 2 seconds with low power, then release attenuator
		{
			Status &= ~HI_INPUT;
			wait = 0;
		}
	}
}


//
//-----------------------------------------------------------------------------------------
// 			Calculate input/output power
//-----------------------------------------------------------------------------------------
//
void calc_Power(void)
{
	#define	BUFFER	200					// Max/Min/Average Buffer size

	// For measurement of peak and average power
	static int16_t p_avg[BUFFER];		// One second window
	static uint16_t i;
	static uint16_t	siggen_stable;		// Patience for stable output for SigGen

	int16_t	p_inst;						// Instantaneous input power (dB * 100)
	int16_t	p_min, p_max;				// Min and Max values observed
	double	v_min, v_max, v_avg;

	// Determine Input power based on the setting of the First attenuator
	if (R.Attenuators & ATT_FIRST)
	{
		p_inst = 100*ad8317_real + 10*R.first_Attenuator;
	}
	else
	{
		p_inst = 100*ad8317_real;
	}

	// Find peaks and averages (done with voltges rather than dB)
	p_avg[i] = p_inst;					// Store dB value in ringbuffer
	i++;
	if (i == BUFFER) i = 0;

	p_min =  32767;						// Maybe a bit excessive as max/min in dB*100 :)
	p_max = -32768;
	for (uint16_t j = 0; j < BUFFER; j++)// Retrieve the min/max values out of the measured window
	{
		if (p_min > p_avg[j]) p_min = p_avg[j];
		if (p_max < p_avg[j]) p_max = p_avg[j];
	}

	// Calculate max/min and average voltage
	v_min = pow(10,p_min/100.0/20.0);	// Normalize dB*100 and convert to Voltage
	v_max = pow(10,p_max/100.0/20.0);
	v_avg = (v_min + v_max) / 2.0;		// Average voltage in the presence of modulation

	// Average power input
	power_in_db = 20 * log10(v_avg);

	if ((Menu_Mode == ATTENUATOR) || (Menu_Mode == SIG_GEN))
	{
		//-------------------------------------------------------------------------------
		// Below is an experiment to compensate for impedance errors in the 
		// Minicircuits ZX76-31R5 Attenuator at low attenuation settings
		//
		// The problem with this approach is that it is frequency dependent.  At 
		// higher frequencies the standing wave pattern error oscillates up to +/- 1.1dB
		// on either side of the true value.  This error totally disappears with 30 dB
		// attenuation or better.
		//	
		//  Table of measured input values, at -10dBm constant/true input
		//  Att (dB) 1,8   6.8  11.8  16.8  21.8  26.8  31.8
		//---------------------------------------------------
		//  10 MHz  -8.9  -9.3  -9.3  -9.5  -9.6  -9.7 -10.0
		//  20 MHz  -9.0  -9.4  -9.3  -9.5  -9.7  -9.8 -10.0
		//  30 MHz  -9.1  -9.5  -9.4  -9.6  -9.7  -9.8 -10.0
		//  50 MHz  -9.6  -9.7  -9.7  -9.8  -9.9  -9.9 -10.1
		// 100 MHz -10.6 -10.4 -10.5 -10.4 -10.2 -10.2 -10.3
		// 145 MHz -11.0 -10.5 -10.6 -10.4 -10.2 -10.1 -10.1
		// 150 MHz -10.9 -10.5 -10.6 -10.4 -10.2 -10.1 -10.2
		// 200 MHz -10.2 -10.0  -9.9  -9.9  -9.8  -9.8 -10.1
		// 435 MHz  -9.7  -9.4  -9.4  -9.2  -9.1  -9.0  -9.6
		// 450 MHz  -9.2  -9.2  -9.4  -9.3  -9.1  -9.0  -9.6
		//
		//---------------------------------------------------------------
		int16_t current_att = total_attenuation - R.fixed_Attenuation;
		
		//  20 MHz  -9.0  -9.4  -9.3  -9.5  -9.7  -9.8 -10.0
		if (R.HF_compensation == 1)	// 0 - 30 MHz (20 MHz)
		{
			// Error 1.0dB at 0dB, 0.6dB at 5dB attenuation
			if(current_att <= 50)	// 5dB or less
			{
				power_in_db  = power_in_db  - 0.6 + (current_att - 50)*0.4/50;
			}
			// Error 0.6dB at 5dB, 0.7dB at 10dB attenuation
			else if(current_att <= 100)	// 10dB or less
			{
				power_in_db  = power_in_db  - 0.7 - (current_att - 100)*0.2/50;
			}
			// Error 0.7dB at 10dB, 0.3dB at 20dB attenuation
			else if(current_att <= 200)	// 20dB or less
			{
				power_in_db  = power_in_db  - 0.3 + (current_att - 200)*0.4/100;
			}
			// Error 0.3B at 20dB, 0dB at 30dB attenuation
			else if(current_att <= 300)	// less than 30dB
			{
				power_in_db  = power_in_db  + (current_att - 300)*0.3/100;
			}
		}
		// 145 MHz -11.0 -10.5 -10.6 -10.4 -10.2 -10.1 -10.1
		if (R.HF_compensation == 2)	// 145 MHz
		{
			// Error -1.0dB at 0dB, -0.5dB at 5dB attenuation
			if(current_att <= 50)	// 5dB or less
			{
				power_in_db  = power_in_db  + 0.5 - (current_att - 50)*0.5/50;
			}
			// Error -0.5dB at 5dB, -0.6dB at 10dB attenuation
			else if(current_att <= 100)	// 10dB or less
			{
				power_in_db  = power_in_db  + 0.6 + (current_att - 100)*0.1/50;
			}
			// Error -0.6dB at 10dB, -0.2dB at 20dB attenuation
			else if(current_att <= 200)	// 20dB or less
			{
				power_in_db  = power_in_db  + 0.2 - (current_att - 200)*0.4/100;
			}
			// Error -0.2B at 20dB, 0dB at 30dB attenuation
			else if(current_att <= 300)	// less than 30dB
			{
				power_in_db  = power_in_db  - (current_att - 300)*0.2/100;
			}
		}
		// 435 MHz      -9.7  -9.4  -9.4  -9.2  -9.1  -9.0  -9.6
		// normalized: -10.1  -9.8  -9.8  -9.6  -9.5  -9.4 -10.0 
		if (R.HF_compensation == 3)	// 435 MHz
		{
			// Error -0.1dB at 0dB, 0.2dB at 5dB attenuation
			if(current_att <= 50)	// 5dB or less
			{
				power_in_db  = power_in_db  - 0.2 - (current_att - 50)*0.3/50;
			}
			// Error 0.2dB at 5dB, 0.2dB at 10dB attenuation
			else if(current_att <= 100)	// 10dB or less
			{
				power_in_db  = power_in_db  - 0.2;
			}
			// Error 0.2dB at 10dB, 0.4dB at 15dB attenuation
			else if(current_att <= 150)	// 15dB or less
			{
				power_in_db  = power_in_db  - 0.4 + (current_att - 150)*0.2/50;
			}
			// Error 0.4dB at 15dB, 0.6dB at 25dB attenuation
			else if(current_att <= 250)	// 25dB or less
			{
				power_in_db  = power_in_db  - 0.6 + (current_att - 250)*0.2/100;
			}
			// Error 0.6dB at 25dB, 0.0dB at 30dB attenuation
			else if(current_att <= 300)	// less than 30dB
			{
				power_in_db  = power_in_db  + (current_att - 300)*0.6/50;
			}
		}
	}
	//-------------------------------------------------------------------------------

	// Average power input, milliwatts
	power_in_mw = pow(10,power_in_db/10.0);	// Input Power in mW

	// PEP (1 second) input
	power_in_db_pep = 20 * log10(v_max);
	power_in_mw_pep = pow(10,power_in_db_pep/10.0);	// Input Power in mW

	// Amplitude Modulation index
	modulation_index = (int8_t) (100.0 * (v_max-v_avg)/v_avg);

	// Calculate current output power (dBm (double)) based on the setting of all attenuators
	power_out_db = power_in_db - (double)total_attenuation/10.0;
	power_out_mw = pow(10,power_out_db/10.0);	// Output Power in mW

	// Used with Signal Generator, waits for stable output
	if (Status & LCK_CHANGE)
	{
		if(siggen_stable > BUFFER)		// Wait for a complete fill of circular buffer
		{
			Status &= ~LCK_CHANGE;		// Clear Power Change flag
			Status |= POW_LOCKED;		// Indicate Power is locked
			siggen_stable = 0;
			power_out_db_lock = power_out_db;// Power Output reference for SigGen function 
		}
		else siggen_stable++;
	}
}


//
//-----------------------------------------------------------------------------------------
// 			Track Output Power and keep stable, by signalling attenuator changes
//-----------------------------------------------------------------------------------------
//
void track_Output(void)
{
	// If we have stable initial output, then track output power for attenuator update
	if(Status & POW_LOCKED)
	{
		// Is actual output power too low?
		if (power_out_db_lock > (power_out_db + 0.26))
		{
			Status |= POW_CHANGE;				// Set Power Tracker flag
			pow_Tracker = 1;			
		}
		// Is actual output power too high?
		if (power_out_db_lock < (power_out_db - 0.26))
		{
			Status |= POW_CHANGE;				// Set Power Tracker flag
			pow_Tracker = -1;			
		}
	}
}


//
//-----------------------------------------------------------------------------------------
// 			Set Attenuators for Power Measurement function
//-----------------------------------------------------------------------------------------
//
void set_Attenuators_Powermeasure(void)
{
	//-------------------------------------------------------------------
	// Set current attenuation level
	//
	R.Attenuators = ATT_SECOND | ATT_THIRD | 0xfc; // 0xfc = Stepp Att, all on

	// Is first attenuator to be force set due to high input power?
	if (Status & HI_INPUT)
	{
		R.Attenuators |= ATT_FIRST;
	}

	// Is input power under limit, first attenuator not force set?
	if ((Status & HI_INPUT) == 0)
	{
		R.Attenuators &= ~ATT_FIRST;
	}

	//-------------------------------------------------------------------
	// Update attenuator outputs

	// And release as appropriate
	// Note inverted logic for att1, att2 & att3
	ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
	ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
}


//
//-----------------------------------------------------------------------------------------
// 			Set Attenuators for Adjustible Attenuate functions
//-----------------------------------------------------------------------------------------
//
void set_Attenuators(void)
{
	int16_t	attenuation;		// dB * 10
	int16_t	attenuation_sub;	// dB * 10

	//-------------------------------------------------------------------
	// Retrieve the current attenuation level
	//
	attenuation = R.fixed_Attenuation;	// Initialize, add insertion loss from ZX76-31R5
	// add current fixed attenuation level
	if (R.Attenuators & ATT_FIRST)
		attenuation += R.first_Attenuator;
	if (R.Attenuators & ATT_SECOND)
		attenuation += R.second_Attenuator;
	if (R.Attenuators & ATT_THIRD)
		attenuation += R.third_Attenuator;
	// add current variable attenuation level (0-31.5dB).
	attenuation = attenuation + ((R.Attenuators & 0x00fc)>>2) * 5;

	//-------------------------------------------------------------------
	// If Encoder change, then Update attenuation,
	// based on Encoder direction, +/-0.5dB each time 
	//
	if (Status & ENC_CHANGE)
	{
		Status &= ~ENC_CHANGE;				// Clear encoder bit
		Status &= ~POW_LOCKED;				// Clear Power Locked bit
		Status |= LCK_CHANGE;				// Indicate we need to lock to new power output,
											// if in Signal Generator mode
		if (encOutput > 0)
		{
			attenuation = attenuation + 5;	// Add 0.5 dB
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			attenuation = attenuation - 5;	// Subtract 0.5 dB
			encOutput = 0;
		}

	}

	//-------------------------------------------------------------------
	// If Power Tracker requests an encoder change, then Update attenuation,
	// in 0.5dB increments/decrements
	//
	if (Status & POW_CHANGE)
	{
		Status &= ~POW_CHANGE;				// Clear Power Tracker bit

		if (pow_Tracker > 0)				// Increase power out
		{
			attenuation = attenuation - 5;	// Add 0.5 dB
			pow_Tracker = 0;
		}
		else if (pow_Tracker < 0)			// Decrease power out
		{
			attenuation = attenuation + 5;	// Subtract 0.5 dB
			pow_Tracker = 0;
		}
	}

	//-------------------------------------------------------------------
	// Select attenuators to fullfil the new_attenuation asked for
	//
	// Initialize, add insertion loss from ZX76-31R5:
	R.Attenuators = 0;
	attenuation_sub = attenuation - R.fixed_Attenuation;


	// Is first attenuator to be force set due to high input power?
	if (Status & HI_INPUT)
	{
		R.Attenuators |= ATT_FIRST;
		attenuation_sub -= R.first_Attenuator;
	}
	// Set third attenuator
	if (attenuation_sub >= R.third_Attenuator)
	{
		R.Attenuators |= ATT_THIRD;
		attenuation_sub -= R.third_Attenuator;
	}
	// Set second attenuator
	if (attenuation_sub >= R.second_Attenuator)
	{
		R.Attenuators |= ATT_SECOND;
		attenuation_sub -= R.second_Attenuator;
	}
	// Is input power under limit, first attenuator not force set?
	if ((Status & HI_INPUT) == 0)
	{
		// Set first attenuator
		if (attenuation_sub >= R.first_Attenuator)
		{
			R.Attenuators |= ATT_FIRST;
			attenuation_sub -= R.first_Attenuator;
		}
	}
	//-------------------------------------------------------------------
	// Have we reached the end of our rope?
	//
	if (attenuation_sub <= 0) // We already have more attenuation than we asked for
	{
		// Do nothing, equals: 
		// R.Attenuators |= 0;
	}
	else if (attenuation_sub >= 315)// Remainder larger than max attenuator setting
	{
		R.Attenuators |= 0xfc;			// Add everything we have got left
	}
	// Nope, we're good, set the 31.5dB variable attenuator
	else
	{
		R.Attenuators = R.Attenuators | (((attenuation_sub)/5)<<2);  // Set 6 bits for 31.5dB att
	}


	//-------------------------------------------------------------------
	// Update attenuator outputs

	// And release as appropriate
	// Note inverted logic for att1, att2 & att3
	ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
	ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
/*
	//-------------------------------------------------------------------
	// Retrieve the current attenuation level once again (a bit redundant) 
	//
	attenuation = 0;	// Initialize
	// add current fixed attenuation level
	if (R.Attenuators & ATT_FIRST)
		attenuation += R.first_Attenuator;
	if (R.Attenuators & ATT_SECOND)
		attenuation += R.second_Attenuator;
	if (R.Attenuators & ATT_THIRD)
		attenuation += R.third_Attenuator;
	// add current variable attenuation level (0-31.5dB).
	attenuation = attenuation + (((R.Attenuators & 0x007c)>>2) * 5) + R.fixed_Attenuation;
*/
	// Report current attenuation
	total_attenuation = attenuation;

}



//
//-----------------------------------------------------------------------------------------
// 							Do stuff while not serving USB
//
// This task runs in an endless loop, whenever the USB tasks are not stealing clock cycles
//-----------------------------------------------------------------------------------------
//
void maintask(void)
{
	// Now we can do all kinds of business, such as scanning Rotary Encoder, updating LCD etc...
	static uint16_t lastIteration, lastIteration1, lastIteration2;	// Counters to keep track of time
	uint16_t Timer1val, Timer1val1, Timer1val2;		// Timers used for 100ms and 10ms polls
	static uint8_t pushcount=0;						// If Shaft Encoder, then used to time a push button (max 2.5s)
	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be run through as often as possible
	// currently measured to be approximately once every 25 - 50 us
	//-------------------------------------------------------------------------------
	{
		encoder_Scan();								// Scan the Rotary Encoder

		#if FAST_LOOP_THRU_LED						// Blink a LED every time when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;					// Blink a led
		#endif
		
		//while (Status & REBOOT);					// If REBOOT flag is set, then get
		//											// stuck here, and reboot by watchdog
	}
	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 5ms
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	Timer1val1 = TCNT1/328; // get current Timer1 value, changeable every ~5ms
	if (Timer1val1 != lastIteration1)				// Once every 5ms, do stuff
	{
		lastIteration1 = Timer1val1;				// Make ready for next iteration
		#if MS_LOOP_THRU_LED						// Blink LED every 0ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif

		ad8317_ad = adc_Read(0);					// Measure input voltage from AD8317
		measure_Input_Pwr();						// Convert to input Power in dBm
		calc_Power();								// Calculate Input/Output Power, includes a 1 second
													// sliding window of the last 200 samples
	}

	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/100th of a second (10ms)
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	Timer1val2 = TCNT1/656; // get current Timer1 value, changeable every ~1/100th sec
	if (Timer1val2 != lastIteration2)				// Once every 1/100th of a second, do stuff
	{
		lastIteration2 = Timer1val2;				// Make ready for next iteration

		#if MED_LOOP_THRU_LED						// Blink LED every 10ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif

		
		if ((Menu_Mode == POWER_METER1) || (Menu_Mode == POWER_METER2))
		{
			set_Attenuators_Powermeasure();			// Set attenuators up for Power Measure (2nd+3rd always on)
		}
		else if (Menu_Mode == ATTENUATOR)
		{
			set_Attenuators();						// Set the attenuators if Encoder input or if Power protect
		}
		else if (Menu_Mode == SIG_GEN)
		{
			track_Output();							// Track output power for attenuator update
			set_Attenuators();						// Set the attenuators for constant output + Power protect
		}
		else if (Menu_Mode & CONFIG)
		{

		}
	}


	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/10th of a second
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	Timer1val = TCNT1/6554; // get current Timer1 value, changeable every ~1/10th sec
	if (Timer1val != lastIteration)	// Once every 1/10th of a second, do stuff
	{
		lastIteration = Timer1val;					// Make ready for next iteration

		#if SLOW_LOOP_THRU_LED						// Blink LED every 100ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;					// Blink a led
		#endif

		//-------------------------------------------------------------------
		// Ensure first attenuator is set if input power becomes too high.
		//
		power_Protect();

		//-------------------------------------------------------------------
		// Read Pushbutton state
		//
		// Enact Long Push (pushbutton has been held down for a certain length of time):
		if (pushcount >= ENC_PUSHB_MAX)				// "Long Push", goto Configuration Mode
		{
			Menu_Mode |= CONFIG;					// Switch into Configuration Menu, while
													// retaining memory of runtime function

			Status |= LONG_PUSH;					// Used with Configuraton Menu functions	
			pushcount = 0;							// Initialize push counter for next time
		}
		// Enact Short Push (react on release if only short push)
		else if (ENC_PUSHB_INPORT & ENC_PUSHB_PIN) 	// Pin high = just released, or not pushed
		{
			// Do nothing if this is a release after Long Push
			if (Status & LONG_PUSH)					// Is this a release following a long push?
			{
					Status &= ~LONG_PUSH;			// Clear pushbutton status
			}
			// Do stuff on command, cycle through modes.
			else if (pushcount >= ENC_PUSHB_MIN)	// Check if this is more than a short spike
			{	
				if (Menu_Mode & CONFIG)
					Status |= SHORT_PUSH;			// Used with Configuraton Menu functions	

				else if (Menu_Mode == POWER_METER1)
					Menu_Mode = POWER_METER2;
				else if (Menu_Mode == POWER_METER2)
					Menu_Mode = ATTENUATOR;
				else if (Menu_Mode == ATTENUATOR)
					Menu_Mode = SIG_GEN;
				else if (Menu_Mode == SIG_GEN)
					Menu_Mode = POWER_METER1;
			}
			pushcount = 0;							// Initialize push counter for next time
		}
		else if (!(Status & LONG_PUSH))				// Button Pushed, count up the push timer
		{											// (unless this is tail end of a long push,
			pushcount++;							//  then do nothing)
		}

		//-------------------------------------------------------------------
		// Different Menu functions which are reiterated at 10Hz
		//
		if (Menu_Mode & CONFIG)					// Pushbutton Configuration Menu
		{
			PushButtonMenu();
		}	
		else if (Menu_Mode == POWER_METER1)		// Power Meter
		{
			lcd_display_power_meter1();
		}
		else if (Menu_Mode == POWER_METER2)		// Power Meter
		{
			lcd_display_power_meter2();
		}
		else if (Menu_Mode == ATTENUATOR)		// Attenuator
		{
			lcd_display_attenuator();
		}
		else if (Menu_Mode == SIG_GEN)			// Signal Generator
		{
			lcd_display_siggen();
		}
	}

	wdt_reset();									// Whoops... must remember to reset that running watchdog
}



//
//-----------------------------------------------------------------------------------------
// 			Setup Ports, timers, start the works and never return, unless reset
//								by the watchdog timer
//						then - do everything, all over again
//-----------------------------------------------------------------------------------------
//
int main(void)
{
	MCUSR &= ~(1 << WDRF);							// Disable watchdog if enabled by bootloader/fuses
	wdt_disable();

	clock_prescale_set(clock_div_1); 				// with 16MHz crystal this means CLK=16000000

	//------------------------------------------
	// 16-bit Timer1 Initialization
	TCCR1A = 0; //start the timer
	TCCR1B = (1 << CS12); // prescale Timer1 by CLK/256
	// 16000000 Hz / 256 = 62500 ticks per second
	// 16-bit = 2^16 = 65536 maximum ticks for Timer1
	// 65536 / 62500 = ~1.05 seconds
	// so Timer1 will overflow back to 0 about every 1 seconds
	// Timer1val = TCNT1; // get current Timer1 value

	//------------------------------------------
	// Init and set output for LED
	LED_DDR = LED;
	LED_PORT = 0;

	//------------------------------------------
	// Init and set outputs for attenuators
	ATT_DDR01 = 0xff;								// Set pins for output
	ATT_DDR02 = ATT_DDR02 | ATT_P2_PIN;
	ATT_PORT01 = R.Attenuators & 0x00ff;			// Set lower 8 bits
	ATT_PORT02 = (R.Attenuators & 0xff00)>>8;		// Set higher 8 bits 

	//------------------------------------------
	// Init Pushbutton input
	ENC_PUSHB_DDR = ENC_PUSHB_DDR & ~ENC_PUSHB_PIN;	// Set pin for input
	ENC_PUSHB_PORT= ENC_PUSHB_PORT | ENC_PUSHB_PIN;	// Set pullup

	//------------------------------------------
	// Set run time parameters to Factory default under certain conditions
	//
	// Enforce "Factory default settings" when firmware is run for the very first time after
	// a fresh firmware installation with a new "serial number" in the COLDSTART_REF #define
	// This may be necessary if there is garbage in the EEPROM, preventing startup
	// To activate, roll "COLDSTART_REF" Serial Number in the Att.h file
	if (eeprom_read_byte(&E.EEPROM_init_check) != R.EEPROM_init_check)
	{
		eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
	}
	else
	{
		eeprom_read_block(&R, &E, sizeof(E));		// Load the persistent data from eeprom
	}

	set_Attenuators();								// Init Attenuators

   	lcd_init(LCD_DISP_ON);							// Init the LCD

	rprintfInit(lcd_data);							// Init AVRLIB rprintf()

	// Initialize the LCD bargraph, load the bargraph custom characters
	lcd_bargraph_init();

	//------------------------------------------
	// LCD Print Version information
	lcd_clrscr();
	lcd_gotoxy(8,3);
	rprintf("V%s, TF3LJ", VERSION);
	_delay_ms(500);
	lcd_gotoxy(0,0);
	rprintf("Intelligent...");
	_delay_ms(500);
	lcd_gotoxy(7,1);
	rprintf("...Attenuator");
	_delay_ms(3000);

	lcd_clrscr();
	lcd_gotoxy(8,3);
	rprintf("V%s, TF3LJ", VERSION);
	_delay_ms(333);
	lcd_gotoxy(0,0);
	rprintf("Power Meter");
	_delay_ms(333);
	lcd_gotoxy(0,1);
	rprintf("Attenuator");
	_delay_ms(333);
	lcd_gotoxy(0,2);
	rprintf("Signal Generator");
	_delay_ms(3000);

	// Start the USB task and "spawn" the three all important functions
	// for the Mobo:
	// void usbFunctionWrite(USB_Notification_Header_t *, unsigned char *, unsigned char);
	// unsigned char usbFunctionSetup(USB_Notification_Header_t *);
	// void maintask(void);

	wdt_enable(WDTO_1S);							// Start the Watchdog Timer, 1 sec

	encoder_Init();									// Init Rotary encoder

	Menu_Mode = DEFAULT_MODE;						// Power Meter Mode is normal default

	Status |= LCK_CHANGE;							// Seek Power lock (used to track output
													// power when in SigGen mode)
	
	// Start the works, we're in business
	while (1)
	{
		maintask();
	}
}

