//*********************************************************************************
//**
//** Project.........: AD8317 based Intelligent Attenuator
//**
//**
//**                   Modes of operation:
//**
//**                   1) Manually adjust a chain of 30 + 30 + 60 + 0.5-31.5dB
//**                      attenuators (Mini-Circuits ZX76-31R5-PP+).
//**                      Range: 0 - 151.5dB, in 0.5dB steps.
//**                   2) Automatically adjust for a predetermined output value,
//**                      in either dBm or mV/uV.
//**                   3) RF power meter, -50dBm to +30dBm, 0 - 1000 MHz
//**
//** Platform........: AT90usb646 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2010-08-05, Loftur Jonasson, TF3LJ
//**
#define                VERSION "1.00"   // 2012-04-13
//**
//** History.........: Check the Att.c file
//**
//*********************************************************************************

#ifndef _TF3LJ_ATT_H_
#define _TF3LJ_ATT_H_ 1

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include <avr/power.h>
#include <rprintf.h>							// AVRLIB (not AVRLibc) functions
#include <lcd.h>	

//-----------------------------------------------------------------------------
// Features Selection
//-----------------------------------------------------------------------------  
// EEPROM settings Serial Number. Increment this number when firmware mods necessitate
// fresh "Factory Default Settings" to be forced into the EEPROM at first boot after
// an upgrade
#define COLDSTART_REF		0x02// When started, the firmware examines this "Serial Number
								// and enforce factory reset if there is a mismatch.
								// This is useful if the EEPROM structure has been modified
//-----------------------------------------------------------------------------
// LED Blink
//
// None, or only one of the four should be selected
#define FAST_LOOP_THRU_LED	0	// Blink PB2 LED every time, when going through the mainloop *OR*
#define	MS_LOOP_THRU_LED	1	// Blink PB2 LED every 1ms, when going through the mainloop *OR*
#define	MED_LOOP_THRU_LED	0	// Blink PB2 LED every 10ms, when going through the mainloop *OR*
#define SLOW_LOOP_THRU_LED	0	// Blink PB2 LED every 100ms, when going through the mainloop

#define ENCODER_DIR_REVERSE	0	// Reverse the direction of the Rotary Encoder



//-----------------------------------------------------------------------------
//Defs for Auto Attenuator Function (dBm *10)
#define	AUTO_MAX_VALUE		151	// If auto attenuation is set to this value, then never activate	
#define	AUTO_MIN_VALUE		-151// If auto attenuation is set to this value, then always active
#define	AUTO_NOR_VALUE		-70	// Activate auto attenuation at this value

//-----------------------------------------------------------------------------
//Defs for Auto Calibration Function (dBm *10)
#define	CAL1_MAX_VALUE		-70	// Calibration with input above this value will be inaccurate
#define	CAL1_MIN_VALUE		-200// Calibration with input below this value will be inaccurate
#define	CAL1_NOR_VALUE		-100// Default for AD8717 at -10dBm
#define	CAL1_RAW_DEFAULT	346	// Default for AD8717 at -10dBm
#define	CAL2_NOR_VALUE		-400// Default for AD8717 at -40dBm
#define	CAL2_RAW_DEFAULT	720	// Default for AD8717 at -40dBm

//-----------------------------------------------------------------------------
//DEFS for Attenuators
#define ATT_DDR01			DDRC
#define ATT_PORT01			PORTC
#define ATT_DDR02			DDRE
#define ATT_PORT02			PORTE

#define ATT_P2_PIN			(1<<0)
#define ATT_FIRST			0b100000000	// First or Front attenuator
#define ATT_SECOND			0b000000001	// Second attenuator
#define ATT_THIRD			0b000000010	// Third attenuator
										// Last 6 bits are for a Minicircuits
										// xxxxxx programmable 0 - 31.5dB attenuator

//-----------------------------------------------------------------------------
//Defs for default Gain Settings (dB *10)
#define	GAINSET1			-400// 40 dB Attenuator
#define	GAINSET2			-150// 15 dB Attenuator
#define	GAINSET3			 180// 18.0 dB Gain (amplifier)

//-----------------------------------------------------------------------------
//DEFS for LED
#define LED_PORT			PORTD		// port for the LED
#define LED_DDR				DDRD		// port for the LED
#define LED					(1<<6)		// pin for LED

//-----------------------------------------------------------------------------
//DEFS for LCD Display
// The below #defines have been moved from the lcd.h file
#define LCD_PORT			PORTB		// port for the LCD lines
#define LCD_DATA0_PORT		LCD_PORT	// port for 4bit data bit 0
#define LCD_DATA1_PORT		LCD_PORT	// port for 4bit data bit 1
#define LCD_DATA2_PORT		LCD_PORT	// port for 4bit data bit 2
#define LCD_DATA3_PORT		LCD_PORT	// port for 4bit data bit 3
#define LCD_DATA0_PIN		4			// pin  for 4bit data bit 0
#define LCD_DATA1_PIN		5			// pin  for 4bit data bit 1
#define LCD_DATA2_PIN		6			// pin  for 4bit data bit 2
#define LCD_DATA3_PIN		7			// pin  for 4bit data bit 3
#define LCD_RS_PORT			PORTB		// port for RS line
#define LCD_RS_PIN			0			// pin  for RS line
#define LCD_RW_PORT			PORTB		// port for RW line
#define LCD_RW_PIN			1			// pin  for RW line
#define LCD_E_PORT			PORTB		// port for Enable line
#define LCD_E_PIN			2			// pin  for Enable line

//-----------------------------------------------------------------------------
// A couple of #defines moved from lcd.h:
#define LCD_LINES			4			// number of visible lines of the display 
#define LCD_DISP_LENGTH		20			// visibles characters per line of the display


//-----------------------------------------------------------------------------
// DEFS for the Rotary Encoder VFO function
// Configuration of the two input pins, Phase A and Phase B
// They can be set up to use any pin on two separate input ports
#define ENC_A_PORT		PORTD				// PhaseA port register
#define ENC_A_DDR		DDRD				// PhaseA port direction register
#define ENC_A_PORTIN	PIND				// PhaseA port input register
#define ENC_A_PIN		(1 << 7)			// PhaseA port pin
#define ENC_B_PORT		PORTD				// PhaseB port register
#define ENC_B_DDR		DDRD				// PhaseB port direction register
#define ENC_B_PORTIN	PIND				// PhaseB port input register
#define ENC_B_PIN		(1 << 5)			// PhaseB port pin

#define	ENC_RES_DIVIDE		8				// Default reduction of the Encoder Resolution

//-----------------------------------------------------------------------------
// Definitions for the Pushbutton Encoder functionality
#define ENC_PUSHB_PORT		PORTE
#define ENC_PUSHB_DDR		DDRE
#define	ENC_PUSHB_INPORT	PINE
#define	ENC_PUSHB_PIN		2				// (1 << 1)

#define ENC_PUSHB_MIN		1				// Min pushdown for valid push (x 10ms)
#define	ENC_PUSHB_MAX		10				// Min pushdown for memory save (x 10ms)

//-----------------------------------------------------------------------------
// Bargraph bounds
#define BAR_MIN_VALUE		600	// -60dB
#define BAR_FULL_RANGE		900// 900dB full range
//#define	BAR_FINE_RES	160	// 20dB for full scale variation if in fine resolution mode
#define BAR_LENGTH			20	// 20 characters long


//-----------------------------------------------------------------------------
// Select Bargraph display style
#define BARGRAPH_STYLE_1	1	// Used if LCD bargraph alternatives.  N8LP LP-100 look alike bargraph         *OR*
#define BARGRAPH_STYLE_2	0	// Used if LCD bargraph alternatives.  Bargraph with level indicators          *OR*
#define BARGRAPH_STYLE_3	0	// Used if LCD bargraph alternatives.  Another bargraph with level indicators  *OR*
#define BARGRAPH_STYLE_4	0	// Used if LCD bargraph alternatives.  Original bargraph, Empty space enframed *OR*
#define BARGRAPH_STYLE_5	0	// Used if LCD bargraph alternatives.  True bargraph, Empty space is empty


//
//-----------------------------------------------------------------------------
// Miscellaneous software defines, functions and variables
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Flags

// DEFS for all kinds of Flags
extern uint8_t		Status;
#define ENC_CHANGE	(1 << 0)				// Indicates that Encoder value has been modified
#define HI_INPUT	(1 << 1)				// Input power above threshold value
#define	LONG_PUSH	(1 << 2)				// Long Push Button Action
#define SHORT_PUSH	(1 << 3)				// Short Push Button Action
#define LCK_CHANGE	(1 << 4)				// Sig Gen power has been changed, unlocked
#define POW_LOCKED	(1 << 5)				// Sig Gen power locked
#define POW_CHANGE	(1 << 6)				// Sig Gen indicates an attenuator update to track 
											// power fluctioations

// Operation Mode Flags
extern	uint8_t		Menu_Mode;				// Which Menu Mode is active
#define POWER_METER1	(1 << 0)
#define POWER_METER2	(1 << 1)
#define ATTENUATOR		(1 << 2)
#define SIG_GEN			(1 << 3)
#define MIXED			(1 << 4)
#define	CONFIG			(1 << 5)
//
#define	DEFAULT_MODE (1 << 0)				// Default Menu Mode

// Bool stuff
#define	TRUE		1
#define FALSE		0

//-----------------------------------------------------------------------------
// Structures and Unions

typedef struct {
	int16_t	db10m;							// Calibrate, value in dBm x 10
	int16_t	ad;								// A/D value
} cal_t;

typedef struct 
{
		uint8_t		EEPROM_init_check;		// If value mismatch,
											// then update EEPROM to factory settings
		int16_t		encoderRes;				// (initial) Encoder resolution
		uint8_t		which_gainset;			// which gain setting is selected?
		int16_t		gainset[4];				// first one always zero, other 3: -70 to +70dB
		uint16_t	Attenuators;			// (initial) Attenuators settings
//		int16_t		front_Attenuator;		// normally  0dB x 10 = 0 (External Att in front)
		int16_t		first_Attenuator;		// normally 30dB x 10 = 300
		int16_t		second_Attenuator;		// normally 30dB x 10 = 300
		int16_t		third_Attenuator;		// normally 60dB x 10 = 600
		int16_t		fixed_Attenuation;		// Insertion loss of ZX76-31R5 is typically 1.4dB
		cal_t		calibrate[2];			// 2 cal points, actual + measured (dBm x 10)
		int16_t		autoattenuate;			// Max Power input before autoswitching the
											// first 30dB attenuator in (dBm x 10).
		uint8_t		HF_compensation;		// Flag to compensate for ZX76-31R5 SWR at 0-30MHz
} var_t;


//-----------------------------------------------------------------------------
// Global variables
extern	EEMEM 		var_t E;				// Default Variables in EEPROM
extern				var_t R;				// Runtime Variables in Ram

extern	int16_t		total_attenuation;		// Current attenuation in dB x 10
extern 	int16_t		ad8317_ad;				// Measured A/D value from the AD8317
extern 	double 		ad8317_real;			// Calculated power measured by the AD8317
extern	double		power_in_db;			// Calculated input power in dBm
extern	double		power_in_db_pep;		// Calculated PEP input power in dBm
extern	double		power_out_db;			// Calculated output power in dBm
extern	double		power_in_mw;			// Calculated input power in mW
extern	double		power_in_mw_pep;		// Calculated PEP input power in mW
extern	double		power_out_mw;			// Calculated output power in mW
extern	int8_t		modulation_index;		// AM modulation index in %

extern	int16_t		encOutput;				// Output From Encoder


//-----------------------------------------------------------------------------
// Prototypes for functions
// Att.c
extern void			set_Attenuators(void);			// Set the attenuators if Encoder input or if Power protect

extern void 		print_v(double voltage);		// Print Voltage, input value in volts
extern void			lcd_initialize(void);			// Initialize the LCD
extern void			lcd_display(void);				// Display stuff on a LCD
extern void			lcd_display_power_meter1(void);
extern void			lcd_display_power_meter2(void);
extern void			lcd_display_attenuator(void);
extern void			lcd_display_siggen(void);
extern void			lcd_display_mixed(void);

// Att_ShaftEncoder.c
extern void 		encoder_Init(void);				// Initialise the VFO Shaft Encoder
extern void			encoder_Scan(void);				// Scan the Shaft Encoder

// Push Button and Rotary Encoder Menu functions
extern void			PushButtonMenu(void);

// LCD Bargraph stuff
extern void 		lcdProgressBar(uint16_t progress, uint16_t maxprogress, uint8_t length);
extern void			lcd_bargraph_init(void);

#endif
