//*********************************************************************************
//**
//** Project.........: AD8317 based Intelligent Attenuator
//**
//**                   PushButtonMenu.c
//**
//**                   This code is salvaged in part from the SDR-Widget Project 
//**                   (see groups.google.com/group/sdr-widget)
//**
//**                   One long push triggers the Menu functions, which utilize the Rotary
//**                   Encoder for menu choice selection, and use the push button for confirmation
//**                   of choice.  The Menu functions may be several levels deep, depending on
//**                   choice.
//**
//**                   See Att.c for further description
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2010-08-05, Loftur Jonasson, TF3LJ
//**
//** History.........: ...
//**
//*********************************************************************************

#include <stdio.h>
#include <math.h>

#include "Att.h"
#include "lcd.h"


// First Level Menu Items
const uint8_t level0_menu_size = 10;
const char *level0_menu_items[] =
				{  " 1-Inp Gain Sel",
				   " 2-Inp Gain Adjust",
				   " 3-Auto Calibrate",
				   " 4-AutoFrontAttSet",
				   " 5-Adj Attenuators",
				   " 6-ZX76 SWR Comp",
				   " 7-Adj Encoder Res",
				   " 8-Display Debug",
				   " 9-Factory Reset",
				   "10-Exit" };

// Flag for Gain Select Menu
#define GAIN_SEL_MENU		1
const uint8_t gain_sel_size = 6;
const char *gain_sel_items[] =
				{  "1-No Gain",
				   "2-Gain Preset 1",
				   "3-Gain Preset 2",
				   "4-Gain Preset 3",
				   "9-Go Back",
				   "0-Exit"};

// Flag for Set Gain values
#define GAIN_MENU		2
const uint8_t gain_menu_size = 5;
const char *gain_menu_items[] =
				{  "1-Gain Preset 1",
				   "2-Gain Preset 2",
				   "3-Gain Preset 3",
				   "9-Go Back",
				   "0-Exit"};
// keep track of which GainPreset is currently selected
int8_t			gain_selection;



// Flags for Gain Submenu functions
#define GAIN_SET1_MENU	201
#define GAIN_SET2_MENU	202
#define GAIN_SET3_MENU	203


// Flag for Reference 1 Menu Selection
#define POW_MENU		3

// Flag for AutoSet dBm for First Attenuator
#define AUTO_FIRST_ATT	4


// Flag for Set Attenuator values
#define ATT_MENU		5
const uint8_t attenuator_menu_size = 6;
const char *attenuator_menu_items[] =
				{  "1-First Att",
				   "2-Second Att",
				   "3-Third Att",
   				   "4-Insert Loss",
				   "9-Go Back",
				   "0-Exit"};

// Flags for Attenuator Submenu functions
#define ATT_SET1_MENU	501
#define ATT_SET2_MENU	502
#define ATT_SET3_MENU	503
#define ATT_SET4_MENU	504

// Flag for ZX76 SWR Error Compensation Menu
#define SWR_COMP_MENU		6
const uint8_t swr_comp_size = 6;
const char *swr_comp_items[] =
				{  "1-No Compensation",
				   "2- 0-30 MHz Comp",
				   "3-  145 MHz Comp",
				   "4-  435 MHz Comp",
				   "9-Go Back",
				   "0-Exit"};
// keep track of which SWR Compensation is currently selected
int8_t			swr_comp;


// Flag for Encoder Resolution Change
#define ENCODER_MENU	7

// Flag for Debug Screen
#define DEBUG_MENU		8

// Flag for Factory Reset
#define FACTORY_MENU	9
// Factory Reset menu Items
const uint8_t factory_menu_size = 3;
const char *factory_menu_items[] =
				{  " 1-Yes  Reset",
				   " 9-No - Go back",
				   "10-No - Exit"	};


uint16_t		menu_level = 0;						// Keep track of which menu we are in
uint8_t			menu_data = 0;						// Pass data to lower menu



//----------------------------------------------------------------------
// Display a 3 or 4 line Menu of choices
//
// **menu refers to a pointer array containing the Menu to be printed
//
// menu_size indicates how many pointers (menu items) there are in the array
//
// current_choice indicates which item is currently up for selection if
// pushbutton is pushed
//
// begin row & begin_col are the coordinates for the upper lefthand corner
// of the three or four lines to be printed
//
// lines can take the value of 3 or 4, indicating how many lines the scrolling
// menu contains
//----------------------------------------------------------------------
void lcd_scroll_Menu(char **menu, uint8_t menu_size,
		uint8_t current_choice, uint8_t begin_row, uint8_t begin_col, uint8_t lines)
{
	uint8_t a, x;


	// Clear LCD from begin_col to end of line.
	lcd_gotoxy(begin_col, begin_row);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
	lcd_gotoxy(begin_col, begin_row+1);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
	lcd_gotoxy(begin_col, begin_row+2);
	for (a = begin_col; a < 20; a++)
		rprintfChar(' ');

	// Using Menu list pointed to by **menu, preformat for print:
	// First line contains previous choice, secon line contains
	// current choice preceded with a '->', and third line contains
	// next choice
	if (current_choice == 0) x = menu_size - 1;
	else x = current_choice - 1;
	lcd_gotoxy(begin_col + 2, begin_row);
	rprintf("%s", *(menu + x));
	lcd_gotoxy(begin_col, begin_row + 1);
	rprintf("->%s", *(menu + current_choice));
	if (current_choice == menu_size - 1) x = 0;
	else x = current_choice + 1;
	lcd_gotoxy(begin_col + 2, begin_row + 2);
	rprintf("%s", *(menu + x));

	// LCD print lines 1 to 3

	// 4 line display.  Preformat and print the fourth line as well
	if (lines == 4)
	{
		if (current_choice == menu_size-1) x = 1;
		else if (current_choice == menu_size - 2 ) x = 0;
		else x = current_choice + 2;
		lcd_gotoxy(begin_col, begin_row+3);
		for (a = begin_col; a < 20; a++)
		rprintfChar(' ');
		lcd_gotoxy(begin_col + 2, begin_row + 3);
		rprintf("  %s", *(menu + x));
	}
}


//----------------------------------------------------------------------
// Menu functions begin:
//----------------------------------------------------------------------


//--------------------------------------------------------------------
// Gain Preset Select Menu functions
//--------------------------------------------------------------------
void gain_select_menu(void)
{
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			gain_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			gain_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = gain_sel_size;
		while(gain_selection >= menu_size)
			gain_selection -= menu_size;
		while(gain_selection < 0)
			gain_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)gain_sel_items, menu_size, gain_selection,1, 3, 3);

		// Indicate Currently selected Gain Preset
		lcd_gotoxy(0,0);
		rprintf("Select:");

		lcd_gotoxy(13,0);
		if (gain_selection < 4)
		{

			int16_t value=0;

			switch (gain_selection)
			{
				case 0:
					value = 0;	// No gain
					break;
				default:
					value = R.gainset[gain_selection];
					break;
			}

			// Format and print current value
			int16_t val_sub = value;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			if (value < 0)
			{
				val*=-1;
				val_sub*=-1;
				rprintf("-%1u.%01udB",val, val_sub);
			}
			else
			{
				rprintf("%2u.%01udB",val, val_sub);
			}
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				

		if (gain_selection < 4)			// Gain Preset was selected
		{
			R.which_gainset = gain_selection;
			eeprom_write_block(&R.which_gainset, &E.which_gainset, sizeof (R.which_gainset));
			rprintf("Value Stored");
		}
		else rprintf("Nothing Changed");

		_delay_ms(500);

		if (gain_selection == 5)		// Exit selected
		{
			Menu_Mode &= ~CONFIG;		// We're done, EXIT
		}
		else Menu_Mode |= CONFIG;		// We're NOT done, just backing off

		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}

//--------------------------------------------------------------------
// Gain Preset Submenu functions
//--------------------------------------------------------------------
void gain_menu_level2(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	if      (menu_level == GAIN_SET1_MENU) current_selection = R.gainset[1];
	else if (menu_level == GAIN_SET2_MENU) current_selection = R.gainset[2];
	else if (menu_level == GAIN_SET3_MENU) current_selection = R.gainset[3];


	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		int16_t max_value = 700;		// Highest Gain value in dB * 10
		int16_t min_value = -700;		// Lowest Gain value in dB * 10
		if(current_selection > max_value) current_selection = max_value;
		if(current_selection < min_value) current_selection = min_value;

		// Store Current value in running storage
		if      (menu_level == GAIN_SET1_MENU) R.gainset[1] = current_selection;
		else if (menu_level == GAIN_SET2_MENU) R.gainset[2] = current_selection;
		else if (menu_level == GAIN_SET3_MENU) R.gainset[3] = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	

		if (menu_level == GAIN_SET1_MENU)
		{
			rprintf("Gain Preset 1:");
		}
		else if (menu_level == GAIN_SET2_MENU)
		{
			rprintf("Gain Preset 2:");
		}
		else if (menu_level == GAIN_SET3_MENU)
		{
			rprintf("Gain Preset 3:");
		}
		lcd_gotoxy(0,1);
		rprintf("Adjust>  ");
		// Format and print current value
		int16_t val_sub = current_selection;
		int16_t val = val_sub / 10;
		val_sub = val_sub % 10;
		if (current_selection < 0)
		{
			val*=-1;
			val_sub*=-1;
			rprintf("-%1u.%01udB",val, val_sub);
		}
		else
		{
			rprintf(" %2u.%01udB",val, val_sub);
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		if (menu_level == GAIN_SET1_MENU)
		{
			eeprom_write_block(&R.gainset[1], &E.gainset[1], sizeof (R.gainset[1]));
		}
		else if (menu_level == GAIN_SET2_MENU)
		{
			eeprom_write_block(&R.gainset[2], &E.gainset[2], sizeof (R.gainset[2]));
		}
		else if (menu_level == GAIN_SET3_MENU)
		{
			eeprom_write_block(&R.gainset[3], &E.gainset[2], sizeof (R.gainset[3]));
		}

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = GAIN_MENU;			// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}



//--------------------------------------------------------------------
// Gain Menu functions
//--------------------------------------------------------------------
void gain_set_menu(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = gain_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)gain_menu_items, menu_size, current_selection,1, 3, 3);

		// Indicate Current value stored under the currently selected Gain Preset
		// The "stored" value indication changes according to which Preset is currently selected.
		lcd_gotoxy(0,0);				

		rprintf("Select:");

		lcd_gotoxy(13,0);
		if (current_selection < 3)
		{
			int16_t value;

			value = R.gainset[current_selection+1];

			int16_t val_sub = value;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;

			// Print value of currently indicated gain Preset
			if (value < 0)
			{
				val*=-1;
				val_sub*=-1;
				rprintf("-%1u.%01udB",val, val_sub);
			}
			else
			{
				rprintf(" %2u.%01udB",val, val_sub);
			}
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0:
				menu_level = GAIN_SET1_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 1:
				menu_level = GAIN_SET2_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 2:
				menu_level = GAIN_SET3_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 3:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Gain");
				_delay_ms(500);
				Menu_Mode |= CONFIG;	// We're NOT done, just backing off
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Gain");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//--------------------------------------------------------------------
// Calibrate Power Meter based on known input
//--------------------------------------------------------------------
void pow_calibrate_menu(void)
{

	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements
	static uint8_t	Calibrating;		// Boolean for indicating calibrating in process
	static uint16_t	calTimer;			// Calibration timer

	// Is first attenuator force set due to high input power?
	if (Status & HI_INPUT)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Input level too high");
		_delay_ms(500);
		Menu_Mode &=  ~CONFIG;			// We're done
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
	
	else
	{
		if (!Calibrating)
		{
			//-------------------------------------------------------------------
			// Set current attenuation level and update attenuator outputs
			//
			R.Attenuators = ATT_SECOND | ATT_THIRD | 0xfc;	// Set all attenuators except for the first
			ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
			ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
			//ATT_PORT01 = (R.Attenuators & 0x003f) | (~R.Attenuators & 0x00c0);

			// Get Current value
			current_selection = R.calibrate[0].db10m;

			// Selection modified by encoder.  We remember last selection, even if exit and re-entry
			if (Status & ENC_CHANGE)
			{
				if (encOutput > 0)
				{
					current_selection++;
			    	// Reset data from Encoder
					Status &=  ~ENC_CHANGE;
					encOutput = 0;
				}
				else if (encOutput < 0)
				{
					current_selection--;
			  	  	// Reset data from Encoder
					Status &=  ~ENC_CHANGE;
					encOutput = 0;
				}
				// Indicate that an LCD update is needed
				LCD_upd = FALSE;
			}
		
			// If LCD update is needed
			if (LCD_upd == FALSE)
			{
				LCD_upd = TRUE;			// We have serviced LCD
		
				// Keep Encoder Selection Within Bounds of the Menu Size
				if(current_selection > CAL1_MAX_VALUE) current_selection = CAL1_MAX_VALUE;
				if(current_selection < CAL1_MIN_VALUE) current_selection = CAL1_MIN_VALUE;
		
				// Store Current value in running storage
				R.calibrate[0].db10m = current_selection;
		
				lcd_clrscr();
				lcd_gotoxy(0,0);	
				rprintf("Auto Calibrate:");
		
				lcd_gotoxy(0,1);
				rprintf("Rotate to Adjust");
				lcd_gotoxy(0,2);
				rprintf("Push to Save");
				//------------------------------------------
				// Format and print current value in dBm
				lcd_gotoxy(0,3);
				rprintf("->");
		
				int16_t val_sub = current_selection;
				if (val_sub < 0) val_sub *= -1;
				int16_t val = val_sub / 10;
				val_sub = val_sub % 10;
				if (current_selection < 0) rprintf("-");
				else rprintf(" ");
				rprintf("%2d.%01udBm",val, val_sub);

				//------------------------------------------
				// Format and print current value in Vp-p/50ohms
				lcd_gotoxy(11,3);
				double vpp = pow(10,(current_selection-300)/10.0/10.0);// Convert to Watts
				vpp = sqrt(vpp*50) * 2 * sqrt(2);	// Convert to Vp-p
				print_v(vpp);
				rprintf("p-p");
			}
		
			// Enact selection by saving in EEPROM
			if (Status & SHORT_PUSH)
			{
				Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		
				// Save modified value
				eeprom_write_block(&R.calibrate[0].db10m, &E.calibrate[0].db10m, sizeof (R.calibrate[0].db10m));
				R.calibrate[0].ad = ad8317_ad;
				eeprom_write_block(&R.calibrate[0].ad, &E.calibrate[0].ad, sizeof (R.calibrate[0].ad));
		
				Status &=  ~SHORT_PUSH;			// Clear pushbutton status
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Calibrating");
				Calibrating = 1;				// Calibration now in progress
			}
		}
		else
		{
			if(calTimer<20)						// Calibrate for 2 seconds
			{
				calTimer++;						// Increment by 0.1 seconds

				//-------------------------------------------------------------------
				// Set current attenuation level and update attenuator outputs
				//
				R.Attenuators = ATT_FIRST | ATT_SECOND | ATT_THIRD | 0xfc;// Set all attenuators
				ATT_PORT02 = (ATT_PORT02 & ~ATT_P2_PIN) | (~(R.Attenuators>>8) & ATT_P2_PIN);
				ATT_PORT01 = (R.Attenuators & ~(ATT_SECOND|ATT_THIRD)) | (~R.Attenuators & (ATT_SECOND|ATT_THIRD));
			}
			else
			{
				Calibrating = 0;				// Calibration done
				calTimer = 0;					// We are done waiting, set second value
				R.calibrate[1].db10m = R.calibrate[0].db10m - R.first_Attenuator;
				eeprom_write_block(&R.calibrate[1].db10m, &E.calibrate[1].db10m, sizeof(R.calibrate[1].db10m));
				R.calibrate[1].ad = ad8317_ad;
				eeprom_write_block(&R.calibrate[1].ad, &E.calibrate[1].ad, sizeof(R.calibrate[1].ad));
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Calibration Done");
				_delay_ms(500);
				Status &= ~SHORT_PUSH;			// Clear pushbutton status
				Menu_Mode &=  ~CONFIG;			// We're done
				menu_level = 0;					// We are done with this menu level
				LCD_upd = FALSE;				// Make ready for next time
			}
		}
	}
}


//--------------------------------------------------------------------
// AutoSet dBm for First Attenuator Menu functions
//--------------------------------------------------------------------
void first_db_menu(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.autoattenuate;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > AUTO_MAX_VALUE) current_selection = AUTO_MAX_VALUE;
		if(current_selection < AUTO_MIN_VALUE) current_selection = AUTO_MIN_VALUE;

		// Store Current value in running storage
		R.autoattenuate = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	
		rprintf("Atten Auto Engage:");

		lcd_gotoxy(0,1);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcd_gotoxy(0,3);
		rprintf("->");
		if (current_selection == AUTO_MAX_VALUE)
		{
			rprintf("Permanently -OFF-");
		}
		else if (current_selection == AUTO_MIN_VALUE)
		{
			rprintf("Permanently -ON-");
		}
		else
		{
			int16_t val_sub = current_selection;
			if (val_sub < 0) val_sub *= -1;
			int16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			if (current_selection < 0) rprintf("  -");
			else rprintf("   ");
			rprintf("%2d.%01udBm",val, val_sub);
		}
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.autoattenuate, &E.autoattenuate, sizeof (R.autoattenuate));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}


//--------------------------------------------------------------------
// Attenuator Submenu functions
//--------------------------------------------------------------------
void attenuator_menu_level2(void)
{
	static int16_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	if 		(menu_level == ATT_SET1_MENU)current_selection = R.first_Attenuator;
	else if (menu_level == ATT_SET2_MENU)current_selection = R.second_Attenuator;
	else if (menu_level == ATT_SET3_MENU)current_selection = R.third_Attenuator;
	else if (menu_level == ATT_SET4_MENU)current_selection = R.fixed_Attenuation;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		int16_t max_value = 700;		// Highest Attenuation value in dB * 10
		int16_t min_value = 0;			// Lowest Attenuation value in dB * 10
		if(current_selection > max_value) current_selection = max_value;
		if(current_selection < min_value) current_selection = min_value;

		// Store Current value in running storage
		if		(menu_level == ATT_SET1_MENU)R.first_Attenuator = current_selection;
		else if (menu_level == ATT_SET2_MENU)R.second_Attenuator = current_selection;
		else if (menu_level == ATT_SET3_MENU)R.third_Attenuator = current_selection;
		else if (menu_level == ATT_SET4_MENU)R.fixed_Attenuation = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	

		if (menu_level == ATT_SET1_MENU)
		{
			rprintf("Attenuator 1 Adjust:");
		}
		else if (menu_level == ATT_SET2_MENU)
		{
			rprintf("Attenuator 2 Adjust:");
		}
		else if (menu_level == ATT_SET3_MENU)
		{
			rprintf("Attenuator 3 Adjust:");
		}
		else if (menu_level == ATT_SET4_MENU)
		{
			rprintf("Insertion Loss Adj:");
		}
		lcd_gotoxy(0,2);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,3);
		rprintf("Push to Save->");
		// Format and print current value
		uint16_t val_sub = current_selection;
		uint16_t val = val_sub / 10;
		val_sub = val_sub % 10;
		rprintf("%2u.%01u",val, val_sub);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		if (menu_level == ATT_SET1_MENU)
		{
			eeprom_write_block(&R.first_Attenuator, &E.first_Attenuator, sizeof (R.first_Attenuator));
		}
		else if (menu_level == ATT_SET2_MENU)
		{
			eeprom_write_block(&R.second_Attenuator, &E.second_Attenuator, sizeof (R.second_Attenuator));
		}
		else if (menu_level == ATT_SET3_MENU)
		{
			eeprom_write_block(&R.third_Attenuator, &E.third_Attenuator, sizeof (R.third_Attenuator));
		}
		else if (menu_level == ATT_SET4_MENU)
		{
			eeprom_write_block(&R.fixed_Attenuation, &E.fixed_Attenuation, sizeof (R.fixed_Attenuation));
		}

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = ATT_MENU;			// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}



//--------------------------------------------------------------------
// Attenuator Menu functions
//--------------------------------------------------------------------
void attenuators_set_menu(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = attenuator_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)attenuator_menu_items, menu_size, current_selection,1, 5, 3);

		// Indicate Current Frequency and the Frequency stored under the currently selected Channel number
		// The "stored" Frequency always changes according to which channel is currently selected.
		lcd_gotoxy(0,0);				
		rprintf("Attenuators Adjust:");
		//lcd_gotoxy(0,2);
		//rprintf("Cur Val");
		lcd_gotoxy(0,3);
		if (current_selection < 5)
		{
			uint16_t value=0;

			switch (current_selection)
			{
				case 0:
					value = R.first_Attenuator;
					break;
				case 1:
					value = R.second_Attenuator;
					break;
				case 2:
					value = R.third_Attenuator;
					break;
				case 3:
					value = R.fixed_Attenuation;
					break;
			}
			uint16_t val_sub = value;
			uint16_t val = val_sub / 10;
			val_sub = val_sub % 10;
			rprintf("%2u.%01udB",val, val_sub);
		}
		else
		{
			rprintf(" --");
		}
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0:
				menu_level = ATT_SET1_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 1:
				menu_level = ATT_SET2_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 2:
				menu_level = ATT_SET3_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 3:
				menu_level = ATT_SET4_MENU;
				LCD_upd = FALSE;	// force LCD reprint
				break;
			case 4:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Attenuators");
				_delay_ms(500);
				Menu_Mode |= CONFIG;	// We're NOT done, just backing off
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Done w. Attenuators");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//--------------------------------------------------------------------
// ZX76-31R5 SWR Error Compensation Select Menu functions
//--------------------------------------------------------------------
void swr_comp_menu(void)
{
	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			swr_comp++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			swr_comp--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = swr_comp_size;
		while(swr_comp >= menu_size)
			swr_comp -= menu_size;
		while(swr_comp < 0)
			swr_comp += menu_size;

		lcd_clrscr();
		
		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)swr_comp_items, menu_size, swr_comp,1, 1, 3);

		// Indicate Currently selected SWR Compensation
		lcd_gotoxy(0,0);
		rprintf("ZX76 SWR Error Comp:");
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				

		if (swr_comp < 4)				// SWR Compensation was selected
		{
			R.HF_compensation = swr_comp;
			eeprom_write_block(&R.HF_compensation, &E.HF_compensation, sizeof (R.HF_compensation));
			rprintf("Value Stored");
		}
		else rprintf("Nothing Changed");

		_delay_ms(500);

		if (swr_comp == 5)				// Exit selected
		{
			Menu_Mode &= ~CONFIG;		// We're done, EXIT
		}
		else Menu_Mode |= CONFIG;		// We're NOT done, just backing off

		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}

//--------------------------------------------------------------------
// Rotary Encoder Resolution
//--------------------------------------------------------------------
void encoder_menu(void)
{

	uint8_t	current_selection;			// Keep track of current Encoder Resolution

	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.encoderRes;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection=current_selection<<1;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection=current_selection>>1;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > 128) current_selection = 128;
		if(current_selection < 1) current_selection = 1;

		// Store Current value in running storage
		R.encoderRes = current_selection;

		lcd_clrscr();
		lcd_gotoxy(0,0);	
		rprintf("Encoder ResDivide:");

		lcd_gotoxy(0,1);
		rprintf("Rotate to Adjust");
		lcd_gotoxy(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcd_gotoxy(0,3);
		rprintf("->");

		int16_t val = current_selection;
		rprintf("%3u",val);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.encoderRes, &E.encoderRes, sizeof (R.encoderRes));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}


//--------------------------------------------------------------------
// Debug Screen, exit on push
//--------------------------------------------------------------------
void debug_menu(void)
{

	set_Attenuators();			// Set the attenuators if Encoder input or if Power protect

	lcd_display_mixed();		// Display diverse debug stuff

	// Exit on Button Push
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;	// Clear pushbutton status

		lcd_clrscr();
		lcd_gotoxy(1,1);				
		rprintf("Nothing Changed");
		_delay_ms(500);
		Menu_Mode |= CONFIG;	// We're NOT done, just backing off
		menu_level = 0;			// We are done with this menu level
	}
}


//--------------------------------------------------------------------
// Factory Reset with all default values
//--------------------------------------------------------------------
void factory_menu(void)
{
	static int8_t	current_selection;
	static uint8_t	LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = factory_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("Set all to default:");

		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)factory_menu_items, menu_size, current_selection,
				1, /*column*/ 3, 6);
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Factory Reset
				// Force an EEPROM update:
				eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
				lcd_clrscr();
				lcd_gotoxy(0,0);				
				rprintf("Factory Reset was");
				lcd_gotoxy(0,1);
				rprintf("selected.");
				lcd_gotoxy(0,2);
				rprintf("All settings reset");
				lcd_gotoxy(0,3);
				rprintf("to default values.");
				while (1);			// Bye bye, Death by Watchdog
			case 1:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode |= CONFIG;// We're NOT done, just backing off
				menu_level = 0;		// We are done with this menu level
				LCD_upd = FALSE;	// Make ready for next time
				break;
			default:
				lcd_clrscr();
				lcd_gotoxy(1,1);				
				rprintf("Nothing Changed");
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//
//--------------------------------------------------------------------
// Manage the first level of Menus
//--------------------------------------------------------------------
//
void menu_level0(void)
{
	static int8_t	current_selection;	// Keep track of current LCD menu selection
	static uint8_t	LCD_upd = FALSE;	// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	if (LCD_upd == FALSE)				// Need to update LCD
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = level0_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcd_clrscr();
		lcd_gotoxy(0,0);
		rprintf("Configuration Menu:");

		// Print the Menu
		lcd_scroll_Menu((char**)level0_menu_items, menu_size, current_selection,1, 0, 3);
	}

	if (Status & SHORT_PUSH)
	{
		Status &= ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Select gain Preset (attenuator is negative)
				menu_level = GAIN_SEL_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				// Jump into the sub-menu indicating the currently selected gain Preset
				gain_selection = R.which_gainset; 
				break;

			case 1: // Adjust Gain compensation
				menu_level = GAIN_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 2: // Perform Power Calibration
				menu_level = POW_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 3: // AutoSet dBm Switchover threshold for First Attenuator
				menu_level = AUTO_FIRST_ATT;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 4: // Adjust Attenuators
				menu_level = ATT_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 5:	// ZX76 SWR Compensation
				menu_level = SWR_COMP_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				// Jump into the sub-menu indicating the currently selected SWR Compensation
				swr_comp = R.HF_compensation; 
				break;
			case 6:// Encoder Resolution
				menu_level = ENCODER_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 7: // Display Debug Stuff
				menu_level = DEBUG_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;

			case 8: // Factory Reset
				menu_level = FACTORY_MENU;
				LCD_upd = FALSE;		// force LCD reprint
				break;
			default:
				// Exit
				lcd_clrscr();
				lcd_gotoxy(1,1);
		   		rprintf("Return from Menu");
				Menu_Mode &=  ~CONFIG;	// We're done
				LCD_upd = FALSE;		// Make ready for next time
		}
	}
}


//
//--------------------------------------------------------------------
// Scan the Configuraton Menu Status and delegate tasks accordingly
//--------------------------------------------------------------------
//
void PushButtonMenu(void)
{
	// Select which menu level to manage
	if (menu_level == 0) menu_level0();

	else if (menu_level == GAIN_SEL_MENU) gain_select_menu();

	else if (menu_level == GAIN_MENU) gain_set_menu();
	else if (menu_level == GAIN_SET1_MENU) gain_menu_level2();
	else if (menu_level == GAIN_SET2_MENU) gain_menu_level2();
	else if (menu_level == GAIN_SET3_MENU) gain_menu_level2();

	else if (menu_level == POW_MENU) pow_calibrate_menu();

	else if (menu_level == AUTO_FIRST_ATT) first_db_menu();

	else if (menu_level == ATT_MENU) attenuators_set_menu();
	else if (menu_level == ATT_SET1_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET2_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET3_MENU) attenuator_menu_level2();
	else if (menu_level == ATT_SET4_MENU) attenuator_menu_level2();

	else if (menu_level == SWR_COMP_MENU) swr_comp_menu();

	else if (menu_level == ENCODER_MENU) encoder_menu();

	else if (menu_level == DEBUG_MENU) debug_menu();

	else if (menu_level == FACTORY_MENU) factory_menu();
}
