//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//---------------------------------------------------------------------------------
// 20x4 LCD Print Routines
//
// lcd_display() should be accessed once every 50ms. These LCD routines only service
// one line at a time, hence it takes 4 calls to update the whole display.  This is
// necessary for not to interfere with the stepper motor timing. Also, nothing is
// written to the LCD, unless the relevant information has changed.
//---------------------------------------------------------------------------------
//

struct  {                            // Keep track of stuff that needs to be reprinted to the LCD
          unsigned line1_init :   1; // "Radio:" and "Hz" in line 1
          unsigned line1_frq  :   1; // Current Radio frequency in line 1
          unsigned line2_init :   1; // "Tuned:" and "Hz" in line 2
          unsigned line2_frq  :   1; // Currently Tuned frequency in line 2
          unsigned line3_init :   1; // "Motor:" in line 3
          unsigned line3_curr :   1; // Current position of Mmotor in line 3
          unsigned line3_dest :   1; // Motor Destination in line 3
          unsigned line4_range:   1; // "Range:" in line 4
          unsigned line4_rnum:    1; // Range printout in line 4
          unsigned line4_power:   1; // "Power:" in line 4
          unsigned line4_pstatus: 1; // Power Status printout (Is Motor powered up or not)
          unsigned line4_err1:    1; // Error Condition 1, Cap out of range - or Endstop 1
          unsigned line4_err2:    1; // Error Condition 2, Frq out of range - or Endstop 2

        } lcd_upd;
//
//-----------------------------------------------------------------------------------------
//      Display Frequency as XX.XXX.XXX
//-----------------------------------------------------------------------------------------
//
void display_frq(int32_t frq)
{
  int32_t f1, f2, f3;					// Used to format frequency printout	
  f1 = frq/1000000;
  f3 = frq%1000000;
  f2 = f3/1000;
  f3 = f3%1000;
  sprintf(lcd_buf,"%2u.%03u.%03u",(uint16_t)f1,(uint16_t)f2,(uint16_t)f3);
  lcd.print(lcd_buf);
}

//
//-----------------------------------------------------------------------------
//      Display Stepper Position
//      Indicate number of revolutions above lowest preset
//-----------------------------------------------------------------------------
//
void display_stepper_pos(int32_t pos)
{
  int8_t sign;
  float pos_rev;

  if (preset[1].Pos>=preset[0].Pos) sign = 1;	// Normalise +/-
  else sign = -1;
	
  pos_rev = sign*(pos - preset[0].Pos)/1.6;
  #if PROTOTYPE  // Cludge to make 8 bit AVR compiler happy when using Teensy 2++
  sprintf(lcd_buf,"%5d ", (int32_t) pos_rev);
  #else          // Normal method
  sprintf(lcd_buf,"%6.3f",pos_rev/1000);
  #endif
  lcd.print(lcd_buf);
}

//
//-----------------------------------------------------------------------------
//      Manage Display
//-----------------------------------------------------------------------------
//
void lcd_display(void)
{
  static uint8_t lcd_line = 0;
  static int32_t line1_frq;                   // Last frequency printed to LCD
  static int32_t line2_frq;                   // Last frequency printed to LCD
  static int32_t curr_pos;                    // Current motor position printed to LCD
  static int32_t dest_pos;                    // Last destination position printed to LCD
  static uint8_t line4_rnum;                  // Last Range number printed to LCD  
  static uint8_t line4_motor;                 // Bool for last Motor condition, On or Off  
  static uint8_t line4_err1;                  // Bool for last Error Condition 1, Cap out of range - or Endstop 1
  static uint8_t line4_err2;                  // Bool for last Error Condition 2, Frq out of range - or Endstop 2 

  //
  // Menu functions have precedence
  //
  if ((Menu_Mode & CONFIG) || (Menu_disp_timer!=0))
  {
    // Clear all LCD print flags to indicate that everything has to be reprinted
    // when we return from LCD
    lcd_upd.line1_init   = FALSE;
    lcd_upd.line1_frq    = FALSE;
    lcd_upd.line2_init   = FALSE;
    lcd_upd.line2_frq    = FALSE;
    lcd_upd.line3_init   = FALSE;
    lcd_upd.line3_curr   = FALSE;
    lcd_upd.line3_dest   = FALSE;
    lcd_upd.line4_range  = FALSE;
    lcd_upd.line4_rnum   = FALSE;
    lcd_upd.line4_power  = FALSE;
    lcd_upd.line4_pstatus= FALSE;
    lcd_upd.line4_err1   = FALSE;
    lcd_upd.line4_err2   = FALSE;
  }

  //
  // Print stuff to the LCD
  //
  else                                                  
  {
    if (lcd_line == 0)                                  // First line of LCD
    {
      //-----------------------------------
      // Display Input Frequency from Radio
      //-----------------------------------
      // Is input from Radio inactive?
      if ((Status & FRQ_RADIO) == 0)                    // Activity flag indicates inactive
      {
        // Indicate Offline if nothing received for 15 seconds
        if (frq_check_timer == 150)
        {
          lcd.setCursor(0,0);
          lcd.print("Radio: Not Connected");
          lcd_upd.line1_init = FALSE;                   // Ensure update of Line 1 when Radio comes on-line 
          lcd_upd.line1_frq  = FALSE;
        }
        // Else measure outage time
        else frq_check_timer++;
      }
      // Updates from Radio are active, print to LCD
      //
      else
      {
        frq_check_timer = 0;                            // Touch the freqquency check timer
        
        if (lcd_upd.line1_init == FALSE)                // Do we need to print "Radio:          Hz" ?
        {
          lcd.setCursor(0,0);
          lcd.print("Radio: ");
          lcd.setCursor(17,0);
          lcd.print(" Hz");
          lcd_upd.line1_init = TRUE;
        }       

        // Do we need to reprint running frequency information?
        if ((line1_frq != running.Frq)||(lcd_upd.line1_frq == FALSE))
        {
          lcd.setCursor(7,0);
          display_frq(running.Frq);
          line1_frq = running.Frq;
          lcd_upd.line1_frq = TRUE;
        }
      }
    }

    if (lcd_line == 1)                                  // Second line of LCD
    {  
      //-----------------------------------
      // Indicate Tuned Frequency
      //-----------------------------------
      if (lcd_upd.line2_init == FALSE)                // Do we need to print "Tuned:          Hz" ?
      {
        lcd.setCursor(0,1);
        lcd.print("Tuned: ");
        lcd.setCursor(17,1);
        lcd.print(" Hz");
        lcd_upd.line2_init = TRUE;
      }       

      // Do we need to reprint running frequency information?
      if ((line2_frq != tunedFrq)||(lcd_upd.line2_frq == FALSE))
      {
        lcd.setCursor(7,1);
        display_frq(tunedFrq);
        line2_frq = tunedFrq;
        lcd_upd.line2_frq = TRUE;
      }
    }

    if (lcd_line == 2)                                  // Third line of LCD
    {  
      // Indicate capacitor position (number of revolutions above lowest preset)
      //
      if (lcd_upd.line3_init == FALSE)                // Do we need to print "Motor:" ?
      {
        lcd.setCursor(0,2);
        lcd.print("Motor: ");
        lcd_upd.line3_init = TRUE;
      }
      // Do we need to update current motor position ?
      if ((curr_pos != stepper_track)||(lcd_upd.line3_curr == FALSE))
      {    
        lcd.setCursor(7,2);
        //display_stepper_pos(stepper_track);			
        display_stepper_pos(running.Pos);			
        curr_pos = stepper_track;
        lcd_upd.line3_curr = TRUE;
      }
      // Do we need to update destination motor position ?
      if ((dest_pos != (running.Pos + delta_Pos + encOutput))||(lcd_upd.line3_dest == FALSE))
      {    
        lcd.setCursor(14,2);
        //display_stepper_pos(running.Pos + delta_Pos + encOutput);			
        display_stepper_pos(stepper_track);			
        dest_pos = running.Pos + delta_Pos + encOutput;
        lcd_upd.line3_dest = TRUE;
      }
    }
    
    if (lcd_line == 3)                                  // Fourth line of LCD
    {  
      // Indicate whether Stepper is Active or not
      //
      if (lcd_upd.line4_power == FALSE)                // Do we need to print "Motor:" ?
      {
        lcd.setCursor(10,3);
        lcd.print("Power: ");
        lcd_upd.line4_power = TRUE;
        lcd_upd.line4_err1  = FALSE;
        lcd_upd.line4_err2  = FALSE;        
      }

      // Do we need to print On or Off for Motor Power status ?
      if ((line4_motor != (Status & FRQ_STORE))||(lcd_upd.line4_pstatus == FALSE))
      {
        lcd.setCursor(17,3);
        if (Status & FRQ_STORE)
          lcd.print(" On");
        else 
          lcd.print("Off");
        line4_motor = Status & FRQ_STORE;
        lcd_upd.line4_pstatus = TRUE;
        lcd_upd.line4_err1    = FALSE;
        lcd_upd.line4_err2    = FALSE;        
      }

      // Indicate which range we're using
      //
      if (lcd_upd.line4_range == FALSE)                // Do we need to print "Range: " ?
      {
        lcd.setCursor(0,3);
        lcd.print("Range: ");
        lcd_upd.line4_range = TRUE;
        lcd_upd.line4_err1  = FALSE;
        lcd_upd.line4_err2  = FALSE;        
      }       
      // Do we need to reprint current range information?
      if ((line4_rnum != range)||(lcd_upd.line4_rnum == FALSE))
      {
        lcd.setCursor(7,3);
        sprintf(lcd_buf,"%2u ", range);
        lcd.print(lcd_buf);
        line4_rnum = range;
        lcd_upd.line4_rnum = TRUE;
        lcd_upd.line4_err1 = FALSE;
        lcd_upd.line4_err2 = FALSE;        
      }
    }
    

      //-----------------------------------
      // Error Conditions
      //-----------------------------------
      lcd.setCursor(0,3);
      #if ENDSTOP_OPT == 1  // Vacuum Capacitor, no end stops

      // Display Frequency out of range condition
      //
      // Do we need to print update the "FRQ out of range" status ?
      if ((line4_err1 != (Status & FRQ_XRANGE))||(lcd_upd.line4_err1 == FALSE))
      {
        if (Status & FRQ_XRANGE)
        {
          lcd.print("FRQ out of range");
        }    
        line4_err1 = Status & FRQ_XRANGE;
        lcd_upd.line4_err1   = TRUE;
        lcd_upd.line4_err2   = FALSE;
        lcd_upd.line4_range  = FALSE;
        lcd_upd.line4_rnum   = FALSE;
        lcd_upd.line4_power  = FALSE;
        lcd_upd.line4_pstatus= FALSE;
      }
      
      // Display Capacitor out of range condition
      //
      // Do we need to print update the "CAP out of range" status ?
      if ((line4_err2 != (Status & CAP_XRANGE))||(lcd_upd.line4_err2 == FALSE))
      {
        if (Status & CAP_XRANGE)
        {
          lcd.print("CAP out of range");
        }    
        line4_err2 = Status & FRQ_XRANGE;
        lcd_upd.line4_err2   = TRUE;
        lcd_upd.line4_err1   = FALSE;
        lcd_upd.line4_range  = FALSE;
        lcd_upd.line4_rnum   = FALSE;
        lcd_upd.line4_power  = FALSE;
        lcd_upd.line4_pstatus= FALSE;        
      }  
      #endif
      #if ENDSTOP_OPT == 2  // End stop sensors implemented
      // Display Lower EndStop condition
      //
      // Do we need to print update the "Lower EndStop" status ?
      if ((line4_err1 != (Status & END_LOWER))||(lcd_upd.line4_err1 == FALSE))
      {
        if (Status & END_LOWER)
        {
          lcd.print("Lower EndStop   ");
        }    
        line4_err1 = Status & END_LOWER;
        lcd_upd.line4_err1   = TRUE;
        lcd_upd.line4_err2   = FALSE;
        lcd_upd.line4_range  = FALSE;
        lcd_upd.line4_rnum   = FALSE;
        lcd_upd.line4_power  = FALSE;
        lcd_upd.line4_pstatus= FALSE;  
      }
   
      // Display Upper Endstop condition
      //
      // Do we need to print update the "Upper Endstop" status ?
      if ((line4_err2 != (Status & END_UPPER))||(lcd_upd.line4_err2 == FALSE))
      {
        if (Status & END_UPPER)
        {
          lcd.print("Upper EndStop   ");
        }    
        line4_err2 = Status & END_UPPER;
        lcd_upd.line4_err2 = TRUE;
        lcd_upd.line4_err1   = FALSE;
        lcd_upd.line4_range  = FALSE;
        lcd_upd.line4_rnum   = FALSE;
        lcd_upd.line4_power  = FALSE;
        lcd_upd.line4_pstatus= FALSE;  
      }
      #endif
      #if ENDSTOP_OPT == 3   // Butterfly capacitor, no end stops required
      if (0);                // Nothing here
      #endif
    
    // Rotate LCD Line Print
    lcd_line++;
    if (lcd_line > 3) lcd_line = 0;
  }
}


