//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//---------------------------------------------------------------------------------
// Mulitipurpose Pushbutton and Rotary Encoder Menu Management Routines
//---------------------------------------------------------------------------------
//

// First Level Menu Items
// Note - number of Frequencies needs to be same as NUM_PRESETS as defined in ML.h
const uint8_t level0_menu_size = 5;
const char *level0_menu_items[] =
                        {  "1-New Position",
                           "2-Rewrite Position",
                           "3-Delete Position",
                           "4-Clear All",
                           "5-Exit" };

// New Pos Flag
#define NEW_POS_MENU      1

// Manage Pos Flag
#define MANAGE_POS_MENU   2

// Delete Pos Flag
#define DELETE_POS_MENU   3

// Flag for Factory Reset
#define FACTORY_MENU	10
// Factory Reset menu Items
const uint8_t factory_menu_size = 2;
const char *factory_menu_items[] =
				{  "1 No  - Exit",
				   "2 Yes - Reset"};

uint16_t   menu_level = 0;  // Keep track of which menu we are in
uint8_t    menu_data = 0;   // Pass data to lower menu


//----------------------------------------------------------------------
// Display a Menu of choices, one line at a time
//
// **menu refers to a pointer array containing the Menu to be printed
//
// menu_size indicates how many pointers (menu items) there are in the array
//
// current_choice indicates which item is currently up for selection if
// pushbutton is pushed
//
// begin row & begin_col are the coordinates for the upper lefthand corner
// of the three or four lines to be printed
//
//----------------------------------------------------------------------
void lcd_scroll_Menu(char **menu, uint8_t menu_size,
  uint8_t current_choice, uint8_t begin_row, uint8_t begin_col, uint8_t lines)
{
  uint8_t a, x;

  // Clear LCD from begin_col to end of line.
  lcd.setCursor(begin_col, begin_row);
  for (a = begin_col; a < 20; a++)
    lcd.print(" ");
  if (lines > 1)
  {
    lcd.setCursor(begin_col, begin_row+1);
    for (a = begin_col; a < 20; a++)
      lcd.print(" ");
  }
  if (lines > 2)
  {
    lcd.setCursor(begin_col, begin_row+2);
    for (a = begin_col; a < 20; a++)
      lcd.print(" ");
  }
  // Using Menu list pointed to by **menu, preformat for print:
  // First line contains previous choice, second line contains
  // current choice preceded with a '->', and third line contains
  // next choice
  if (current_choice == 0) x = menu_size - 1;
  else x = current_choice - 1;
  if (lines > 1)
  {
    lcd.setCursor(begin_col + 2, begin_row);
    sprintf(lcd_buf,"%s", *(menu + x));
    lcd.print(lcd_buf);

    lcd.setCursor(begin_col, begin_row + 1);
    sprintf(lcd_buf,"->%s", *(menu + current_choice));
    lcd.print(lcd_buf);
    if (current_choice == menu_size - 1) x = 0;
    else x = current_choice + 1;

    if (lines > 2)
    {
      lcd.setCursor(begin_col + 2, begin_row + 2);
      sprintf(lcd_buf,"%s", *(menu + x));
      lcd.print(lcd_buf);
    }  
  }
  else
  {
    lcd.setCursor(begin_col, begin_row);
    sprintf(lcd_buf,"->%s", *(menu + current_choice));
    lcd.print(lcd_buf);
  }
  // LCD print lines 1 to 3

  // 4 line display.  Preformat and print the fourth line as well
  if (lines == 4)
  {
    if (current_choice == menu_size-1) x = 1;
    else if (current_choice == menu_size - 2 ) x = 0;
    else x = current_choice + 2;
    lcd.setCursor(begin_col, begin_row+3);
    for (a = begin_col; a < 20; a++)
      lcd.print(" ");
    lcd.setCursor(begin_col + 2, begin_row + 3);
    sprintf(lcd_buf,"  %s", *(menu + x));
    lcd.print(lcd_buf);
  }
}


//----------------------------------------------------------------------
// Menu functions begin:
//----------------------------------------------------------------------

//
//--------------------------------------------------------------------
// New Pos Menu
//--------------------------------------------------------------------
//
void new_pos_menu(void)
{
  static int8_t	current_selection;	// Keep track of current menu selection
  static uint8_t	LCD_upd = FALSE;// Keep track of LCD update requirements
  
  if (num_presets == MAX_PRESETS)       // Check if no more preset memories available
  {
    lcd.setCursor(0,1);
    lcd.print("Memory Full!!!");
    Menu_disp_timer = 50;    // Show on LCD for 5 seconds
    menu_level = 0;          // We're done with this menu level
    Menu_Mode &=  ~CONFIG;   // We're done
    LCD_upd = FALSE;         // Make ready for next time
  }
  else
  {
    if ((Status & FRQ_RADIO) == 0)  // No Frequency information being receiced from Radio
    {
      lcd.clear();
      lcd.setCursor(0,1);
      lcd.print("No FRQ information");
      lcd.setCursor(0,2);
      lcd.print("received from Radio");
      Menu_disp_timer = 30;    // Show on LCD for 5 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time      
    }
    else
    {
      // Store Current value above the highest existing preset
      preset[num_presets].Frq = running.Frq;
      // Set following positions
      running.Pos = running.Pos + delta_Pos;
      stepper_track = running.Pos;
      delta_Pos = 0;
      preset[num_presets].Pos = running.Pos;
      num_presets++;

      // Sort all presets in an ascending order,
      // but with empty poitions on top
      preset_sort();
      normalize_stepper_pos();
      // Store the whole block of frequencies and positions in eeprom      
      EEPROM_writeAnything(17,preset);    
    
      lcd.clear();
      lcd.setCursor(0,1);
      lcd.print("New Preset Stored!!!");
      Menu_disp_timer = 30;    // Show on LCD for 5 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time
    }
  }
}

//--------------------------------------------------------------------
// Overwrite existing Position with new
//--------------------------------------------------------------------
void manage_pos_menu(void)
{
  static int8_t	current_selection;	// Keep track of current menu selection
  static uint8_t	LCD_upd = FALSE;// Keep track of LCD update requirements
  // Selection modified by encoder.  We remember last selection, even if exit and re-entry
  
  if (num_presets == 0) 
  {
    Status|=SHORT_PUSH;    // Nothing has been stored - nothing to do.
    LCD_upd = TRUE;        // No LCD servicing required
  }
  
  if (Enc.read()/ENC_MENURESDIVIDE != 0)
  {
    if (Enc.read()/ENC_MENURESDIVIDE > 0)
    {
      current_selection++;
    }
    else if (Enc.read()/ENC_MENURESDIVIDE < 0)
    {
      current_selection--;
    }
    // Reset data from Encoder
    Enc.write(0);

    // Indicate that an LCD update is needed
    LCD_upd = FALSE;
  }

  if (LCD_upd == FALSE)                // Need to update LCD
  {
    LCD_upd = TRUE;                    // We have serviced LCD
  

   // Keep Encoder Selection Within Bounds of the Menu Size
    uint8_t menu_size = num_presets+1;
    while(current_selection >= menu_size)
    current_selection -= menu_size;
    while(current_selection < 0)
      current_selection += menu_size;

    lcd.clear();
    lcd.setCursor(0,0);
    
    // Print the Menu
    if (current_selection < num_presets)
    {
      lcd.print("Overwrite FRQ Preset");
      lcd.setCursor(0,1);
      sprintf(lcd_buf,"->%2u ",current_selection);
      lcd.print(lcd_buf);
      display_frq(preset[current_selection].Frq);
      lcd.print(" Hz");
      
      lcd.setCursor(0,2);
      lcd.print("with ");
      display_frq(running.Frq);
      lcd.print(" Hz");
      lcd.setCursor(0,3);
      lcd.print("Rotate and select:");
    }    
    else
    {
      lcd.print("No change");
      lcd.setCursor(0,1);
      lcd.print("-> Exit");  
      lcd.setCursor(0,3);
      lcd.print("Rotate and select:");
    }
  }

  if (Status & SHORT_PUSH)
  {
    Status &= ~SHORT_PUSH;     // Clear pushbutton status

    if (num_presets == 0)      // Nothing has been stored, lets get out of here
    {
      lcd.clear();
      lcd.setCursor(0,1);
      lcd.print("Memory is Empty!!");
      Menu_disp_timer = 30;    // Show on LCD for 3 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time     
    }
    else if (current_selection < num_presets)
    {
      // Store Current value in the selected preset
      preset[current_selection].Frq = running.Frq;
      // Set following positions
      running.Pos = running.Pos + delta_Pos;
      stepper_track = running.Pos;
      delta_Pos = 0;
      preset[current_selection].Pos = running.Pos;

      // Sort all presets in an ascending order,
      // but with empty poitions on top
      preset_sort();
      normalize_stepper_pos();
      // Store the whole block of frequencies and positions in eeprom      
      EEPROM_writeAnything(17,preset);

      lcd.clear();
      lcd.setCursor(0,3);
      lcd.print("New value stored");
      Menu_disp_timer = 30;    // Show on LCD for 3 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time
    }
    else
    {
      lcd.clear();
      lcd.setCursor(0,3);
      lcd.print("Return from Menu");
      Menu_disp_timer = 20;    // Show on LCD for 2 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time
    }
  }
}

//--------------------------------------------------------------------
// Delete Position
//--------------------------------------------------------------------
void delete_pos_menu(void)
{
  static int8_t	current_selection;	// Keep track of current menu selection
  static uint8_t	LCD_upd = FALSE;	// Keep track of LCD update requirements
  // Selection modified by encoder.  We remember last selection, even if exit and re-entry
  
  if (num_presets == 0)
  {
    Status|=SHORT_PUSH;    // Nothing has been stored - nothing to do.
    LCD_upd = TRUE;        // No LCD servicing required
  }
 
  if (Enc.read()/ENC_MENURESDIVIDE != 0)
  {
    if (Enc.read()/ENC_MENURESDIVIDE > 0)
    {
      current_selection++;
    }
    else if (Enc.read()/ENC_MENURESDIVIDE < 0)
    {
      current_selection--;
    }
    // Reset data from Encoder
    Enc.write(0);

    // Indicate that an LCD update is needed
    LCD_upd = FALSE;
  }

  if (LCD_upd == FALSE)                // Need to update LCD
  {
    LCD_upd = TRUE;                    // We have serviced LCD

    // Keep Encoder Selection Within Bounds of the Menu Size
    uint8_t menu_size = num_presets+1;
    while(current_selection >= menu_size)
    current_selection -= menu_size;
    while(current_selection < 0)
      current_selection += menu_size;

    lcd.clear();
    lcd.setCursor(0,0);
    
    // Print the Menu
    if (current_selection < num_presets)
    {
      lcd.print("Delete FRQ Preset");
      lcd.setCursor(0,1);
      sprintf(lcd_buf,"->%2u ",current_selection);
      lcd.print(lcd_buf);
      display_frq(preset[current_selection].Frq);
      lcd.print(" Hz");
      
      lcd.setCursor(0,3);
      lcd.print("Rotate and select:");
    }    
    else
    {
      lcd.print("No change");
      lcd.setCursor(0,1);
      lcd.print("-> Exit");  
      lcd.setCursor(0,3);
      lcd.print("Rotate and select:");
    }
  }

  if (Status & SHORT_PUSH)
  {
    Status &= ~SHORT_PUSH;      // Clear pushbutton status

    if (num_presets == 0)       // Nothing has been stored, lets get out of here
    {
      lcd.clear();
      lcd.setCursor(0,1);
      lcd.print("Memory is Empty!!");
      Menu_disp_timer = 30;    // Show on LCD for 3 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time     
    }
    else if (current_selection < num_presets)
    {
      // Store a Zero value in the selected preset
      preset[current_selection].Frq = 0;
      preset[current_selection].Pos = 1000000;

      // Sort all presets in an ascending order,
      // but with empty poitions on top
      preset_sort();
      normalize_stepper_pos();
      // Store the whole block of frequencies and positions in eeprom      
      EEPROM_writeAnything(17,preset);

      lcd.clear();
      lcd.setCursor(0,3);
      lcd.print("MemoryPreset Deleted");
      Menu_disp_timer = 30;    // Show on LCD for 3 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time
    }
    else
    {
      lcd.clear();
      lcd.setCursor(0,3);
      lcd.print("Return from Menu");
      Menu_disp_timer = 20;    // Show on LCD for 2 seconds
      menu_level = 0;          // We're done with this menu level
      Menu_Mode &=  ~CONFIG;   // We're done
      LCD_upd = FALSE;         // Make ready for next time
    }
  }
}

//--------------------------------------------------------------------
// Factory Reset with all default values
//--------------------------------------------------------------------
void factory_menu(void)
{
  static int8_t	current_selection;
  static uint8_t	LCD_upd = FALSE;  // Keep track of LCD update requirements

  // Selection modified by encoder.  We remember last selection, even if exit and re-entry
  if (Enc.read()/ENC_MENURESDIVIDE != 0)
  {
    if (Enc.read()/ENC_MENURESDIVIDE > 0)
    {
      current_selection++;
    }
    else if (Enc.read()/ENC_MENURESDIVIDE < 0)
    {
      current_selection--;
    }
    // Reset data from Encoder
    Enc.write(0);

    // Indicate that an LCD update is needed
    LCD_upd = FALSE;
  }

  // If LCD update is needed
  if (LCD_upd == FALSE)
  {
    LCD_upd = TRUE;                      // We have serviced LCD

    // Keep Encoder Selection Within Bounds of the Menu Size
    uint8_t menu_size = factory_menu_size;
    while(current_selection >= menu_size)
      current_selection -= menu_size;
    while(current_selection < 0)
      current_selection += menu_size;

    lcd.clear();
    lcd.setCursor(0,0);
    lcd.print("All to default?");

      // Print the Rotary Encoder scroll Menu
    lcd_scroll_Menu((char**)factory_menu_items, menu_size, current_selection, 1, 0,2);
  }

  // Enact selection
  if (Status & SHORT_PUSH)
  {
    Status &=  ~SHORT_PUSH;      // Clear pushbutton status

    switch (current_selection)
    {
      case 0:
        lcd.clear();
        lcd.setCursor(1,1);				
        lcd.print("Nothing Changed");
        Menu_disp_timer = 30;   // Show on LCD for 3 seconds
        //Menu_Mode |= CONFIG;  // We're NOT done, just backing off to Config Menu
        Menu_Mode &=  ~CONFIG;  // We're done
        menu_level = 0;         // We are done with this menu level
        LCD_upd = FALSE;        // Make ready for next time
        break;
      case 1: // Factory Reset
        // Force an EEPROM update upon reboot by storing 0xff in the first address
        EEPROM.write(0,0xff);
        lcd.clear();
        lcd.setCursor(0,0);				
        lcd.print("Factory Reset");
        lcd.setCursor(0,1);
        lcd.print("All default");
        SOFT_RESET();
        //while (1);            // Bye bye, Death by Watchdog
      default:
        lcd.clear();
        lcd.setCursor(1,1);				
        lcd.print("Nothing Changed");
        Menu_disp_timer = 30;   // Show on LCD for 3 seconds
        Menu_Mode &=  ~CONFIG;  // We're done
        menu_level = 0;         // We are done with this menu level
        LCD_upd = FALSE;        // Make ready for next time
        break;
    }
  }
}

//
//--------------------------------------------------------------------
// Manage the first level of Menus
//--------------------------------------------------------------------
//
void menu_level0(void)
{
  static int8_t	current_selection;      // Keep track of current menu selection
  static uint8_t	LCD_upd = FALSE;// Keep track of LCD update requirements

  // Selection modified by encoder.  We remember last selection, even if exit and re-entry
  if (Enc.read()/ENC_MENURESDIVIDE != 0)
  {
    if (Enc.read()/ENC_MENURESDIVIDE > 0)
    {
      current_selection++;
    }
    else if (Enc.read()/ENC_MENURESDIVIDE < 0)
    {
      current_selection--;
    }
    // Reset data from Encoder
    Enc.write(0);

    // Indicate that an LCD update is needed
    LCD_upd = FALSE;
  }

  if (LCD_upd == FALSE)                  // Need to update LCD
  {
    LCD_upd = TRUE;                      // We have serviced LCD

    // Keep Encoder Selection Within Bounds of the Menu Size
    uint8_t menu_size = level0_menu_size;
    while(current_selection >= menu_size)
    current_selection -= menu_size;
    while(current_selection < 0)
    current_selection += menu_size;

    lcd.clear();
    lcd.setCursor(0,0);
    lcd.print("Config Menu:");

    // Print the Menu
    lcd_scroll_Menu((char**)level0_menu_items, menu_size, current_selection,1, 0,3);
  }

  if (Status & SHORT_PUSH)
  {
    Status &= ~SHORT_PUSH;                // Clear pushbutton status

    switch (current_selection)
    {
      case 0: // SWR Alarm Threshold Set
        menu_level = NEW_POS_MENU;
        LCD_upd = FALSE;                  // force LCD reprint
        break;

      case 1: // SWR Alarm Power Threshold Set
        menu_level = MANAGE_POS_MENU;
        LCD_upd = FALSE;                  // force LCD reprint
        break;

      case 2: // SWR Alarm Power Threshold Set
        menu_level = DELETE_POS_MENU;
        LCD_upd = FALSE;                  // force LCD reprint
        break;

      case 3: // SWR Alarm Power Threshold Set
        menu_level = FACTORY_MENU;
        LCD_upd = FALSE;                  // force LCD reprint
        break;

      default:
        // Exit
        lcd.clear();
        lcd.setCursor(1,1);
        Menu_disp_timer = 20;             // Show on LCD for 2 seconds
        lcd.print("Return from Menu");
        Menu_Mode &=  ~CONFIG;            // We're done
        LCD_upd = FALSE;                  // Make ready for next time
    }
  }
}

//
//--------------------------------------------------------------------
// Scan the Configuraton Menu Status and delegate tasks accordingly
//--------------------------------------------------------------------
//
void ConfigMenu(void)
{
  // Select which menu level to manage
  if (menu_level == 0) menu_level0();
  else if (menu_level == NEW_POS_MENU) new_pos_menu();
  else if (menu_level == MANAGE_POS_MENU) manage_pos_menu();
  else if (menu_level == DELETE_POS_MENU) delete_pos_menu();
  else if (menu_level == FACTORY_MENU) factory_menu();
}

//
//--------------------------------------------------------------------
// Multipurpose Pushbutton
//
// Returns 0 for no push, 1 for short push and 2 for long push
// (routine should be called be scanned once every 100ms)
//--------------------------------------------------------------------
//
uint8_t multipurpose_pushbutton(void)
{
  static uint8_t pushcount=0;      // Measure push button time (max 2.5s)
  uint8_t retval=0;                // 1 for short push, 2 for long push
  
  //-------------------------------------------------------------------    
  // Enact Long Push (pushbutton has been held down for a certain length of time):
  //
  if (pushcount >= ENACT_MAX)      // "Long Push", goto Configuration Mode
  {
    Menu_Mode |= CONFIG;           // Switch into Configuration Menu, while		
                                   // retaining memory of runtime function

    Status |= LONG_PUSH;           // Used with Configuraton Menu functions	
    pushcount = 0;                 // Initialize push counter for next time
    retval = 2;
  }
    
  //-------------------------------------------------------------------    
  // Enact Short Push (react on release if only short push)
  //
  else if (digitalRead(EnactSW) == HIGH)  // Pin high = just released, or not pushed
  {
    // Do nothing if this is a release after Long Push
    if (Status & LONG_PUSH)               // Is this a release following a long push?
    {
      Status &= ~LONG_PUSH;               // Clear pushbutton status
    }
    // Do stuff on command
    else if (pushcount >= ENACT_MIN)      // Check if this is more than a short spike
    {	
      Status |= SHORT_PUSH;               // Used with Configuraton Menu functions
      retval = 1;
    }
    pushcount = 0;                        // Initialize push counter for next time
  }
  
  //-------------------------------------------------------------------
  // Determine whether Long Push or Short Push (while pushbutton is down)
  //
  else if (!(Status & LONG_PUSH))         // Button Pushed, count up the push timer
  {                                       // (unless this is tail end of a long push,
    pushcount++;                          //  then do nothing)
  }
  return retval;
}

