//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//---------------------------------------------------------------------------------
// Stepper Position and Memories Management
//---------------------------------------------------------------------------------
//

//
//-----------------------------------------------------------------------------------------
// Initialize Presets
//-----------------------------------------------------------------------------------------
//
void init_Presets(void)
{
  uint8_t i;

  // Scan through all "active" memories
  for (i=0; i<MAX_PRESETS; i++)
  {
    preset[i].Frq = 0;      // An unused frequency preset is Zero Hz.
    preset[i].Pos = 1000000;// An unused Position is 1000000
  }
}


//
//-----------------------------------------------------------------------------------------
// Normalize Stepper Positions, lowest is always 1000000
//-----------------------------------------------------------------------------------------
//
void normalize_stepper_pos(void)
{
  uint8_t i;
  int32_t delta;

  if (num_presets > 1)
  {
    delta = preset[0].Pos - 1000000;
    preset[0].Pos = 1000000;
    // Scan through all "active" memories
    for (i=1; i<num_presets; i++)
    {
      preset[i].Pos = preset[i].Pos - delta;
    }
    running.Pos = running.Pos - delta;
    stepper_track = stepper_track - delta;
  } 
}


//----------------------------------------------------------------------
// A super inelegant quick and dirty Sort() function for
// Frequency and Position Presets from lowest and up,
// but with zero values in highest positions when empty
//----------------------------------------------------------------------
void preset_sort(void)
{
  uint8_t x, sort;
  int32_t temp;

  // Very slow and inelegant :)
  for (x = 0; x < num_presets-1; x++)
  {
    for (sort = 0; sort < num_presets-1; sort++)
    {
      // Empty elements go up
      if ((preset[sort].Frq == 0) && (preset[sort+1].Frq > 0))
      {
        preset[sort].Frq = preset[sort+1].Frq;
        preset[sort+1].Frq = 0;
	
        preset[sort].Pos = preset[sort+1].Pos;
        preset[sort+1].Pos = 0;
      }
      // Empty elements are not shifted down
      else if (preset[sort+1].Frq == 0);
      // Shift elements, one by one
      else if (preset[sort].Frq > preset[sort+1].Frq)
      {
        temp = preset[sort].Frq;
        preset[sort].Frq = preset[sort+1].Frq;
        preset[sort+1].Frq = temp;
	
        temp = preset[sort].Pos;
        preset[sort].Pos = preset[sort+1].Pos;
        preset[sort+1].Pos = temp;
      }
    }
  }
  // Set all uncalibrated Preset Positions at same value as first Preset
  for (x = 1; x < num_presets; x++)
  {
    if (preset[x].Frq == 0)
        preset[x].Pos = preset[0].Pos;
  }		
}


//
//-----------------------------------------------------------------------------------------
// Recalibrate Stepper Position if Short Push
//-----------------------------------------------------------------------------------------
//
void recalibrate_stepper_pos(void)
{
  uint8_t i;

  // Scan through all "active" memories
  for (i=0; i<num_presets; i++)
  {
    preset[i].Pos = preset[i].Pos + delta_Pos;
  }
  // Update all preset memories in EEPROM
  EEPROM_writeAnything(17,preset);

  // Reset the position delta (delta_Pos) to Zero
  delta_Pos = 0;
  EEPROM_writeAnything(9,delta_Pos);  
}


//
//-----------------------------------------------------------------------------------------
// Determine active range, between which two stored presets are we?
//-----------------------------------------------------------------------------------------
//
void determine_range(void)
{
  uint8_t i;

  // Find preset immediately above current frequency
  // Returns 0 if frequency is lower than lowest or
  // higher than highest reference frequency
  range = 0;
  for (i=0; i<num_presets; i++)
  {
    if (running.Frq <= preset[i].Frq)
    {
      range = i;
      break;
    }
  }
}


//
//-----------------------------------------------------------------------------------------
// Derive Stepper Position from Frequency
//-----------------------------------------------------------------------------------------
//
int32_t derive_pos_from_frq(void)
{
  int32_t delta_R, delta_F;
  double  delta_calc;
  int32_t pos;

  if(running.Frq < preset[0].Frq)                   // We are below lowest preset
  {
    pos = preset[0].Pos;
  }
  else if(running.Frq > preset[num_presets-1].Frq)  // We are above highest preset
  {
    pos = preset[num_presets-1].Pos;
  }
  else                                              // We are within presets
  {
    // deltas used to derive Stepper Position from Frequency
    delta_R = preset[range].Pos - preset[range-1].Pos;
    delta_F = preset[range].Frq - preset[range-1].Frq;
    // calculate Stepper Position
    delta_calc  = (double)  delta_R / delta_F;
    delta_calc  = delta_calc * (running.Frq - preset[range-1].Frq);
    pos = (int32_t) delta_calc + preset[range-1].Pos;
  }
  return pos;
}


//
//-----------------------------------------------------------------------------------------
// Derive tuned Frequency from Stepper Position
//-----------------------------------------------------------------------------------------
//
int32_t derive_frq_from_pos(void)
{
  int8_t  i;
  uint8_t pos_range;            // Somewhat redundant, as we already have range
                                // calculated based on frequency. should be same.
                                
  int8_t sign;                  // Used to normalise direction of travel
  int32_t delta_R, delta_F;
  double  delta_calc;
  int32_t frq=0;

  //
  // Calculate Tuned Frequency based on current stepper position
  //
  if (num_presets > 1)          // We need at least 2 presets to calculate
  {
    // Find preset immediately above current stepper position
    // Returns 0 if position is lower than lowest or
    // higher than highest stored reference
    if (preset[1].Pos>preset[0].Pos) sign = 1;
    else sign = -1;
  
    // Find preset immediately above current stepper position
    // Returns 0 if stepper position is lower than or higher than the highest
    // stored position
    // (mabye a bit redundant, as we already calculate range based on frequency)
    pos_range = 0;
    for (i=0; i<=num_presets; i++)
    {
      if ((stepper_track*sign) <= (preset[i].Pos*sign))
      {
        pos_range = i;
        break;
      }
    }
  
    // If we are below lowest preset, use lowest preset interval to calculate
    if(stepper_track < preset[0].Pos)
    {
      pos_range = 1;
    }
    // If we are above highest preset, use highest preset interval to calculate
    else if(stepper_track > preset[num_presets-1].Pos)
    {
      pos_range = num_presets-1;
    }
    
    // deltas used to derive Frequency from Stepper Position
    delta_R = preset[pos_range].Pos - preset[pos_range-1].Pos;
    delta_F = preset[pos_range].Frq - preset[pos_range-1].Frq;
    // calculate Tuned Frequency
    delta_calc = (double)  delta_F / delta_R;
    delta_calc = delta_calc * (stepper_track - preset[pos_range-1].Pos);
    frq  = (int32_t) delta_calc + preset[pos_range-1].Frq;
  }  
  return frq;
}

//
//-----------------------------------------------------------------------------------------
// Variable rate, increases away from start/stop positions
//-----------------------------------------------------------------------------------------
//
uint8_t determine_stepper_rate(void)
{
  static uint32_t rampup;  // Used to ramp up the stepper rate when movement starts
  uint32_t  distance_left; // Used to keep tabs on how far we are from destination
  
  uint8_t  rate;           // Stepper rate to return (rate is inverse of resolution)
                           // 0 is slowest, full resolution of 8 microsteps
                           // 1 for 4 microsteps, 2 for 2 microsteps
                           // or 3 for no microsteps
                           
  // Base rate as determined by User Menu settings - typical value is 0 for 8 microsteps
  rate = STEPPER_MICROSTEPS;
  
  distance_left = abs(stepper_track - (running.Pos + delta_Pos));
  
  // Stay at minimal rate and init rampup if just started or close to desired end position
  if (distance_left <= BACKLASH_ANGLE)
  {
    rampup = 0;            // Initialize rampup
  }
  // Otherwise determine the rate, higher as we are further away from origin and destination
  else
  {
    rampup++;                 // We are moving, moving, moving
  
    #if VARIABLE_RATE == 8    // Eight times normal Rate when tuning "long distance" - defined in ML.h
    // Eight times the stepper rate if tuning distance is more than 4x BACKLASH ANGLE 
    if      ((rampup > (4*BACKLASH_ANGLE)) && (distance_left > (4*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 3;
    // Quadruple the stepper rate if tuning distance is more than 3x BACKLASH ANGLE 
    else if ((rampup > (3*BACKLASH_ANGLE)) && (distance_left > (3*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 2;
    // Double the stepper rate if tuning distance is more than 2x BACKLASH ANGLE 
    else if ((rampup > (2*BACKLASH_ANGLE)) && (distance_left > (2*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 1;
    
    #elif VARIABLE_RATE == 4    // Quadruple Rate when tuning "long distance" - defined in ML.h
    // Quadruple the stepper rate if tuning distance is more than 3x BACKLASH ANGLE 
    if      ((rampup > (3*BACKLASH_ANGLE)) && (distance_left > (3*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 2;
    // Double the stepper rate if tuning distance is more than 2x BACKLASH ANGLE 
    else if ((rampup > (2*BACKLASH_ANGLE)) && (distance_left > (2*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 1;
    
    #elif VARIABLE_RATE == 2  // Double Rate when tuning "long distance"    - defined in ML.h
    // Double the stepper rate if tuning distance is more than 2x BACKLASH ANGLE 
    if      ((rampup > (2*BACKLASH_ANGLE)) && (distance_left > (2*BACKLASH_ANGLE))) rate = STEPPER_MICROSTEPS + 1;
    
    #endif                    // No Variable Rate if not 2, 4 or 8 defined
  }
  
  // Ensure that rate doesn't go above the highest possible setting, 0 microsteps (equals 3)
  if (rate > 3) rate = 3;
  
  return rate;  
}

//
//-----------------------------------------------------------------------------------------
// Rotate Stepper Up or Down, based on calculated position vs stepper_track comparison
//-----------------------------------------------------------------------------------------
//
void rotate_stepper(void)
{
  uint8_t stepper_rate;                   // Determined by microstep_resolution; 
                                          // 0 for full resolution of 8 microsteps
                                          // 1 for 4 microsteps, 2 for 2 microsteps
                                          // or 3 for no microsteps
  uint8_t step_size;
    
  stepper_rate = determine_stepper_rate();
 
  step_size = pow(2,stepper_rate);        // 1 microstep is our smallest unit.   
  
  //
  // Position the Stepper according to Frequency
  //
  // Tune Up
  #if   ENDSTOP_OPT == 1                  // Vacuum Cap, No End stop sensors implemented
  if((stepper_track + (step_size-1)) < (running.Pos + delta_Pos))
  #elif ENDSTOP_OPT == 2                  // End stop sensors implemented
  if(!(Status&END_UPPER) && ((stepper_track + (step_size-1)) < (running.Pos + delta_Pos)))
  #elif ENDSTOP_OPT == 3                  // No End stop sensors implemented
  if((stepper_track + (step_size-1)) < (running.Pos + delta_Pos))
  #endif
  {
    stepper_Incr(stepper_rate);	
    stepper_track += step_size;           // Increase counter in accordance with step size
    Status |= FRQ_TIMER;                  // New frequency, these flags seed timer to monitor if
    Status |= FRQ_STORE;                  // frequency has been stable for a while
  }
  // Tune Down
  #if   ENDSTOP_OPT == 1                  // Vacuum Cap, No End stop sensors implemented
  if(stepper_track > (running.Pos + delta_Pos))  
  #elif ENDSTOP_OPT == 2                  // End stop sensors implemented  
  if(!(Status&END_LOWER) && (stepper_track > (running.Pos + delta_Pos)))  
  #elif ENDSTOP_OPT == 3                  // No End stop sensors implemented
  if(stepper_track > (running.Pos + delta_Pos))  
  #endif
  {
    stepper_Decr(stepper_rate);	
    stepper_track -= step_size;           // Decrease counter in accordance with step size
    Status |= FRQ_TIMER;                  // New frequency, these flags seed timer to monitor if
    Status |= FRQ_STORE;                  // frequency has been stable for a while
  }
}

//
//-----------------------------------------------------------------------------------------
// Rotate Stepper Up or Down, based on calculated position vs stepper_track comparison 
// Backlash on down
//-----------------------------------------------------------------------------------------
//
void rotate_stepper_backlash(void)
{ 
  uint8_t stepper_rate;                   // Determined by microstep_resolution; 
                                          // 0 for full resolution of 8 microsteps
                                          // 1 for 4 microsteps, 2 for 2 microsteps
                                          // or 3 for no microsteps
  uint8_t step_size;
                                      
  stepper_rate = determine_stepper_rate();
  
  step_size = pow(2,stepper_rate);        // 1 microstep is our smallest unit.
  
  //
  // Position the Stepper according to Frequency
  //
  // Tune Up
  #if   ENDSTOP_OPT == 1                  // Vacuum Cap, No End stop sensors implemented
  if(!(Status&BACKLASH) && ((stepper_track + (step_size-1)) < (running.Pos + delta_Pos)))
  #elif ENDSTOP_OPT == 2                  // End stop sensors implemented
  if(!(Status&BACKLASH) && !(Status&END_UPPER) && ((stepper_track + (step_size-1)) < (running.Pos + delta_Pos)))
  #elif ENDSTOP_OPT == 3                  // No End stop sensors implemented
  if(!(Status&BACKLASH) && ((stepper_track + (step_size-1)) < (running.Pos + delta_Pos)))
  #endif
  {
    stepper_Incr(stepper_rate);	
    stepper_track += step_size;           // Increase counter in accordance with step size
    Status |= FRQ_TIMER;                  // New frequency, these flags seed timer to monitor if
    Status |= FRQ_STORE;                  // frequency has been stable for a while
  }
  // Backlash Compensation.  Set flag to Tune Down and then back Up
  if(stepper_track > (running.Pos + delta_Pos))
  {
    Status |= BACKLASH;                   // We need to tune town, set Backlash flag
  }
  // Tune Down, overshooting by STEPPER_BACKLASH
  #if   ENDSTOP_OPT == 1                  // Vacuum Cap, No End stop sensors implemented
  if((Status&BACKLASH) && (stepper_track > (running.Pos + delta_Pos - BACKLASH_ANGLE)) && (stepper_track > preset[0].Pos))  
  #elif ENDSTOP_OPT == 2                  // End stop sensors implemented  
  if((Status&BACKLASH) && !(Status&END_LOWER) && (stepper_track > (running.Pos + delta_Pos - BACKLASH_ANGLE)))  
  #elif ENDSTOP_OPT == 3                  // No End stop sensors implemented
  if((Status&BACKLASH) && (stepper_track > (running.Pos + delta_Pos - BACKLASH_ANGLE)))  
  #endif
  {
    stepper_Decr(stepper_rate);	
    stepper_track -= step_size;           // Decrease counter in accordance with step size--;
    Status |= FRQ_TIMER;                  // New frequency, these flags seed timer to monitor if
    Status |= FRQ_STORE;                  // frequency has been stable for a while
  }
  else Status &= ~BACKLASH;               // Clear the Backlash flag once we have tuned all the way down
}

