//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//---------------------------------------------------------------------------------
// UART Management Routines
//
// Parse Frequency Input from Transceiver and deliver it to the global variable:
// running.Frq
//---------------------------------------------------------------------------------
//

#if ASYNC_SER_READ                              // Read anything that comes off the serial port
//
//---------------------------------------------------------------------------------
// Some Transceivers may be broadcasting data on the serial port at any
// time. Hence incoming traffic from these transceivers must be handled
// in an asynchronous manner, rather than just assuming that anything
// received is in response to a polled query by the Controller.
// Examples of this include ICOM CI-V, Elecraft K3/KX3...
//---------------------------------------------------------------------------------
//

#if TRX_CIV_AUTO or TRX_CIV_POLL                // ICOM CI-V data, automatic or polled
const char end_delimeter = 0xfd;                // Message 'End" symbol
#endif

#if TRX_FT2000_POLL                             // Yaesu FT450/950/1200/2000/3000/5000 CAT poll
const char end_delimeter = ';';                 // Message 'End" symbol
#endif

#if TRX_K3_AUTO or TRX_K3_POLL                  // Elecraft K3/KX3
const char end_delimeter = ';';                 // Message 'End" symbol
#endif

char serial_in_string[80];                      // Processed Incoming serial string to parse
static bool  Incoming = FALSE;                  // Flag - Is there is a message ready to parse

void serial_async_read(void)
{
  static uint8_t i=0;		                // message character count
  
  if(Uart.available())                          // Are incoming characters waiting to be processed?
  {
    if (i < 80)                                 // Make sure we don't overstep buffer
    {
      serial_in_string[i] = Uart.read();        // Read the character from UART buffer
      
      if (serial_in_string[i] == end_delimeter) // End of string - Indicate that we have stuff to process
      {
        serial_in_string[i] == 0;               // Terminate string with a zero
        i = 0;                                  // set character count to zero
        Incoming = TRUE;                        // Indicate that we have a new message to parse
      }
      else i++;                                 // character count +1
    }
    else
    {
      i = 0;                                    // set character count to zero
      serial_in_string[0] == 0;                 // Terminate string with a zero
    }
  }
}
#endif


#if TRX_CIV_AUTO or TRX_CIV_POLL // ICOM style CI-V read frequency data
//
//---------------------------------------------------------------------------------
// ICOM style CI-V Read for either polled or auto
//---------------------------------------------------------------------------------
//
// This function uses input from serial_async_read()
void parse_serial_input(void)
{
  uint32_t civ_value;	// CI-V data as a 32bit integer

  if (Incoming == TRUE)
  {
    // Check for valid beginning of stringBeginning of string
    if (((uint8_t)serial_in_string[0]==0xfe) && ((uint8_t)serial_in_string[1]==0xfe))
    {                                          // start parsing
      // Check if this is indicated as a Frequency message
      // 0x00=Transfer operating frq, 0x03/0x05=Read/Write operating frq
      if ((serial_in_string[4] == 0x00)||(serial_in_string[4] == 0x03)||(serial_in_string[4] == 0x05))
      {
        // Convert BCD to value
        civ_value = (serial_in_string[5] & 0x0f);				 // Hz
        civ_value += 10 * ((serial_in_string[5] & 0xf0)>>4);			 // 10 x Hz

        civ_value += 100 *  (serial_in_string[6] & 0x0f);			 // 100 x Hz
        civ_value += 1000 * ((serial_in_string[6] & 0xf0)>>4);		 // kHz

        civ_value += (uint32_t) 10000 * (serial_in_string[7] & 0x0f);		 // 10 x kHz
        civ_value += (uint32_t) 100000 * ((serial_in_string[7] & 0xf0)>>4);	 // 100 x kHz

        civ_value += (uint32_t) 1000000 * (serial_in_string[8] & 0x0f);	 // MHz
        civ_value += (uint32_t) 10000000 * ((serial_in_string[8] & 0xf0)>>4);	 // 10 x MHz

        civ_value += (uint32_t) 100000000 * (serial_in_string[9] & 0x0f);	 // 100 x MHz
        civ_value += (uint32_t) 1000000000 * ((serial_in_string[9] & 0xf0)>>4);// GHz

        //
        // Update running frequency of the application
        //
        if ((civ_value > 1000000) && (civ_value < 31000000))	// Impose some sane boundaries
        {
          running.Frq = civ_value;
          Status |= FRQ_RADIO;	// Indicate that we are receiving frq data from Radio
        }

      }
    }
  }
  Incoming = FALSE;
}
#endif

#if TRX_CIV_POLL                        // ICOM CI-V Poll for frequency  
//
//---------------------------------------------------------------------------------
// ICOM CI-V style polling for frequency
//---------------------------------------------------------------------------------
//
void request_frequency(void)
{
  // Transmit a request to read operating frequency
  Uart.write(0xfe);
  Uart.write(0xfe);
  Uart.write(CIV_TRX_ADDRESS);
  Uart.write(0xe0);
  Uart.write(0x03);
  Uart.write(0xfd);
}
#endif


#if TRX_FT2000_POLL                     // Yaesu FT450/950/1200/2000/3000/5000 CAT poll
//
//---------------------------------------------------------------------------------
// Yaesu FT-2000 / FT-950 / FT-450 style polling for frequency
//---------------------------------------------------------------------------------
//
// This function uses input from serial_async_read()
void parse_serial_input(void)
{
  char *pEnd;
  
  if (Incoming == TRUE)
  {
    if (!strncmp("FA",serial_in_string,2))  // Grab frequency if header is correct
    {
      running.Frq = strtol(serial_in_string+2,&pEnd,10);
      Status |= FRQ_RADIO;	        // Indicate that we are receiving frq data from Radio 
    }
  }
  Incoming = FALSE;
}

void request_frequency(void)
{
  Uart.print("FA;");                    // Transmit a frequency poll request to transceiver
}
#endif


#if TRX_FT8X7_POLL                      // Yaesu FT817/847/857/897 - Implemened, may work but not tested
//
//---------------------------------------------------------------------------------
// Yaesu FT-817 / FT-847 / FT-857 / FT-897 style polling for frequency
//---------------------------------------------------------------------------------
//
void ft857_parse_serialinput(void)
{
  uint8_t x;
  uint32_t incoming;
  
  uint8_t waiting;                      // Number of chars waiting in receive buffer

  // Find out how many characters are waiting to be read. 
  waiting = Uart.available();
  if (waiting >= 5)                     // Incoming string of correct length detected
  { 
    // Decode frequency content of the incoming message
    x = Uart.read();
    incoming = 100000000 * ((x & 0xf0) >> 4); // 100 x MHz
    incoming += 10000000 * (x & 0x0f);        // 10 x MHz    
    x = Uart.read();
    incoming += 1000000 * ((x & 0xf0) >> 4);  // 1 x MHz
    incoming += 100000 * (x & 0x0f);          // 100 x kHz
    x = Uart.read();
    incoming += 10000 * ((x & 0xf0) >> 4);    // 10 x kHz
    incoming += 1000 * (x & 0x0f);            // 1 x kHz
    x = Uart.read();
    incoming += 100 * ((x & 0xf0) >> 4);      // 100 x Hz
    incoming += 10 * (x & 0x0f);              // 10 x kHz
      
    if ((incoming > 1000000) && (incoming < 31000000))	// Impose some sane boundaries
    {
      running.Frq = incoming;
      Status |= FRQ_RADIO;	        // Indicate that we are receiving frq data from Radio
    }
  }
  Uart.clear();                         // Flush/Discard any received data that has not been read.
}

void request_frequency(void)
{
  ft857_parse_serialinput();            // Grab input from latest poll

  // Transmit a Status request to Yaesu FT100
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x03);
}
#endif


#if TRX_FT100_POLL                      // Yaesu FT100 CAT poll
//
//---------------------------------------------------------------------------------
// Yaesu FT-100 style polling for frequency
//---------------------------------------------------------------------------------
//
void ft100_parse_serialinput(void)
{
  uint8_t x;
  uint32_t incoming;
  
  //char *pEnd;
  uint8_t waiting;                      // Number of chars waiting in receive buffer

  // Find out how many characters are waiting to be read. 
  waiting = Uart.available();
  if (waiting >= 16)                    // Incoming string of correct length detected
  {                                     // (for some reason FT100 responds with 2x 16 bytes)
    Uart.read();                        // Flush the first byte (band information)
    
    // Decode frequency content of the incoming message
    incoming = Uart.read()<<24;
    incoming += Uart.read()<<16;
    incoming += Uart.read()<<8;
    incoming += Uart.read();
    incoming = incoming * 1.25;

    if ((incoming > 1000000) && (incoming < 31000000))	// Impose some sane boundaries
    {
      running.Frq = incoming;
      Status |= FRQ_RADIO;	        // Indicate that we are receiving frq data from Radio
    }
  }
  Uart.clear();                         // Flush/Discard any received data that has not been read.
}

void request_frequency(void)
{
  ft100_parse_serialinput();            // Grab input from latest poll

  // Transmit a Status request to Yaesu FT100
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x00);
  Uart.write(0x10);
}
#endif


#if TRX_K3_AUTO or TRX_K3_POLL          // Elecraft K3/KX3
//
//---------------------------------------------------------------------------------
// Elecraft K3 or KX3 Read for either polled or auto
//---------------------------------------------------------------------------------
//
// This function uses input from serial_async_read()
void parse_serial_input(void)
{
  char *pEnd;

  if (Incoming == TRUE)
  {
    if (!strncmp("IF",serial_in_string,2))  // Grab frequency if header is correct
    {
      running.Frq = strtol(serial_in_string+2,&pEnd,10);
      Status |= FRQ_RADIO;	            // Indicate that we are receiving frq data from Radio 
    }
  }
  Incoming = FALSE;
}
#endif

#if TRX_K3_POLL                         // Elecraft K3/KX3
//
//---------------------------------------------------------------------------------
// Elecraft K3 or KX3 style polling for frequency
//---------------------------------------------------------------------------------
//
void request_frequency(void)
{
  Uart.print("IF;");                    // Transmit a frequency poll request to transceiver
}
#endif



