//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//-----------------------------------------------------------------------------------------
//      Parse and act upon an incoming USB command; USB port enumerates as a COM port
//
//      Commands are:
//
//      $frqget                  Retrieve running frequency
//
//      $frqset mmkkkhhh         Update running frequency (equivalent to Transceiver serial input)
//                               Frequency is entered in Hz, e.g. 14100000 is 14.1 MHz.
//
//      $memoryget               Retrieve all memory presets.
//                               Format of retrieved values is:
//                                A FFFFFFFF PPPPPPP
//                                  where
//                                A is memory position
//                                F is frequency in Hz
//                                P is stepper motor position (relative value)
//                                  e.g.
//                                0 14000000 1000000
//                                1 14100000 1000100
//                                2 14200000 1000200
//                                ...
//
//      $memoryset A FFFFFFFF PPPPPPP
//                               Enter new memory preset.
//                               Format is same as in $memoryget, however only one preset
//                               is entered at a time.  The Presets MUST be entered with the
//                               Frequency memories in an ascending order and the highest
//                               preset memory entered MUST not be higher than the maximum
//                               number of presets available.
//
//
//      $version                 Report version and date of firmware.
//
//      $debug                   This one may provide unexpected results !!! :-)
//-----------------------------------------------------------------------------------------
//

char incoming_command_string[50];			// Input from USB Serial

void usb_parse_incoming(void)
{
  uint8_t x;
  char *pEnd;
  uint16_t inp_val;
  double inp_double;

  if (!strcmp("frqget",incoming_command_string))	// Retrieve calibration values
  {
    Serial.print(running.Frq);
  }
  
  else if (!strncmp("frqset",incoming_command_string,6))	// Write new calibration values
  {
    running.Frq = strtol(incoming_command_string+6,&pEnd,10);
    Status |= FRQ_RADIO;	// Indicate that we are receiving frq data from Radio 
  }
  
  else if (!strcmp("memoryget",incoming_command_string))	// Retrieve calibration values
  {
    // If there are no active presets, then there is nothing to do
    if (num_presets == 0)
    {
      Serial.print("Nothing stored\r\n");
    }
    else
    {
      for(x=0; x<num_presets; x++)
      {
        sprintf(lcd_buf,"%u %lu %lu\r\n",x,preset[x].Frq,preset[x].Pos);
        Serial.print(lcd_buf);
      }
    }
  }
  
  else if (!strncmp("memoryset",incoming_command_string,9))  // Write new calibration values to EEPROM
  {
    if (strlen(incoming_command_string) > 20)                // Only process if string contains data
    {
      x =  strtol(incoming_command_string+9,&pEnd,10);
      preset[x].Frq = strtol(pEnd,&pEnd,10);
      preset[x].Pos = strtol(pEnd,&pEnd,10);
      EEPROM_writeAnything(17 + x*8,preset[x]);
    }
    
    // Ensure that the Stepper Motor doesn't start moving - Assume that it is correctly tuned
    // Is input from Radio active?
    if (Status & FRQ_RADIO)
    {
      determine_range();                                    // Find stored preset immediately above current frequency
                                                            // Returns 0 if frequency is lower than lowest or
                                                            // higher than highest reference frequency
                                                            
      if(range)                                             // We are within a range of stored positions
      {
        stepper_track = derive_pos_from_frq();              // Initialize stepper_track and running Position
        running.Pos = stepper_track;
        delta_Pos = 0;
      }
      else if(running.Frq < preset[0].Frq)                  // We are below lowest preset
      {
        stepper_track = preset[0].Pos;                      // Initialize stepper_track and running Position
        running.Pos = stepper_track;                        // to the lowest preset
        delta_Pos = 0;
        tunedFrq =  preset[0].Frq;                          // Indicate which frequency stepper is tuned to
      }
      else                                                  // We are above highest preset
      {
        stepper_track = preset[num_presets-1].Pos;          // Initialize stepper_track and running Position
        running.Pos = stepper_track;                        // to the highest preset
        delta_Pos = 0;
        tunedFrq =  preset[num_presets-1].Frq;              // Indicate which frequency stepper is tuned to
      } 
    }  
  }
  
  else if (!strcmp("version",incoming_command_string))      // Poll for one single Human Readable report
  {
    sprintf(lcd_buf,"TF3LJ/VE2LJX Magloop Controller\r\n");
    Serial.print(lcd_buf);
    Serial.print("Version ");
    Serial.print(VERSION);
    Serial.print(" ");
    Serial.println(DATE);
  }
  
  else if (!strcmp("debug",incoming_command_string))	    // Retrieve calibration values
  {
    if (num_presets == 0)                                   // If no stored presets, then nothing to do
    {
      Serial.print("Nothing stored\r\n");
    }
    else
    {
      for(x=0; x<num_presets; x++)
      {
        sprintf(lcd_buf,"%u %lu %lu\r\n", x, preset[x].Frq, preset[x].Pos);
        Serial.print(lcd_buf);
      }
    }
    sprintf(lcd_buf,"Debug Running: %ld %ld\r\n", running.Frq, running.Pos);
    Serial.print(lcd_buf);
    sprintf(lcd_buf,"Stepper Track: %ld\r\n", stepper_track);
    Serial.print(lcd_buf);
    sprintf(lcd_buf,"Delta Pos    : %ld\r\n", delta_Pos);
    Serial.print(lcd_buf);
    sprintf(lcd_buf,"Range        : %ld\r\n", range);
    Serial.print(lcd_buf);  
    sprintf(lcd_buf,"Presets      : %ld\r\n", num_presets);
    Serial.print(lcd_buf);  
  }
}	

//
//-----------------------------------------------------------------------------------------
// 			Monitor USB Serial port for an incoming USB command
//-----------------------------------------------------------------------------------------
//
void usb_read_serial(void)
{
  static uint8_t a;    // Indicates number of chars received in an incoming command
  static bool Incoming;
  uint8_t ReceivedChar;
  uint8_t waiting;     // Number of chars waiting in receive buffer

  // Find out how many characters are waiting to be read. 
  waiting = Serial.available();
  // Scan for a command attention symbol -> '$'
  if (waiting && (Incoming == FALSE))
  {
    ReceivedChar = Serial.read();
    // A valid incoming message starts with an "attention" symbol = '$'.
    // in other words, any random garbage received on USB port is ignored.
    if (ReceivedChar == '$')  // Start command symbol was received,
    {                         // we can begin processing input command
      Incoming = TRUE;
      a = 0;
      waiting--;
    }
  }	
  // Input command is on its way.  One or more characters are waiting to be read
  // and Incoming flag has been set. Read any available bytes from the USB OUT endpoint
  while (waiting && Incoming)
  {
    ReceivedChar = Serial.read();
    waiting--;
    if (a == sizeof(incoming_command_string)-1)// Line is too long, discard input
    {
      Incoming = FALSE;
      a = 0;
    }
    // Check for End of line
    else if ((ReceivedChar=='\r') || (ReceivedChar=='\n'))
    {
      incoming_command_string[a] = 0;// Terminate line
      usb_parse_incoming();// Parse the command
      Incoming = FALSE;
      a = 0;
    }
    else              // Receive message, char by char
    {
      incoming_command_string[a] = ReceivedChar;
    }
    a++;              // String length count++
  }
}


