//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//** History.........: Check the ML_xx.c file
//**
//**
//*********************************************************************************

//
//---------------------------------------------------------------------------------
// Arduino style Bipolar Stepper Motor Control Routine,
// using 2x Allegro A4975 Microstepping Full-Bridge Motor Drivers
//---------------------------------------------------------------------------------
//

const uint8_t out_seq[]= { 0xcc,0xbd,0xae,0x9f,  // As per A4975 Datasheet
                           0x0f,0x1f,0x2e,0x3d,  // "Table 4 - Step Sequencing"	
                           0x4c,0x5b,0x6a,0x79,  // Order of bits is:
                           0x70,0x71,0x62,0x53,  // b7, b6, b5, b4, b3, b2, b1, b0
                           0x44,0x35,0x26,0x17,  // PhA,D2A,D1A,D0A,PhB,D2B,D1B,D0B
                           0x07,0x97,0xa6,0xb5,  // where A denotes Bridge A
                           0xc4,0xd3,0xe2,0xf1,  // and   B denotes Bridge B
                           0xf0,0xf9,0xea,0xdb };
/*
// Do this in ML.h:
// Assign Output Pins to Stepper
const int PhA = 22;
const int D2A = 21;
const int D1A = 20;
const int D0A = 19;
const int PhB = 18;
const int D2B = 17;
const int D1B = 16;
const int D0B = 15;
*/

int8_t phase=0;	    // tracks the current step out of a total of 32

void stepper_Write(uint8_t out)
{
  if (out & 0x80) digitalWrite(PhA, HIGH);
  else digitalWrite(PhA, LOW);  
  if (out & 0x40) digitalWrite(D2A, HIGH);
  else digitalWrite(D2A, LOW);  
  if (out & 0x20) digitalWrite(D1A, HIGH);
  else digitalWrite(D1A, LOW);  
  if (out & 0x10) digitalWrite(D0A, HIGH);
  else digitalWrite(D0A, LOW);  
  if (out & 0x08) digitalWrite(PhB, HIGH);
  else digitalWrite(PhB, LOW);  
  if (out & 0x04) digitalWrite(D2B, HIGH);
  else digitalWrite(D2B, LOW);  
  if (out & 0x02) digitalWrite(D1B, HIGH);
  else digitalWrite(D1B, LOW);  
  if (out & 0x01) digitalWrite(D0B, HIGH);
  else digitalWrite(D0B, LOW);  
}

//
// Init Stepper Outputs
//
void stepper_Init(void)
{
  //
  // Set outputs
  //
  pinMode(PhA, OUTPUT);      
  pinMode(D2A, OUTPUT);      
  pinMode(D1A, OUTPUT);      
  pinMode(D0A, OUTPUT);      
  pinMode(PhB, OUTPUT);      
  pinMode(D2B, OUTPUT);      
  pinMode(D1B, OUTPUT);      
  pinMode(D0B, OUTPUT);      

  // Ensure Power Off state
  stepper_Write(0); 
}

//
// Increment Stepper
//
void stepper_Incr(uint8_t res)
{
  phase = phase + pow(2,res);          // 0 for full resolution of 8 microsteps
                                       // 1 for 4 microsteps, 2 for 2 microsteps
                                       // or 3 for no microsteps
  if (phase >= 32) phase = phase - 32;
  stepper_Write(out_seq[phase]);
}

//
// Decrement Stepper
//
void stepper_Decr(uint8_t res)
{
  phase = phase - pow(2,res);          // 0 for full resolution of 8 microsteps
                                       // 1 for 4 microsteps, 2 for 2 microsteps
                                       // or 3 for no microsteps
  if (phase < 0) phase = phase + 32;
  stepper_Write(out_seq[phase]);
}

void stepper_PwrOff(void)
{
  stepper_Write(0); 
}

