//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//** Copyright (C) 2014  Loftur E. Jonasson  (tf3lj [at] arrl [dot] net)
//**
//** This program is free software: you can redistribute it and/or modify
//** it under the terms of the GNU General Public License as published by
//** the Free Software Foundation, either version 3 of the License, or
//** (at your option) any later version.
//**
//** This program is distributed in the hope that it will be useful,
//** but WITHOUT ANY WARRANTY; without even the implied warranty of
//** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//** GNU General Public License for more details.
//**
//** You should have received a copy of the GNU General Public License
//** along with this program.  If not, see <http://www.gnu.org/licenses/>.
//**
//** Platform........: Teensy++ 2.0, Teensy 3 or Teensy 3.1 (http://www.pjrc.com)
//**                   (target platform is Teensy 3.1)
//**                   (Some other Arduino type platforms may also work
//**                    with no more than minimal tweaks being necessary)
//**
//** Initial version.: 0.00, 2012-10-20  Loftur Jonasson, TF3LJ / VE2LJX
//**                   (pre-alpha version)
//**
//**
//*********************************************************************************

// For the latest version of the Magnetic Loop Controller Firmware, see webpage:
// https://sites.google.com/site/lofturj/to-automatically-tune-a-magnetic-loop-antenna
// This is an OLD and less capable version - maintaining compatibility with Teensy 2++

//------------------------------------------------------------------------
// For selection of features such as:
// - Frequency Input Format from Transceiver;
// - features such as endstop sensor functionality; and
// - Microcontroller pin assignments:
//
//   See ML.h
//

#include <LiquidCrystalFast.h>
#include <Metro.h>
#include <EEPROM.h>
#include <Encoder.h>

#include "_EEPROMAnything.h"
#include "ML.h"


var_track running;                  // Running Frequency and Position (struct defined in ML.h)
var_track preset[MAX_PRESETS];      // Frequency and Position presets

uint8_t   num_presets;              // Number of active (stored) presets
uint8_t   range;                    // Active range (which presets are we in-between?)

uint8_t   Status = 0;               // Contains various status flags (bools)
uint8_t   Menu_Mode;                // Menu Mode Flags
uint8_t   Menu_disp_timer;          // Used for a timed display when returning from Menu
int16_t   encOutput;                // Output From Encoder

// Timers to monitor whether we are receiving Frequency information from Radio
uint8_t   frq_check_timer;          // Measures frequency receive outage time, 10x is 1 second
uint8_t   frq_reset_timer;          // 5 second frequency receive poll time

char      lcd_buf[80];              // Print format buffer for lcd/serial/usb output

int32_t   stepper_track;            // Current position of Stepper
int32_t   delta_Pos;                // delta or difference between position based on presets and
                                    // the actual current position as a result of tuning Encoder
                                    // or Up/Down buttons
int32_t   tunedFrq;                 // Frequency calculated from the current position of Stepper

// Instanciate an Encoder Object
Encoder   Enc(EncI, EncQ);

// Instanciate a couple of metro objects
Metro     fastMetro = Metro(LOOP_RATE); // Typically 2 milliseconds
Metro     slowMetro = Metro(100);       // 100 milliseconds for various tasks
Metro     lcd_Metro = Metro(50);        // 50 milliseconds for LCD update

#if TRX_FREQUENCY_POLL                  // Used if TRX CAT Protocol requires frequency to be polled
// Poll Transceiver for frequency data (typical POLL_RATE is 1000 = 1 second
Metro     trxpollMetro = Metro(POLL_RATE);
#endif

// initialize the library with the numbers of the interface pins
LiquidCrystalFast lcd(LCD_RS, LCD_RW, LCD_E, LCD_D4, LCD_D5, LCD_D6, LCD_D7);

// Define a "Uart" object to access the serial port
HardwareSerial Uart = HardwareSerial();


//
//-----------------------------------------------------------------------------------------
// Track frequency and position
//-----------------------------------------------------------------------------------------
//
void track_frq(void)
{
  int8_t  i;

  // Find total number of active preset
  for (num_presets=0; (num_presets<MAX_PRESETS) && (preset[num_presets].Frq != 0); num_presets++);

  #if ENDSTOP_OPT == 1        // Vacuum Capacitor, no end stops
  //
  // Protect capacitor from damage in case of misbehavior by software by
  // blocking automatic tuning if we are outside of stored presets
  //
  // Indicate if frequency is outside of the range of configured presets
  // (need a minimum of 2 presets for this to work)
  if((num_presets < 2) || (running.Frq < preset[0].Frq) || (running.Frq > preset[num_presets - 1].Frq))
  {
    Status |= FRQ_XRANGE;     // Frequency is out of range
  }
  else
  {
    Status &=  ~FRQ_XRANGE;   // Frequency is within range
  }
  // Indicate if capacitor position is outside of preset range 
  if(((stepper_track - 1 > preset[0].Pos) && (stepper_track - 1 > preset[num_presets - 1].Pos)) ||
        ((stepper_track + 1 < preset[0].Pos) && (stepper_track + 1 < preset[num_presets - 1].Pos)))
  {
    Status |= CAP_XRANGE;     // Capacitor is out of range
  }
  else
  {
    Status &=  ~CAP_XRANGE;   // Capacitor is within range
  }

  //
  // Inside presets, It is OK to do stuff
  //
  if (!(Status & (FRQ_XRANGE | CAP_XRANGE)))
  {
    //
    // Calculate Stepper position to tune to, based on indicated frequency
    //
    determine_range();                     // Find stored preset immediately above current frequency
                                           // Returns 0 if frequency is lower than lowest or
                                           // higher than highest reference frequency    
    running.Pos = derive_pos_from_frq();   // Calculate Stepper Pos based on Frequency
  }
  
  tunedFrq = derive_frq_from_pos();        // Calculate Tuned Frequency based on Stepper Pos 
  
  #else  // No EndStop smart logic needs to be applied
  //
  // Calculate Stepper position to tune to, based on indicated frequency
  // (this can only be done if we already have two active presets)
  if(num_presets >= 2)
  {
    determine_range();                     // Find stored preset immediately above current frequency
                                           // Returns 0 if frequency is lower than lowest or
                                           // higher than highest reference frequency 
    //
    // Calculate stepper position to tune to if transceiver frequency
    // is within the range of stored positions
    //
    if (range)
    {
      running.Pos = derive_pos_from_frq(); // Calculate Stepper Pos based on Frequency
    }
    
    tunedFrq = derive_frq_from_pos();      // Calculate Tuned Frequency based on Stepper Pos
  }
  #endif
}


void loop()
{
  static uint16_t frq_store_timer;         // Keep track of when last update, in order to store
                                           // current status if stable
  uint8_t         buttonstate;             // State of Multipurpose Enact Switch\
                                           // (Multipurpose pushbutton)
  static int8_t   rate_reducer=0;          // Stepper scan rate divisor when Up/Down pushbutton
  static bool     UpDown;                  // Indicates Up/Down pushbuttons being pressed

  //-------------------------------------------------------------------------------
  // Here we do routines which are to be run through as often as possible
  //-------------------------------------------------------------------------------

  #if STUPIDENCODER                        // If Encoder is not connected to interrupt pins
  Enc.read();                              // then it has to be scanned regularly
  #endif

  //-------------------------------------------------------------------------------
  // Here we do routines which are to be accessed once every approx 2ms
  //-------------------------------------------------------------------------------
  if (fastMetro.check() == 1)              // check if the metro has passed its interval .
  {
    //-------------------------------------------------------------------
    // Asynchronous management of USB and Serial ports
    //
    usb_read_serial();                     // Read and parse anything on the USB serial port
   
    #if ASYNC_SER_READ                     // Read anything that comes off the serial port
    serial_async_read();                   // Read anything that comes off the serial port  
    parse_serial_input();                  // Parse captured input string for Frequency Data 
    #endif 

    //-------------------------------------------------------------------
    // Track frequency
    //
    track_frq();

    #if ENDSTOP_OPT == 2                   // End stop sensors implemented
    //-------------------------------------------------------------------
    // Check End Stop Sensors   
    if (digitalRead(EndStopUpper) == LOW)
      Status |= END_UPPER;                 // Set Flag indicating Upper Endstop limit
    else Status &= ~END_UPPER;             // Clear Flag indicating Upper Endstop limit
    if (digitalRead(EndStopLower) == LOW)
      Status |= END_LOWER;                 // Set Flag indicating Lower Endstop limit
    else Status &= ~END_LOWER;             // Clear Flag indicating Lower Endstop limit
    #endif

    //-------------------------------------------------------------------
    // Manually Move Stepper back and forth unless in Config Mode
    //
    if (!(Menu_Mode & CONFIG))
    {						
      //-------------------------------------------------------------------
      // Manually Move Stepper back and forth using Encoder, unless in Config Mode
      //						
      encOutput = Enc.read();
      #if ENDSTOP_OPT == 2                 // End stop sensors implemented
      if (!(Status&END_UPPER) && (encOutput/ENC_TUNERESDIVIDE > 0))
      #else
      if (encOutput/ENC_TUNERESDIVIDE > 0)
      #endif
      {						
        delta_Pos++;                       // Update position
        Enc.write(encOutput - ENC_TUNERESDIVIDE);
        rotate_stepper();
      }
      #if ENDSTOP_OPT == 2                 // End stop sensors implemented
      if (!(Status&END_LOWER) && (encOutput/ENC_TUNERESDIVIDE < 0))
      #else
      if (encOutput/ENC_TUNERESDIVIDE < 0)
      #endif
      {	
        delta_Pos--;                       // Update position
        Enc.write(encOutput + ENC_TUNERESDIVIDE);
        rotate_stepper();                  // No backlash on down when Encoder is used
      }

      //-------------------------------------------------------------------
      // Move stepper by using UP or DOWN push buttons
      //
      if (rate_reducer == 0)                 // Only read pushbutton if rate reducer has matured
      {          
        #if ENDSTOP_OPT == 1                 // Vacuum Capacitor, no end stops
        // (this can only be done if we are inside of known FRQ and CAP range)

        // UP switch has been pressed and We're at or above min but below max      
        if((digitalRead(UpSW) == LOW) && (stepper_track >= preset[0].Pos) && (stepper_track < preset[num_presets - 1].Pos))
        {
          delta_Pos++;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }
        // Down switch has been pressed and We're at or below max but above min      
        if((digitalRead(DnSW) == LOW) && (stepper_track > preset[0].Pos) && (stepper_track <= preset[num_presets - 1].Pos))
        {
          delta_Pos--;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }        
        #elif ENDSTOP_OPT == 2               // End stop sensors implemented
        // UP switch has been pushed and we're not at upper limit of range
        if ((digitalRead(UpSW) == LOW) && !(Status&END_UPPER))
        {
          delta_Pos++;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }
        // DOWN switch has been pushed and we're not at lower limit of range
        else if ((digitalRead(DnSW) == LOW) && !(Status&END_LOWER))
        {
          delta_Pos--;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }
        #elif ENDSTOP_OPT == 3               // Butterfly capacitor, no end stops required
        if (digitalRead(UpSW) == LOW)        // UP switch has been pushed
        {
          delta_Pos++;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }
        else if (digitalRead(DnSW) == LOW)   // DOWN switch has been pushed
        {
          delta_Pos--;                       // Update position
          rate_reducer = UP_DOWN_RATE;       // Reduce stepper rate
        }
        #endif
      }    
    }

    //-------------------------------------------------------------------
    // Rotate stepper based on changes by auto track_frq or Up/Down switches
    //
    // Move stepper if rate_reducer has matured (is zero),
    // used with Up/Down switches. rate_reducer is always zero if auto track_frq
    if (rate_reducer < 2)
    {
      #if STEPPER_BACKLASH
      rotate_stepper_backlash();             // Down direction with a backlash
      #else
      rotate_stepper();                      // No backlash
      #endif
    }
    if (rate_reducer)
    {
      rate_reducer--;
    }   
  }

  //-------------------------------------------------------------------------------
  // Here we do routines which are to be accessed once every 1/10th of a second
  //-------------------------------------------------------------------------------
  if (slowMetro.check() == 1)              // check if the metro has passed its interval .
  {
    //-------------------------------------------------------------------
    // The Menu function has 5 seconds lag time precedence
    if (Menu_disp_timer) Menu_disp_timer--;
    if (Menu_disp_timer == 1) lcd.clear();

    //-------------------------------------------------------------------
    // Store Frequency in EEPROM  and power down the Stepper Motor when stable
    //
    if (Status & FRQ_STORE) 
    {
      if(Status & FRQ_TIMER)
      {
        Status &= ~FRQ_TIMER;              // Clear flag
        frq_store_timer = 0;               // Reset timer
      }
      frq_store_timer++;

      if (frq_store_timer == 50)           // Store current pos if stable for 5 seconds
      {
        frq_store_timer = 0;               // Reset timer
        Status &= ~FRQ_STORE;              // Clear flag
        EEPROM_writeAnything(1,running);   // Write current frq/position into EEPROM
        EEPROM_writeAnything(9,delta_Pos); // Write current delta Position into EEPROM
                                           // (delta-pos is accumulated offset, by turning
                                           //  of Encoder and Up/Down buttons)
        EEPROM_writeAnything(13,stepper_track); // Write current stepper tracl into EEPROM
                                           // stepper_track may be different from running.Pos
                                           // + delta_Pos, if frequency is outside of range 
        stepper_PwrOff();                  // Power down the stepper
      }
    }

    //-------------------------------------------------------------------
    // Pushbutton state stuff
    //    
    buttonstate = multipurpose_pushbutton();    
    if (buttonstate == 1)                  // Short Push detected
    {
      Status |= SHORT_PUSH;                // Used with Configuraton Menu functions
    }
    else if (buttonstate == 2)             // Long Push detected
    {
      Menu_Mode |= CONFIG;                 // Activate Configuration Menu
    }    
    //-------------------------------------------------------------------
    // Configuration Mode Menu Mode
    //
    if (Menu_Mode & CONFIG)
    {
      ConfigMenu();    
    }

    //-------------------------------------------------------------------
    // Normal running Mode
    //   
    // Short Push of the Push Button
    else if (Status & SHORT_PUSH)
    {
      Status &= ~SHORT_PUSH;               // Clear short push flag     
      //
      // Various things to be done if short push... depending on which mode is active
      //
      recalibrate_stepper_pos();
    }
    // Business as usual
    else
    {
      //
      // Check against a timer whether we are receiving new Frequency data from Transceiver
      //
      #if TRX_FREQUENCY_POLL               // We only do this if in Polled Mode
      frq_reset_timer++;
      if (frq_reset_timer == 50)
      {
        Status &= ~FRQ_RADIO;              // Clear Radio Frequncy Received Status once every 5 sec
        // to check whether it is being set by the civ_parser()
        frq_reset_timer = 0;	
      }
      #endif
    }
  }
  
  //-------------------------------------------------------------------------------
  // Display stuff
  //-------------------------------------------------------------------------------
  if (lcd_Metro.check() == 1)              // check if the metro has passed its interval .
  {
    lcd_display();                         // The LCD routines only service one line at a time,
                                           // hence it takes 4 calls to update the whole
                                           // display.  This is necessary for not to interfere
                                           // with the stepper motor timing.
  }

  #if TRX_FREQUENCY_POLL
  //-------------------------------------------------------------------------------
  // Here we do routines needed if TRX Serial Comms Protocol requires frequency to be polled
  //-------------------------------------------------------------------------------
  if (trxpollMetro.check() == 1)           // check if the metro has passed its interval .
  {
    request_frequency();                   // Poll for frequency    
  }
  #endif
}

void setup()
{
  // Initialize our microstepping routine
  stepper_Init();

  // Initalize Swticthes as input
  pinMode(EnactSW, INPUT_PULLUP);
  pinMode(UpSW, INPUT_PULLUP);
  pinMode(DnSW, INPUT_PULLUP);

  #if ENDSTOP_OPT == 2                     // End stop sensors implemented
  pinMode(EndStopUpper, INPUT_PULLUP);
  pinMode(EndStopLower, INPUT_PULLUP);
  #endif

  lcd.begin(20, 4);                        // Initialize a 20x4 LCD

  #if !PROTOTYPE
  Uart.begin(USART_BAUD, USART_CONFIG);    // initialize UART serial
  #else
  Uart.begin(USART_BAUD);                  // Teensyduino for Teensy2++ has a
  #endif                                   // more limited Serial library
  
  Serial.begin(9600);                      // initialize USB serial

  if (EEPROM.read(0) != COLDSTART_REF)     // First EEPROM pos is uninitialized,
  {                                        // initialize all memories...
    running.Frq=14000000;                  // 14 MHz as a starting point (irrelevant)
    running.Pos=1000000;                   // Some large round number
    stepper_track=1000000;                 // Some large round number
    delta_Pos = 0;                         // Position offset is 0.
    init_Presets();                        // 0 Hz and 1000000 into all unused presets

    EEPROM.write(0,COLDSTART_REF);         // COLDSTART_REF in first pos indicates all initialized
    EEPROM_writeAnything(1,running);       // write running frequency/pos into eeprom
    EEPROM_writeAnything(9,delta_Pos);     // write running position offset into eeprom
    EEPROM_writeAnything(13,stepper_track);// write initial Stepper Position
    EEPROM_writeAnything(17,preset);       // write initialized empty frq/pos pairs...
  }
  else
  {
    EEPROM_readAnything(1,running);        // EEPROM was not empty, read the current position
    EEPROM_readAnything(9,delta_Pos);      // the current Offset (delta_Pos) and
    EEPROM_readAnything(13,stepper_track); // the current actual Stepper Position
    EEPROM_readAnything(17,preset);        // the Frequency/Position memory presets
  }

  //------------------------------------------
  // LCD Print Version information (5 seconds)
  lcd.clear();
  lcd.setCursor(0,0);
  lcd.print(STARTUPDISPLAY1);
  delay(300);
  sprintf(lcd_buf,STARTUPDISPLAY2);
  lcd.setCursor(20-strlen(lcd_buf),1);
  lcd.print(lcd_buf);
  
  delay(1200);
  lcd.setCursor(0,2);
  lcd.print(STARTUPDISPLAY3);
  lcd.setCursor(0,3);
  lcd.print(STARTUPDISPLAY4);
  
  delay(2000);
  lcd.setCursor(0,2);
  // Now that you have found this line, feel free to change it!!! :)
  lcd.print("TF3LJ / VE2LJX      ");
  lcd.setCursor(0,3);
  lcd.print("        ");
  sprintf(lcd_buf,"Version: %s", VERSION);
  lcd.setCursor(20-strlen(lcd_buf),3);
  lcd.print(lcd_buf);
  delay(1500);
  lcd.clear();
}

