//************************************************************************
//**
//** Project......: Firmware USB AVR Si570 control for the Mobo.
//**
//** Platform.....: ATmega32U2
//**
//** Licence......: This software is freely available for non-commercial 
//**                use - i.e. for research and experimentation only!
//**                Copyright: (c) 2006 by OBJECTIVE DEVELOPMENT Software GmbH
//**                Based on ObDev's AVR USB driver by Christian Starkjohann
//**
//** Description..: Si570 oscillator manipulations
//**
//** History......: This is an ugly quick kludge, replacing the PE0FKO assembler
//**                based functions which do not work for some reason when using
//**                Atmel Studio 6 or higher.
//**
//**
//** Last edit:...: 2017-01-15, Loftur E. Jonasson, TF3LJ / VE2AO
//**
//**************************************************************************

#include "Mobo.h"

static uint32_t	FreqSmoothTune;			// The smooth tune center frequency

static	void		Si570Write(void);
static	void		Si570Load(void);
static	void		Si570FreezeNCO(void);
static	void		Si570UnFreezeNCO(void);
static	void		Si570NewFreq(void);

Si570_t		Si570_Data;							// Si570 register values

static double	rfreq, delta_rfreq, old_rfreq;// Used by Si570 freq calc functions below

// This function is borrowed from the SDR Widget project
uint8_t set_Regs_and_Dividers(uint32_t ff)
{
    double f;

	uint8_t		validDividers = 0;
	double	 	rfreq_fraction;
	uint32_t	rfreq_integer_part;
	uint32_t	rfreq_fraction_part;

	// Registers finding the lowest DCO frequenty - code from Fred
	unsigned char	xHS_DIV;			// HSDIV divider can be 4,5,6,7,9,11
	unsigned int	xN1;				// N1 divider can be 1,2,4,6,8...128
	unsigned int	xN;

	// Registers to save the found dividers
	unsigned char	sHS_DIV=0;
	unsigned char	sN1=0;
	unsigned int	sN=0;				// Total division
	unsigned int	N0;					// Total divider needed (N1 * HS_DIV)

	// Derive f in unpacked form (because this is an ugly kludge and I am lazy)
	f = (double) ff / _2(21);
	
	// Find the total division needed.
	// It is always one too low (not in the case reminder is zero, reminder not used here).

	N0 = (double) DCO_MIN / f;
	sN = 11*128;
	for(xHS_DIV = 11; xHS_DIV > 3; xHS_DIV--)
	{
		// Skip the unavailable divider's
		if (xHS_DIV == 8 || xHS_DIV == 10)
		continue;

		// Calculate the needed low speed divider
		xN1 = N0 / xHS_DIV + 1;

		if (xN1 > 128)
		continue;

		// Skip the unavailable divider's
		if (xN1 != 1 && (xN1 & 1) == 1)
		xN1 += 1;

		xN = xHS_DIV * xN1;
		if (sN > xN)
		{
			sN		= xN;
			sN1		= xN1;
			sHS_DIV	= xHS_DIV;
		}
	}

	if (sHS_DIV == 0) 					// no valid dividers found
	{
		return validDividers;
	}

	rfreq = f * (double) sN;			// DCO freq

	if (rfreq > (double) DCO_MAX)		// calculated DCO freq > max
	{
		return validDividers;
	}

	validDividers = 1;

	// rfreq is a 38 bit number, MSB 10 bits integer portion, and LSB 28 fraction
	// in the Si570 registers, tempBuf[1] has 6 bits, and tempBuf[2] has 4 bits of the integer portion

	rfreq = rfreq / ((double) R.FreqXtal/_2(24));// DCO divided by fcryst
//	rfreq = rfreq / ((double) 114.285000);// DCO divided by fcryst
	rfreq_integer_part = rfreq;
	rfreq_fraction = rfreq - rfreq_integer_part;
	rfreq_fraction_part = rfreq_fraction * (1L << 28);

	sHS_DIV -= 4;
	sN1 -= 1;
	Si570_Data.bData[0] = (sHS_DIV << 5) | (sN1 >> 2);
	Si570_Data.bData[1] = (sN1 & 3) << 6;
	Si570_Data.bData[1] |= ((rfreq_integer_part >> 4) & 0x3f);
	Si570_Data.bData[2] = ((rfreq_integer_part & 0x0f) << 4) | (rfreq_fraction_part >> 24);
	Si570_Data.bData[3] = rfreq_fraction_part >> 16;
	Si570_Data.bData[4] = rfreq_fraction_part >> 8;
	Si570_Data.bData[5] = rfreq_fraction_part;
	return validDividers;
}

/*
// This function is borrowed from the SDR Widget project
uint32_t Freq_From_Register(void)			// side effects: rfreq and delta_rfreq are set
{
	double 	freq_double;
	uint8_t	n1;
	uint8_t	hsdiv;
	uint32_t	rfreq_integer_portion, rfreq_fraction_portion;

	// Now find out the current rfreq and freq
	hsdiv = ((Si570_Data.bData[0] & 0xE0) >> 5) + 4;
	n1 = ((Si570_Data.bData[0] & 0x1f ) << 2 ) + ((Si570_Data.bData[1] & 0xc0 ) >> 6 );

	//	if(n1 == 0) n1 = 1;
	//	else if((n1 & 1) !=0) n1 += 1;
	n1 += 1;

	rfreq_integer_portion = ((uint32_t)(Si570_Data.bData[1] & 0x3f)) << 4 |
	((uint32_t)(Si570_Data.bData[2] & 0xf0)) >> 4;

	rfreq_fraction_portion = ((uint32_t) (Si570_Data.bData[2] & 0x0f)) << 24;
	rfreq_fraction_portion += ((uint32_t)(Si570_Data.bData[3])) << 16;
	rfreq_fraction_portion += ((uint32_t)(Si570_Data.bData[4])) << 8;
	rfreq_fraction_portion += ((uint32_t)(Si570_Data.bData[5]));

	rfreq = (double)rfreq_integer_portion + ((double)rfreq_fraction_portion / (1L << 28));

	if (rfreq >= old_rfreq) delta_rfreq = rfreq - old_rfreq;
	else delta_rfreq = old_rfreq - rfreq;

	freq_double = (double) (R.FreqXtal/_2(24)) * rfreq / (double) hsdiv / (double) n1;
	return (freq_double * _2(24));
}
*/
/////////////////////////////////////////////////////////////////////////////////////////////////////////


static uint8_t Si570_Small_Change(uint32_t current_Frequency)
{
	uint32_t delta_F, delta_F_MAX;
	sint32_t previous_Frequency;

	// Get previous_Frequency   -> [11.21]
	previous_Frequency.dw = FreqSmoothTune;

	// Delta_F (MHz) = |current_Frequency - previous_Frequency|  -> [11.21]
	delta_F = current_Frequency - previous_Frequency.dw;

	if (delta_F >= _2(31)) delta_F = 0 - delta_F;

	// Delta_F (Hz) = (Delta_F (MHz) * 1_000_000) >> 16 not possible, overflow
	// replaced by:
	// Delta_F (Hz) = (Delta_F (MHz) * (1_000_000 >> 16)
	//              = Delta_F (MHz) * 15  (instead of 15.258xxxx)
	// Error        = (15 - 15.258) / 15.258 = 0.0169 < 1.7%

	delta_F = delta_F * 15;          // [27.5] = [11.21] * [16.0]

	// Compute delta_F_MAX (Hz)= previous_Frequency(MHz) * 3500 ppm
	delta_F_MAX = (uint32_t)previous_Frequency.w1.w * R.SmoothTunePPM;
	//   [27.5] =                          [11.5] * [16.0]

	//debug
	#if DEBUG_SMTH_OFFS_2LN			// Debug
	char buf[10];
	lcd_gotoxy(0,1);
	ltoa(delta_F_MAX,buf,10);
	lcd_puts(buf);
	lcd_putc(';');
	ltoa(delta_F,buf,10);
	lcd_puts(buf);
	lcd_putc('.');
	#endif

	#if DEBUG_SMTH_OFFS_1LN			// Debug
	char buf[10];
	lcd_gotoxy(0,0);
	ltoa(delta_F_MAX,buf,10);
	lcd_puts(buf);
	lcd_putc(';');
	ltoa(delta_F,buf,10);
	lcd_puts(buf);
	lcd_putc('.');
	lcd_gotoxy(0,1);
	#endif

	// return TRUE if output changes less than 3500 ppm from the previous_Frequency
	return (delta_F <= delta_F_MAX) ? True : False;
}


void SetFreq(uint32_t freq)		// frequency [MHz] * 2^21
{
	uint8_t status = 0;
	
	#if CALC_BAND_MUL_ADD		// Band dependent Frequency Subtract and Multiply
	static uint8_t band;		// which BPF frequency band?
	#endif

	#if !FRQ_CGH_DURING_TX		// Do not allow Si570 frequency change and corresponding filter change during TX
	if (Status1 & TX_FLAG) 		// Oops, we are transmitting... return without changing frequency
		return;
	#endif

	R.Freq[0] = freq;			// Some Command calls to this func do not update R.Freq[0]

	#if !FLTR_CGH_DURING_TX		// Do not allow Filter changes when frequency is changed during TX
	if (!(Status1 & TX_FLAG))	// Only change filters when not transmitting
	#endif
	#if CALC_BAND_MUL_ADD		// Band dependent Frequency Subtract and Multiply
	band = SetFilter(freq);		// Select Band Pass Filter, according to the frequency selected
	#else
	SetFilter(freq);			// Select Band Pass Filter, according to the frequency selected
	#endif

	#if CALC_FREQ_MUL_ADD		// Frequency Subtract and Multiply Routines (for smart VFO)
								// Modify Si570 frequency according to Mul/Sub values
	freq = CalcFreqMulAdd(freq);
	#endif
	#if CALC_BAND_MUL_ADD		// Band dependent Frequency Subtract and Multiply
								// Modify Si570 frequency according to Mul/Sub values
	freq = CalcFreqMulAdd(freq, R.BandSub[band], R.BandMul[band]);
	#endif
	
	status = set_Regs_and_Dividers(freq);
	
	if (status != 0)			// Manipulate Si570 if successful
	{
		// Smoothtune change frequency
		if ((R.SmoothTunePPM != 0) && Si570_Small_Change(freq) && !(Status2 & SI570_OFFL))
		{
			Si570Write();
		}
		else
		{
			//		if (!Si570CalcDivider(freq) || !Si570CalcRFREQ(freq))
			//return;
			Status2 &= ~SI570_OFFL;
			FreqSmoothTune = freq;
			Si570Load();
		}
	}
}

void
DeviceInit(void)
{
	// Check if Si570 is online and intialize if nessesary
	// SCL Low is now power on the SI570 chip in the Softrock V9
	if ((I2C_PIN & _BV(BIT_SCL)) != 0)
	{
		if (Status2 & SI570_OFFL)
		{
			FreqSmoothTune = 0;				// Next SetFreq call no smoodtune

			SetFreq(R.Freq[0]);

			Status2 &= ~SI570_OFFL;
			//SI570_OffLine = I2CErrors;
			//SI570_OffLine = 0;
		}
	}
	else 
	{
		Status2 |= SI570_OFFL;
		//SI570_OffLine = True;
	}
}

static uint8_t
Si570CmdStart(uint8_t cmd)
{
	I2CSendStart();
	I2CSendByte((R.Si570_I2C_addr<<1)|0);// send device address 
	if (I2CErrors == 0)
	{
		I2CSendByte(cmd);				// send Byte Command
		return True;
	}
	return False;
}

void
Si570CmdReg(uint8_t reg, uint8_t data)
{
	//i2c_queue();						// Wait for I2C port to become free

	if (Si570CmdStart(reg))
	{
		I2CSendByte(data);
	}
	I2CSendStop();

	//i2c_release();						// Release I2C port
}

static void
Si570NewFreq(void)
{
	Si570CmdReg(135, 0x40);
}

static void
Si570FreezeNCO(void)
{
	Si570CmdReg(137, 0x10);
}

static void
Si570UnFreezeNCO(void)
{
	Si570CmdReg(137, 0x00);
}

// write all registers in one block.
static void
Si570Write(void)
{
	//i2c_queue();						// Wait for I2C port to become free

	if (Si570CmdStart(7))				// send Byte address 7
	{
		uint8_t i;
		for (i=0;i<6;i++)				// all 6 registers
			I2CSendByte(Si570_Data.bData[i]);// send data 
	}
	I2CSendStop();

	//i2c_release();						// Release I2C port
}

// read all registers in one block to replyBuf[]
uint8_t GetRegFromSi570(void)
{
	//i2c_queue();						// Wait for I2C port to become free

	if (Si570CmdStart(7))				// send Byte address 7
	{
		uint8_t i;
		I2CSendStart();	
		I2CSendByte((R.Si570_I2C_addr<<1)|1);
		for (i=0; i<5; i++)
		{
			((uint8_t*)replyBuf)[i] = I2CReceiveByte();
			I2CSend0();					// 0 more bytes to follow
		}
		((uint8_t*)replyBuf)[5] = I2CReceiveByte();
		I2CSend1();						// 1 Last byte
	}
	I2CSendStop(); 

	//i2c_release();						// Release I2C port

	return I2CErrors ? 0 : sizeof(Si570_t);
}

static void Si570Load(void)
{
	Si570FreezeNCO();
	if (I2CErrors == 0)
	{
		Si570Write();
		Si570UnFreezeNCO();
		Si570NewFreq();
	}
	//i2c_release();						// Release I2C port
}
