//************************************************************************
//**
//** Project......: USB controller firmware for the Softrock 6.3 SDR,
//**                   enhanced with the 9V1AL Motherboard v 3.6, an
//**                   external LPF bank and other essentials to create
//**                   an all singing and all dancing HF SDR amateur
//**                   radio transceiver
//**
//** Platform.....:    ATmega168 @ 16.0 MHz
//**
//**                   Simple GPIO Poll Type Rotary Encoder function
//**
//** Initial version.: 2009-09-23, Loftur Jonasson, TF3LJ
//**                   Check the main.c file
//**
//**                   Last updates, Variable speed function:
//**                   2010-04-19, Loftur Jonasson, TF3LJ
//**
//************************************************************************


#include "_main.h"

//
// Init Encoder for use
//
void shaftEncoderInit(void)
{
	//
	// Set inputs with pullup
	//
	ENC_A_DDR &= ~ENC_A_PIN;					// Enable pins for input
	ENC_A_PORT |= ENC_A_PIN;					// Activate internal pullup resistors
    ENC_B_DDR &= ~ENC_B_PIN;
	ENC_B_PORT |= ENC_B_PIN;
	ENC_PUSHB_DDR |= ENC_PUSHB_PIN;
	ENC_PUSHB_PORT |= ENC_PUSHB_PIN;
}


//
// Scan the Shaft Encoder
//
void encoder_scan(void)
{
	static uint8_t old_pha, old_phb;			// Variables conaining the previous encoder states

	uint8_t pha = 0, phb = 0;					// Variables containing the current encoder states

	int8_t	increment = 0;						// This interim variable used to add up changes
												// rather than enacting them directly on the
												// uint32_t variable which contains the frequency
												// this saves a bunch of bytes

	#if ENCODER_FAST_ENABLE						// Variable speed Rotary Encoder feature
	uint16_t		enc_time;					// Read and use time from TCNT1
	static uint16_t	enc_last;					// Measure the time elapsed since the last encoder pulse
	static uint8_t	fast_counter=0;				// Number of rapid movements in succession
	static int8_t	direction;					// Direction of last encoder pulse
	#endif

	uint16_t enc_fast_sense = ENC_FAST_SENSE;
	uint16_t enc_fast_trig = ENC_FAST_TRIG;

	uint16_t encoder_resolution;
	encoder_resolution = 8388/R.Resolvable_States;
	enc_fast_sense = 96000/R.Resolvable_States;
	enc_fast_trig = R.Resolvable_States/5;

	if (ENC_A_PORTIN & ENC_A_PIN) pha=1;		// Read Phase A
	if (ENC_B_PORTIN & ENC_B_PIN) phb=1;		// Read Phase B

	if ((pha != old_pha) && (phb != old_phb));	// Both states have changed, invalid
	
	else if (pha != old_pha)					// State of Phase A has changed
	{
		old_pha = pha;							// Store for next iteration

		if(old_pha != old_phb)					// Decide direction and
			increment= 1;						// increment
		else
			increment=-1;						// or decrement
		Status2 |= (ENC_NEWFREQ | ENC_CHANGE);	// Frequency was modified
	}
	
	else if (phb != old_phb)					// State of Phase B has changed
	{
		old_phb = phb;							// Store for next iteration

		if(old_pha != old_phb)					// Decide direction and
			increment=-1;						// decrement
		else
			increment= 1;						// or increment
		Status2 |= (ENC_NEWFREQ | ENC_CHANGE);	// Frequency was modified
	}

	#if ENCODER_FAST_ENABLE						// Feature for variable speed Rotary Encoder
	//
	// Variable speed function
	//
	if (Status2 & ENC_CHANGE)					// If encoder activity
	{
		Status2 &= ~ENC_CHANGE;					// Clear activity flag

		// Measure the time since last encoder activity in units of appr 1/65536 seconds
		enc_time=TCNT1;
		if (enc_last > enc_time); 				// Timer overrun, it is code efficient to do nothing
		else if (enc_fast_sense >= (enc_time-enc_last)) fast_counter++;// Fast movement detected
		else fast_counter = 0;					// Slow movement, reset counter
		enc_last = enc_time;					// Store for next time measurement

		// We have the required number of fast movements, enable FAST mode
		if (fast_counter > enc_fast_trig) Status2 |= ENC_FAST;
		// If direction has changed, force a drop out of FAST mode
		if (direction != increment)
		{
			if (Status2 & ENC_DIR)	Status2 &= ~ENC_DIR;// Previous change was just a one shot event, clear flag
			else Status2 |= ENC_DIR;					// This is the first event in a new direction, set flag
		}
		else if (Status2 & ENC_DIR)				// Second shot with different direction, 
		{										// now we take action and drop out of fast mode

			Status2 = Status2 & ~ENC_FAST & ~ENC_DIR;
		}
		
		direction = increment;					// Save encoder direction for next time

		// When fast mode, multiply the increments by the set MULTIPLY factor
		if (Status2 & ENC_FAST)	increment = increment * ENC_FAST_MULTIPLY;

		// This one could just as well be outside of the parental if sentence
		R.Freq[0] += (int32_t) increment*encoder_resolution;// Add or subtract VFO frequency


		// This one could just as well be outside of the parental if sentence
		#if	ENCODER_DIR_REVERSE
		R.Freq[0] -= (int32_t) increment*encoder_resolution;// Add or subtract VFO frequency
		#else
		R.Freq[0] += (int32_t) increment*encoder_resolution;// Add or subtract VFO frequency
		#endif
	}
	#else
	#if	ENCODER_DIR_REVERSE
	R.Freq[0] -= (int32_t) increment*encoder_resolution;// Add or subtract VFO frequency
	#else
	R.Freq[0] += (int32_t) increment*encoder_resolution;// Add or subtract VFO frequency
	#endif
	#endif

}

