//************************************************************************
//**
//** Project.....: USB controller firmware for the Softrock 6.3 SDR,
//**                   enhanced with the 9V1AL Motherboard v 3.6, an
//**                   external LPF bank and other essentials to create
//**                   an all singing and all dancing HF SDR amateur
//**                   radio transceiver
//**
//** Platform.....:    ATmega168 @ 16.0 MHz
//**
//** Licence......:    This software is freely available for non-commercial 
//**                   use - i.e. for research and experimentation only!
//**
//** Copyright for USB routines: (c) 2006 by OBJECTIVE DEVELOPMENT Software GmbH
//**                   Based on ObDev's AVR USB driver by Christian Starkjohann
//**
//** History &
//** Acknowledgements: This firmware, although bloated up to three times its original
//**                   size to cater for the new functions included, is 100% based on
//**                   source code by F.W. Krom, PE0FKO. See:
//**                   http://home.ict.nl/~fredkrom/pe0fko/SR-V9-Si570/
//**
//**                   The code retains the original command structure
//**                   and is compatible with V 15.12 of the PE0FKO code.
//**                   Thanks Fred!!!
//**
//** 
//**                   Functionality highlights description:
//**
//**                    1) Control of the Si570 Programmable Crystal Oscillator,
//**                       including "smoothtune" for seamless tuning between small
//**					   frequency steps
//**                    2) Variable speed VFO, using a rotary encoder, and 9 short/long term
//**                       memories, using a pushbutton (cycle with short pushes, store with long)
//**					3) Accomodate different Encoder Resolutions (PPT) through a User command
//**                    4) CW Paddle inputs
//**                    5) PTT output
//**                    6) PTT2 output (connected to SWR protect function)
//**                    7) Automatic selection between 4 Bandpass filters,
//**                       with user selectable switchpoints
//**                    8) Automatic selection between 6 Lowpass filters, with user selectable
//**                       switchpoints, controls 3 GPIO output bits.
//**                    9) Measurement of Power forward and Power refleced 
//**                       (uses input from an external Power/SWR bridge)
//**                   10) Hi-SWR protect function, when using an external Power/SWR bridge
//**                       (assert an external signal, for instance to lower the PA output)
//**                   11) PEP indication for power output, shows the highest measured value within a
//**                       user adjustable time window of up to 2 seconds (10 samples per second).
//**                   12) 16x2 LCD display for Frequency readout and Power/SWR metering, using
//**                       a bargraph style display to emulate analog meters.
//**                   13) When switching to TX, Power bargraph display is delayed by 0.5s to enable
//**                       display of frequency change when switching to TX mode, using PowerSDR-IQ   
//**
//**                   Note that some of the items described above, such as the Rotary
//**                   Encoder VFO, the PTT2 output, the Power/SWR bridge and the LCD,
//**                   while not necessary, are optional features provided by the
//**				   firmware.  Further description may be found here:
//**                   http://sites.google.com/site/lofturj/softrock6_3
//**
//**
//** Initial version.: 2009-03-18, Loftur Jonasson, TF3LJ
//**
//**
//** This update:....: 2010-04-19, Version 1.03 & USB Protocol V. 16.2
//**
//*********************************************************************************


// Pin assignment:
// PD0 = USB -Data line
// PD2 = USB +Data line
// PD5 = I2C SDA to Si570
// PD6 = I2C SCL to Si570
// PD3 = CW input 1
// PD4 = CW input 2
// PD1 = PTT out
// PD7 = PTT2 out including SWR protection (inactive if hi SWR)
//
// PB0 = LP filter bit 1
// PB1 = LP filter bit 2
// PB2 = Rotary Encoder, Phase A, for optional Rotary Encoder VFO
// PB3 = Rotary Encoder, Phase B, for optional Rotary Encoder VFO
//       (also ICSP header, used for programming)
// PB4 = Rotary Encoder, Push button, for optional Rotary Encoder VFO
//       (also ICSP header, used for programming)
// PB5 = Debug LED 
//       (also ICSP header, used for programming)
//
// PC0 = TX LP filter bit 1
// PC1 = TX LP filter bit 2
// PC2 = TX LP filter bit 3
// PC3 = ADC input, P/SWR head, Power Out
// PC4 = ADC input, P/SWR head, Power In
// PC5 = ADC input or digital I/O, not used.
//

#include "_main.h"

EEMEM	var_t		E;							// Variables in eeprom
		var_t		R							// Variables in ram
					=							// Variables in flash rom
					{	DEVICE_XTAL				// FreqXtal
					,{	 7.050 * 4.0 * _2(21)	// Default Freq at startup, 
					,	 1.820 * 4.0 * _2(21) 	// Freq Channel 1 
					,	 3.520 * 4.0 * _2(21) 	// Freq Channel 2
					,	 7.020 * 4.0 * _2(21) 	// Freq Channel 3
					,	10.120 * 4.0 * _2(21)	// Freq Channel 4
					,	14.020 * 4.0 * _2(21) 	// Freq Channel 5
					,	18.090 * 4.0 * _2(21) 	// Freq Channel 6
					,	21.020 * 4.0 * _2(21) 	// Freq Channel 7
					,	24.910 * 4.0 * _2(21) 	// Freq Channel 8
					,	28.020 * 4.0 * _2(21)}	// Freq Channel 9
					, 	3						// Which Channel was last in use
					,	0.000 * _2(21)			// Freq subtract value is 0.0MHz (11.21bits)
					,	1.000 * _2(21)			// Freq multiply value os 1.0    (11.21bits)
					,	3500					// SmoothTunePPM
					, {	{  2.0 * 4.0 * _2(5) }	// Default filter cross over
					,	{  8.0 * 4.0 * _2(5) }	// frequencies for softrock V6.3
					,	{ 20.0 * 4.0 * _2(5) }	// BPF. Four value array.
					,	{ true } }				
					, { { 2.0 * 4.0 * _2(5) }	// Default filter crossover
					,	{ 4.0 * 4.0 * _2(5) }	// frequencies for the K5OOR
					,	{ 7.5 * 4.0 * _2(5) }	// HF Superpacker Pro LPF bank
					,	{14.5 * 4.0 * _2(5) }
					,	{21.5 * 4.0 * _2(5) }
					,	{ true } }
					,	{ BP_FLT0				// Band Pass filter selection
					,	  BP_FLT1				// these values are mapped against the result of the
					,	  BP_FLT2				// filter crossover point comparison  
					,	  BP_FLT3 }				// Filter selected by writing value to output port
					,	{ LP_FLT0				// External LPF filter selection
					,	  LP_FLT1				// these values are mapped against the result of the
					,	  LP_FLT2				// filter crossover point comparison 
					,	  LP_FLT3
					,	  LP_FLT4
					,	  LP_FLT5 }
					,	{ 0.000 * _2(21)		// Freq subtract value is 0.0MHz (11.21bits)
					,	  0.000 * _2(21)
					,	  0.000 * _2(21)
					,	  0.000 * _2(21) }
					,	{ 1.000 * _2(21)		// Freq multiply value is 1.0MHz (11.21bits)
					,	  1.000 * _2(21)
					,	  1.000 * _2(21)
					,	  1.000 * _2(21) }
					,	DEVICE_I2C				// I2C address or ChipCrtlData
					,	LCD_I2C_ADDRESS			// Some I2C displays come default with I2C address 0x28
					,	P_MIN_TRIGGER			// Min P out measurement for SWR trigger
					,	SWR_PROTECT_TIMER		// Timer loop value
					,	SWR_TRIGGER				// Max SWR threshold (10 x SWR)
					,	PWR_CALIBRATE			// Power meter calibration value
					,	PWR_FULL_SCALE			// Full Scale setting for Power Output Bargraph, in W
					,	SWR_FULL_SCALE			// Full Scale setting for SWR Bargraph,
					,	20						// Number of PEP measurement samples for LCD power display
					,	ENC_PULSES				// Number of Resolvable States per Revolution
					//,	9						// Freq offset value is a signed value in kHz
					,	0						// Freq offset value is a signed value in kHz
					,	DEF_FEATURES			// Select which firmware features to make active
					,	'0'						// Default USB SerialNumber ID
					};

				Si570_t		Si570_Data;			// Si570 register values
				sint16_t	replyBuf[4];		// USB Reply buffer

				uint16_t	measured_SWR;		// SWR value x 10, in unsigned int format
				uint32_t 	adc_read[2];		// last read P_out and P_in values

static  		uint8_t		bIndex;				// Used to transfer Index from
												// usbFunctionSetup to usbFunctionWrite

volatile uint8_t Status1 = 0;					// Contains various status flags
volatile uint8_t Status2 = 0;					// (broken into two 8bit registers for more compact code size)


EMPTY_INTERRUPT( __vector_default );			// Redirect all unused interrupts to reti

#include "FreqFromSi570.c"						// Include code is small size

int	     usbDescriptorStringSerialNumber[] = {
	USB_STRING_DESCRIPTOR_HEADER(USB_CFG_SERIAL_NUMBER_LEN),
	USB_CFG_SERIAL_NUMBER
};


//
//-----------------------------------------------------------------------------------------
//			Process USB Host to Device transmissions.  No result is returned.
//
//			This function processes control of USB commands 0x30 - 0x35
//-----------------------------------------------------------------------------------------
//

uchar usbFunctionWrite(uchar *data, uchar len)	//sends len bytes to SI570
{
	//usbRequest_t* rq = (usbRequest_t*)data; 	// Appears to contain garbage

	PORTB = PORTB ^ 0x20;						//blink a LED to indicate USB traffic

	SWITCH_START(usbRequest)

	SWITCH_CASE(0x30)
		if (len == 6) {
			CalcFreqFromRegSi570(data);			// Calc the freq from the Si570 register value
			R.Freq[0] = *(uint32_t*)data;
			Status2 |= ENC_NEWFREQ;				// Refresh the active frequency to R.Freq[0]
		}


	SWITCH_CASE(0x31)							// Write the frequency subtract multiply to the eeprom
		if (len == 2*sizeof(uint32_t)) {
			bIndex &= 3;
			memcpy(&R.BandSub[bIndex], &data[0], sizeof(uint32_t));
			eeprom_write_block(&data[0], &E.BandSub[bIndex], sizeof(uint32_t));
			memcpy(&R.BandMul[bIndex], &data[4], sizeof(uint32_t));
			eeprom_write_block(&data[4], &E.BandMul[bIndex], sizeof(uint32_t));
		}


	SWITCH_CASE(0x32)							// Set frequency by value and load Si570
		if (len == 4) {
			if (bIndex < 10)					// Is it a "legal" memory location
				{
					R.Freq[bIndex] = *(uint32_t*)data;
					Status2 |= ENC_NEWFREQ;		// Refresh the active frequency to R.Freq[0]
												// (even if the change was made to a different frq)
				}
		}

	SWITCH_CASE(0x33)							// write new crystal frequency to EEPROM and use it.
		if (len == 4) {
			R.FreqXtal = *(uint32_t*)data;
			eeprom_write_block(data, &E.FreqXtal, sizeof(E.FreqXtal));
			Status2 |= ENC_NEWFREQ;				// Refresh the active frequency to R.Freq[0]
		}

	SWITCH_CASE(0x34)							// Write new startup frequency to eeprom
		if (len == 4) {
			if (bIndex < 10)					// Is it a "legal" memory location
			{
				eeprom_write_block(data, &E.Freq[bIndex], sizeof(E.Freq[0]));
				eeprom_write_block(&bIndex, &E.SwitchFreq, sizeof(E.SwitchFreq));
			}
			eeprom_write_block(data, &E.Freq[0], sizeof(E.Freq[0]));
		}

	SWITCH_CASE(0x35)							// Write new smooth tune to eeprom and use it.
		if (len == 2) {
			R.SmoothTunePPM = *(uint16_t*)data;
			eeprom_write_block(data, &E.SmoothTunePPM, sizeof(E.SmoothTunePPM));
		}

/*
	SWITCH_CASE(0x36)// Modify Rotary Encoder Increment Resolution
					// Frequency increments in units of ~0.12 Hz (or 8 ~ 1 Hz)
					// Also note that if this value is set too high,
					// or more than 3500ppm of the operating frequency, then
					// "smoothtune does not work"
		if (len == 4) {
			R.Encoder_Resolution = *(uint32_t*)data;
			eeprom_write_block(data, &E.Encoder_Resolution, sizeof(E.Encoder_Resolution));
		}
*/
	SWITCH_END

	return 1;
}

usbMsgLen_t 


//
//-----------------------------------------------------------------------------------------
// 			Process USB query commands and return a result (flexible size data payload)
//
//			This function processes control of all USB commands except for 0x30 - 0x35
//-----------------------------------------------------------------------------------------
//
usbFunctionSetup(uchar data[8])
{
	uint8_t	i;

	usbRequest_t* rq = (usbRequest_t*)data;
	usbRequest = rq->bRequest;

    usbMsgPtr = (uchar*)replyBuf;
	replyBuf[0].b0 = 0xff;						// return value 0xff => command not supported 

	PORTB = PORTB ^ 0x20;						//blink a LED to indicate USB traffic

	SWITCH_START(usbRequest)

	SWITCH_CASE(0x00)							// Return software version number
		replyBuf[0].w = (VERSION_MAJOR<<8)|(VERSION_MINOR);
		return sizeof(uint16_t);

//	SWITCH_CASE(0x00)							// ECHO value
//		replyBuf[0].w = rq->wValue.word;		// rq->bRequest identical data[1]!
//		return sizeof(uint16_t);

	#if  INCLUDE_NOT_USED
	SWITCH_CASE(0x01)							// set port directions
		IO_DDRB = data[2];

	SWITCH_CASE(0x02)							// read ports (pe0fko changed)
		replyBuf[0].b0 = IO_PIN;
		return sizeof(uint8_t);

	SWITCH_CASE(0x03)							// read port states 
		replyBuf[0].b0 = IO_PORT;
		return sizeof(uint8_t);

	SWITCH_CASE(0x04)							// set ports 
		IO_PORT = data[2];
		return 0;
	#endif//INCLUDE_NOT_USED

	SWITCH_CASE(0x0F)							// Watchdog reset
		while(true) ;

	SWITCH_CASE(0x15)							// Set IO port with mask and data bytes
		// SoftRock V9 only had 2 I/O pins from tiny45 available.
		uint8_t msk,dat;		
		msk = (rq->wValue.bytes[0] << IO_BIT_START) & (IO_BIT_MASK << IO_BIT_START);
		dat = (rq->wIndex.bytes[0] << IO_BIT_START) & (IO_BIT_MASK << IO_BIT_START);
		IO_DDR  = (IO_DDR & ~(IO_BIT_MASK << IO_BIT_START)) | msk;
		IO_PORT = (IO_PORT & ~msk) | dat;
		
		// Return I/O pin's
		replyBuf[0].w = (IO_PIN>>IO_BIT_START) & IO_BIT_MASK;
        return sizeof(uint16_t);

	SWITCH_CASE(0x16)							// Read I/O bits
		replyBuf[0].w = (IO_PIN>>IO_BIT_START) & IO_BIT_MASK;
        return sizeof(uint16_t);

	SWITCH_CASE(0x17)							// Read and Write the Filter Cross over point's and use it.

		if (rq->wIndex.word < 0x100)
		{
			// RX Filter cross over point table.
			if (rq ->wIndex.bytes[0] < 4)		// Make sure we don't overwrite other parts of table
			{
				R.FilterCrossOver[rq->wIndex.word].w = rq->wValue.word;

				eeprom_write_block(&R.FilterCrossOver[rq->wIndex.word].w, 
					&E.FilterCrossOver[rq->wIndex.word].w, 
					sizeof(E.FilterCrossOver[0].w));
			}
			usbMsgPtr = (uint8_t*)&R.FilterCrossOver;
			return 4 * sizeof(uint16_t);
		}
		else {
			// TX Filter cross over point table.
			rq->wIndex.word -= 0x100;

			if (rq ->wIndex.bytes[0] < 6)		// Make sure we don't overwrite other parts of table
			{
				R.TXFilterCrossOver[rq->wIndex.word].w = rq->wValue.word;

				eeprom_write_block(&R.TXFilterCrossOver[rq->wIndex.word].w, 
					&E.TXFilterCrossOver[rq->wIndex.word].w, 
					sizeof(E.TXFilterCrossOver[0].w));
			}
			usbMsgPtr = (uint8_t*)&R.TXFilterCrossOver;
			return 6 * sizeof(uint16_t);
		}

	SWITCH_CASE(0x18)								// Set the Band Pass Filter Address for one band: 0,1,2...7
			eeprom_write_block(&rq->wValue.bytes[0], &E.FilterNumber[rq->wIndex.bytes[0] & 0x03], sizeof (uint8_t));
			R.FilterNumber[rq->wIndex.bytes[0] & 0x03] = rq->wValue.bytes[0];
			// passthrough to case 0x19

	SWITCH_CASE(0x19)								// Read the Band Pass Filter Addresses for bands 0,1,2...7
			usbMsgPtr = (uint8_t*)R.FilterNumber; 
			return sizeof(R.FilterNumber);


	SWITCH_CASE(0x1a)								// Set the Low Pass Filter Address for one band: 0,1,2...15
			eeprom_write_block(&rq->wValue.bytes[0], &E.TXFilterNumber[rq->wIndex.bytes[0] & 0x07], sizeof (uint8_t));
			R.TXFilterNumber[rq->wIndex.bytes[0] & 0x03] = rq->wValue.bytes[0];


	SWITCH_CASE(0x1b)								// Read the Low Pass Filter Addresses for bands 0,1,2...15
			usbMsgPtr = (uint8_t*)R.TXFilterNumber;
			return sizeof(R.TXFilterNumber);


	SWITCH_CASE(0x20)							// [DEBUG] Write byte to Si570 register
		Si570CmdReg(rq->wValue.bytes[1], rq->wIndex.bytes[0]);
		FreqSmoothTune = 0;						// Next SetFreq call no smoodtune
		replyBuf[0].b0 = I2CErrors;				// return I2C transmission error status
        return sizeof(uint8_t);

	SWITCH_CASE7(0x30,0x31,0x32,0x33,0x34,0x35,0x36)
		//	0x30						      	// Set frequnecy by register and load Si570
		//	0x31								// Write the FREQ mul & add to the eeprom
		//	0x32								// Set frequency by value and load Si570
		//	0x33								// write new crystal frequency to EEPROM and use it.
		//	0x34								// Write new startup frequency to eeprom
		//	0x35								// Write new smooth tune to eeprom and use it.
        //	0x36								// Modify Rotary Encoder Increment Resolution
		bIndex = rq->wIndex.bytes[0];
		return USB_NO_MSG;						// use usbFunctionWrite to transfer data

	SWITCH_CASE(0x39)							// Return the frequency subtract multiply
		i = rq->wIndex.bytes[0] & 0x03;
		memcpy(&replyBuf[0].w, &E.BandSub[i], sizeof(uint32_t));
		memcpy(&replyBuf[4].w, &E.BandMul[i], sizeof(uint32_t));
       	return 2 * sizeof(uint32_t);

	SWITCH_CASE(0x3A)							// Return running frequnecy
		usbMsgPtr = (uint8_t*)&R.Freq[rq->wIndex.bytes[0]];
        return sizeof(uint32_t);

	SWITCH_CASE(0x3B)							// Return smooth tune ppm value
		usbMsgPtr = (uint8_t*)&R.SmoothTunePPM;
        return sizeof(uint16_t);

	SWITCH_CASE(0x3C)							// Return the startup frequency
		eeprom_read_block(replyBuf, &E.Freq[rq->wIndex.bytes[0]], sizeof(E.Freq[rq->wIndex.bytes[0]]));
		return sizeof(uint32_t);

	SWITCH_CASE(0x3D)							// Return the XTal frequnecy
		usbMsgPtr = (uint8_t*)&R.FreqXtal;
        return sizeof(uint32_t);

//	SWITCH_CASE(0x3E)							// read out calculated frequency control registers
//		usbMsgPtr = (uint8_t*)&Si570_Data;
//		return sizeof(Si570_t);

	SWITCH_CASE(0x3F)							// read out chip frequency control registers
		return GetRegFromSi570();				// read all registers in one block to replyBuf[]

	SWITCH_CASE(0x40)							// return I2C transmission error status
		replyBuf[0].b0 = I2CErrors;
		return sizeof(uint8_t);

	SWITCH_CASE(0x41)							// Set the new i2c address or factory default (pe0fko: function changed)
						// Set a new i2c address for a device, or reset the EEPROM to factory default
						// if Value contains 0xff, then factory defaults will be loaded on reset.
						
						// Index 0 reads/modifies the Si570_I2C_addr
						// Index 7 reads/modifies the I2C address for the I2C connected LCD display

			if (rq->wValue.bytes[0] == 0xff)
        	{
				// Force an EEPROM update:
				// eeprom_write_block appears to take (14 bytes) less pgm space than eeprom_write_byte 
				eeprom_write_block(&rq->wValue.bytes[0], &E.ChipCrtlData, sizeof (uint8_t));
				while(true) ;					// Reboot by watchdog timer
			}
			
			i = rq->wIndex.bytes[0];			// This substitution makes the code smaller

			if (rq->wValue.bytes[0])			// If value field > 0, then update EEPROM settings
			{
				replyBuf[0].b0 = rq->wValue.bytes[0];

				switch (i) 
				{
					case 0:
						eeprom_write_block(&rq->wValue.bytes[0], &E.ChipCrtlData, sizeof (uint8_t));
						break;
					case 7:	
						eeprom_write_block(&rq->wValue.bytes[0], &E.LCD_I2C_addr, sizeof (uint8_t));
						break;
				}
			}
			else								// Else just read and return the current value
			{
				switch (i) 
				{
					case 0:	
						replyBuf[0].b0 = R.ChipCrtlData;
						break;
					case 15:
						replyBuf[0].b0 = R.LCD_I2C_addr;
						break;
				}
			}
			return sizeof(uint8_t);


	SWITCH_CASE(0x43)							// Get/Set the USB SeialNumber ID
		replyBuf[0].b0 = R.SerialNumber;
		if (rq->wValue.bytes[0] != 0) {			// Only set if Value != 0
			R.SerialNumber = rq->wValue.bytes[0];
			eeprom_write_byte(&E.SerialNumber, R.SerialNumber);
		}
		return sizeof(R.SerialNumber);


	SWITCH_CASE(0x50)  //set RXTX and get cw-key status
		if (rq->wValue.bytes[0] == 0)
		{
			// Clear PTT flag
			Status1 = Status1 & ~TX_FLAG;
			Status1 = Status1 | LCD_UPDATE;		// Force LCD write
			PORTD = PORTD & ~RXTX;
		}
		else 
		{
			Status1 = Status1 | TX_FLAG;		// Set the TX flag
			PORTD = PORTD | RXTX;
		}
		replyBuf[0].b0 = 0x00;
		if (PIND & CWKEY1) replyBuf[0].b0 |= CWSHORT;	// read pin and set regbit accordingly
		if (PIND & CWKEY2) replyBuf[0].b0 |= CWLONG;	// read pin and set regbit accordingly
        return sizeof(uint8_t);


	SWITCH_CASE(0x51)									// read CW key levels
		replyBuf[0].b0 = 0x00;
		if (PIND & CWKEY1) replyBuf[0].b0 |= CWSHORT;	// read pin and set regbit accordingly
		if (PIND & CWKEY2) replyBuf[0].b0 |= CWLONG;	// read pin and set regbit accordingly
        return sizeof(uint8_t);


	SWITCH_CASE(0x60)// Features select
		if (rq->wValue.word)					// If value field > 0, then update EEPROM settings
		{
			// Write to Features Selecton Register
			eeprom_write_block(&rq->wValue.bytes[0], &E.Features, sizeof (E.Features));
			replyBuf[0].b0 = R.Features = rq->wValue.bytes[0];
		}
		else
			replyBuf[0].b0 = R.Features;
		return sizeof(uint8_t);			


	SWITCH_CASE(0x61)// Read ADC inputs,
					// Index byte points to which ADC input to read.
					// This command is tailored towards the Mobo 4.3.
					// This implementation does only have The Vfwd and Vref
											
					// Index 0 = nil {PA current, full scale 0xfff0 = 2.5A}
					// Index 1 = Power Measurement Voltage Output
					// Index 2 = Power Measurement Voltage reflected
					// Index 3 = nil {Supply voltage, full scale 0xfff0 = 15.64V}
					// Index 4 = nil {Temperature in degC.Signed Int.  0 = 0 deg C
					// 			 32640 =  128 deg C, 32768 = -128 deg C}
					// NEW!! Index 5 = SWR

		if (rq->wIndex.bytes[0] > 0 && rq->wIndex.bytes[0] < 3)	// Power or Reflected
		{
			replyBuf[0].w = ReadADC(rq->wIndex.bytes[0]-1);
		}				
		
		else if (rq->wIndex.bytes[0] == 5)
		{
			replyBuf[0].w = ReadADC(3);							// SWR
		}

		else									// return zero {Read current temperature}
		{
			replyBuf[0].w = 0;
		}
		return sizeof(uint16_t);

	SWITCH_CASE(0x66)// Read/Modify four settings (all 16 bit values).
					// If Value = 0 then read, else modify and read back:

					// 	Index 0: Min P out measurement for SWR trigger 
					// 	Index 1: SWR Timer expiry value (in units of 10ms, can be set awfully long:)
					// 	Index 2: Max SWR threshold in units of 10x SWR (SWR of 2.7 = 27)
					//	Index 3: Power meter calibration value
					//	Index 4: Power Meter bargraph Fullscale in W
					//	Index 5: SWR Meter bargraph Fullscale in SWR - 1 (if enabled by #define)
					//  Index 6: Number of PEP measurement samples for LCD power display (1-20)

		i = rq->wIndex.bytes[0]; 				// This substitution makes the code smaller
			
		if (rq->wValue.word)					// If value field > 0, then update EEPROM settings
		{
			switch (i) 
			{
				case 0:							// Min P out measurement for SWR trigger
					eeprom_write_block(&rq->wValue.word, &E.P_Min_Trigger, sizeof (E.P_Min_Trigger));
					replyBuf[0].w = R.P_Min_Trigger = rq->wValue.word;
					break;
	
				case 1:							// Timer loop value
					eeprom_write_block(&rq->wValue.word, &E.SWR_Protect_Timer, sizeof (E.SWR_Protect_Timer));
					replyBuf[0].w = R.SWR_Protect_Timer = rq->wValue.word;
					break;
				case 2:							// Max SWR threshold
					eeprom_write_block(&rq->wValue.word, &E.SWR_Trigger, sizeof (E.SWR_Trigger));
					replyBuf[0].w = R.SWR_Trigger = rq->wValue.word;
					break;
				case 3:							// Max SWR threshold
					eeprom_write_block(&rq->wValue.word, &E.PWR_Calibrate, sizeof (E.PWR_Calibrate));
					replyBuf[0].w = R.PWR_Calibrate = rq->wValue.word;
					break;
				case 4:							// Fullscale Power Bargraph value
					eeprom_write_block(&rq->wValue.bytes[0], &E.PWR_fullscale, sizeof (E.PWR_fullscale));
					replyBuf[0].b0 = R.PWR_fullscale = rq->wValue.bytes[0];
					break;
				case 5:							// Fullscale SWR Bargraph value
					eeprom_write_block(&rq->wValue.bytes[0], &E.PWR_fullscale, sizeof (E.PWR_fullscale));
					replyBuf[0].b0 = R.SWR_fullscale = rq->wValue.bytes[0];
					break;
				case 6:							// Number of samples in PEP measurement
					eeprom_write_block(&rq->wValue.bytes[0], &E.PEP_samples, sizeof (E.PEP_samples));
					replyBuf[0].b0 = R.PEP_samples = rq->wValue.bytes[0];
					break;
			}
		}

		else									// Else just read and return the current value
		{
			switch (i) 
			{
				case 0:							// Min P out measurement for SWR trigger
					replyBuf[0].w = R.P_Min_Trigger;
					break;
				case 1:							// Timer loop value
					replyBuf[0].w = R.SWR_Protect_Timer;
					break;
				case 2:							// Max SWR threshold
					replyBuf[0].w = R.SWR_Trigger;
					break;
				case 3:							// Max SWR threshold
					replyBuf[0].w = R.PWR_Calibrate;
					break;
				case 4:							// Fullscale Power Bargraph value
					replyBuf[0].b0 = R.PWR_fullscale;
					break;
				case 5:							// Fullscale SWR Bargraph value
					replyBuf[0].b0 = R.SWR_fullscale;
					break;
				case 6:							// Number of samples in PEP measurement
					replyBuf[0].b0 = R.PEP_samples;
					break;
			}
		}
		return sizeof(uint16_t);

/*	SWITCH_CASE(0x67)// Read/Modify Rotary Encoder Increment Resolution Setting
					// Frequency increments in units of ~0.12 Hz (or 8 ~ 1 Hz)
					// Max value is 0x7fff.  If set higher, then the routine breaks.
					// Also note at lower frequencies that if this value is set too high,
					// or more than 3500ppm of the operating frequency, then
					// "smoothtune does not work"

		if (rq->wValue.word)
		{		// New value
				eeprom_write_block(&rq->wValue.word, &E.Encoder_Resolution, sizeof (uint16_t));
				R.Encoder_Resolution = rq->wValue.word;
		}
		// Return current value
		replyBuf[0].w = R.Encoder_Resolution;
		return sizeof(uint16_t);
*/

	SWITCH_CASE(0x67)				// Read/Modify Rotary Encoder Increment Resolution Setting
									// Normally set to the number of resolvable states per revolution
			if (rq->wValue.word)
			{		// New value
					eeprom_write_block(&rq->wValue.word, &E.Resolvable_States, sizeof (uint16_t));
					R.Resolvable_States = rq->wValue.word;
			}
			// Return current value
			replyBuf[0].w = R.Resolvable_States;
			return sizeof(uint16_t);


	SWITCH_CASE(0x68)				// Display a fixed frequency offset during RX only.
			if (rq->wIndex.word)	// If Index>0, then New value contained in Value
			{
				eeprom_write_block(&rq->wValue.bytes[0], &E.LCD_RX_Offset, sizeof (uint8_t));
				R.LCD_RX_Offset = rq->wValue.bytes[0];// used frequency, when using PowerSDR-IQ
				Status2 |= ENC_NEWFREQ;				// Refresh the active frequency to R.Freq[0]
			}
			// Return current value
			replyBuf[0].b0 = R.LCD_RX_Offset;
			return sizeof(uint8_t);
	

	SWITCH_CASE(0x7F)	// LCD Commands
		lcd_command(rq->wIndex.bytes[0]);
		lcd_data(rq->wValue.bytes[0]);
		return 0;

	
	SWITCH_END

    return 1;
}


// This function is neded, otherwise the USB device is not
// recognized after a reboot.
// The watchdog will need to be reset (<16ms). Fast (div 2K) prescaler after watchdog reset!
// MCUSR must be cleared (datasheet) it is not done within the wdt_disable().
void __attribute__((naked))
     __attribute__((section(".init3")))
dotInit3(void)
{
	MCUSR = 0;
	wdt_disable();

	IO_DDR = _BV(IO_P1);		// All port pins inputs except IO_P1 switching output
	IO_PORT = 0;				// Inp on startup, no pullups
}



//
// ----------------------------------------main-----------------------------------------
//
//--------------------------------------------------------------------------------------
// 			Setup Ports, timers, start the works and loop forever, unless reset
//								by the watchdog timer
//						then - do everything, all over again
//--------------------------------------------------------------------------------------
//
//int __attribute__((noreturn)) 
int
main(void)
{
	static uint8_t pushcount=0;						// Used to time a push button (max 2.5s)
	static uint16_t lastIteration1, lastIteration2;	// Counters to keep track of time
	uint16_t Timer1val, Timer1val2;					// Timers used for 100ms and 10ms polls

	// 16-bit Timer1 Initialization
	TCCR1A = 0; //start the timer
	TCCR1B = (1 << CS12); // prescale Timer1 by CLK/256
	// 16000000 Hz / 256 = 62500 ticks per second
	// 16-bit = 2^16 = 65536 maximum ticks for Timer1
	// 65536 / 62500 = ~1.05 seconds
	// so Timer1 will overflow back to 0 about every 1 seconds

	// Check if eeprom is initialized, use only the field ChipCrtlData.
	if (eeprom_read_byte(&E.ChipCrtlData) == 0xFF)
		eeprom_write_block(&R, &E, sizeof(E));	// Initialize eeprom to "factory defaults".
	else
		eeprom_read_block(&R, &E, sizeof(E));	// Load the persistend data from eeprom.

	// Modify the last byte of the USB descriptor Serial number ("TF3LJ-1.X")
	usbDescriptorStringSerialNumber[
		sizeof(usbDescriptorStringSerialNumber)/sizeof(int)-1] = R.SerialNumber;

	DDRD = RXTX | RXTX2;						// all portd pins input except for the PTT pins
	PORTD = (CWKEY1 | CWKEY2);  				// RX on startup, pullups for CW key

	DDRB = 0x20;								// PB5 for output, prepare to blink a LED

	if(R.Features & ROTARY_ENC)
		shaftEncoderInit();						// Init Rotary Encoder

	InitADC();									// Init the AD connverters

	SI570_OffLine = true;						// Si570 is offline, not initialized

	DeviceInit();								// Initialize the Si570 device.

	usbDeviceDisconnect();						// Start USB enumeration
	_delay_ms(500);
	usbDeviceConnect();

	usbInit();									// Init the USB used ports

	if(R.Features & LCD_DISPLAY)
	{
		lcd_bargraph_init();					// load the first 8 custom characters
		lcd_display();							// A first write to the LCD
	}

	sei();										// Enable interupts, not that we are using any :)
	wdt_enable(WDTO_250MS);						// Watchdog 250ms

	//
	//-------------------------------------------------------------------------------
	// Loop over and over again, and perform whatever tasks that come our way
	//-------------------------------------------------------------------------------
	//
	while(true)
	{
		//-------------------------------------------------------------------------------
		// Here we do routines which are to be run through as often as possible
		// currently measured to be approximately once every 30 us (1/33333 s)
		//-------------------------------------------------------------------------------
	    wdt_reset();							// Reset the watchdog

	    usbPoll();								// Poll for USB input

		if(R.Features & ROTARY_ENC)
			encoder_scan();						// Scan the Rotary Encoder

		//-------------------------------------------------------------------------------
		// Here we do routines which are to be accessed once every 1/10th of a second
		// We have a free running timer which matures once every ~1.05 seconds
		//-------------------------------------------------------------------------------
		Timer1val = TCNT1/6554; 				// get current Timer1 value, changeable every ~1/10th sec
		if (Timer1val != lastIteration1)		// Once every 1/10th of a second, do stuff
		{
			lastIteration1 = Timer1val;			// Make ready for next iteration
	
			PORTB = PORTB ^ 0x20;				// Blink a LED

			if(R.Features & ROTARY_ENC)
			{
				#if ENCODER_FAST_ENABLE							// Variable speed Rotary Encoder feature
				//
				// Encoder activity watchdog, if Fast mode is active
				//
				static int fast_patience;						// Patience timer

				if (Status2 & ENC_FAST)							// Is fast mode active?
				{
					if (Status2 & ENC_NEWFREQ)					// Encoder activity, reset timer
						fast_patience=0;
					else										// No activity, increase timer
						fast_patience++;
					
					if(fast_patience>=ENC_FAST_PATIENCE)		// No activity for a long time, revert to normal mode
					{
						Status2 &= ~ENC_FAST;
						fast_patience = 0;
					}
				}
				#endif
				//
				// Read Pushbutton state from Shaft encoder and manage Frequency band memories
				//
				if (pushcount >= ENC_PUSHB_MAX)	// "Long Push", store settings
				{
					eeprom_write_block(&R.Freq[0], &E.Freq[R.SwitchFreq], sizeof(R.Freq[0]));
					eeprom_write_block(&R.SwitchFreq, &E.SwitchFreq, sizeof (uint8_t));
					// // Maybe a bit redundant: Store in memory location 0:
					//eeprom_write_block(&R.Freq[0], &E.Freq[0], sizeof(R.Freq[0]));
					Status2 = Status2 | ENC_NEWFREQ | ENC_STORED;	// We have a new frequency stored.
												// NEWFREQ signals a frq update
												// STORED signals an LCD message
				}
				else if (ENC_PUSHB_INPORT & ENC_PUSHB_PIN) 	// Pin high = just released, or not pushed
				{
					if (pushcount >= ENC_PUSHB_MIN)			// Release after a "Short push"
					{	
						R.SwitchFreq++;						// rotate through memories
						if (R.SwitchFreq > 9) R.SwitchFreq = 1;
						R.Freq[0] = R.Freq[R.SwitchFreq];	// Fetch last stored frequency in next band
						Status2 |= ENC_NEWFREQ;				// Signal a new frequency to be written
															// to the Si570 device
					}
					else
					{										// No push or a very short push, do nothing
						pushcount = 0;						// Initialize push counter for next time
					}
				}
				else if (!(Status2 & ENC_STORED))			// Button Pushed, count up the push timer
				{											// (unless this is tail end of a long push,
					pushcount++;							//  then do nothing)
				}
			}
		}

		//-------------------------------------------------------------------------------
		// Here we do routines which are to be accessed once every 1/100th of a second (10ms)
		// We have a free running timer which matures once every ~1.05 seconds
		//-------------------------------------------------------------------------------
		Timer1val2 = TCNT1/656; // get current Timer1 value, changeable every ~1/100th sec
		if (Timer1val2 != lastIteration2)					// Once every 1/100th of a second, do stuff
		{
			lastIteration2 = Timer1val2;					// Make ready for next iteration
	
			//
			// Enact (write) frequency changes resulting from USB input or Rotary Encoder or
			// from the pushbutton memory management routine above
			//
			if (Status2 & ENC_NEWFREQ)						// VFO was turned or freq updated above
			{
				R.Freq[R.SwitchFreq] = R.Freq[0];			// Keep track, move into short term memory
				SetFreq(R.Freq[0]);							// Write the new frequency to Si570
				Status2 &= ~ENC_NEWFREQ;					// and clear flag
				Status1 |= LCD_UPDATE;						// Signal an LCD frequency update
				pushcount = 0;								// Clear the push counter for next time
			}

			pollADC();	// Check the ADC ports during every pass, read if ready, calculate SWR
						// and set/reset PTT2 pin according to SWR
			
			Test_SWR();	// Measure SWR and take action if too high

			if(R.Features & LCD_DISPLAY)
			{
				//
				// Print to LCD Display
				//
				//if((Status1 & TX_FLAG) || (Status1 & LCD_UPDATE) || (Status1 & ENC_STORED))
				if(Status1 & (TX_FLAG | LCD_UPDATE) || (Status2 & ENC_STORED))
				{
					lcd_display();				// Update display if new frequency or regularly during TX
					Status1 &= ~LCD_UPDATE;		// Clear LCD_UPDATE flag
				}
			}
		}
	}
}



/* Version 1.00-16.1: 2009-09-26 - WinAVR20080430
Device: atmega168

Program:   13902 bytes (84.9% Full)
(.text + .data + .bootloader)

Data:        636 bytes (62.1% Full)
(.data + .bss + .noinit)

EEPROM:      133 bytes (26.0% Full)
(.eeprom)
*/

/* Version 1.01-16.1: 2009-09-28 - WinAVR20080430
Device: atmega168

Program:   14086 bytes (86.0% Full)
(.text + .data + .bootloader)

Data:        652 bytes (63.7% Full)
(.data + .bss + .noinit)

EEPROM:      133 bytes (26.0% Full)
(.eeprom)
*/

/* Version 1.02-16.1: 2009-09-29 - WinAVR20080430
Device: atmega168

Program:   14302 bytes (87.3% Full)
(.text + .data + .bootloader)

Data:        658 bytes (64.3% Full)
(.data + .bss + .noinit)

EEPROM:      133 bytes (26.0% Full)
(.eeprom)
*/

/* Version 1.02a-16.1: 2009-09-29 - WinAVR20080430
Device: atmega168

Program:   14372 bytes (87.7% Full)
(.text + .data + .bootloader)

Data:        659 bytes (64.4% Full)
(.data + .bss + .noinit)

EEPROM:      133 bytes (26.0% Full)
(.eeprom)

*/
