//************************************************************************
//**
//** Project.....: USB controller firmware for the Softrock 6.3 SDR,
//**                   enhanced with the 9V1AL Motherboard v 3.6, an
//**                   external LPF bank and other essentials to create
//**                   an all singing and all dancing HF SDR amateur
//**                   radio transceiver
//**
//** Platform.....:    ATmega168 @ 16.0 MHz
//**
//** Licence......:    This software is freely available for non-commercial 
//**                   use - i.e. for research and experimentation only!
//**
//** Copyright for USB routines: (c) 2006 by OBJECTIVE DEVELOPMENT Software GmbH
//**                   Based on ObDev's AVR USB driver by Christian Starkjohann
//**
//** History &
//** Acknowledgements: This firmware, although bloated up to three times its original
//**                   size to cater for the new functions included, is 100% based on
//**                   source code by F.W. Krom, PE0FKO. See:
//**                   http://home.ict.nl/~fredkrom/pe0fko/SR-V9-Si570/
//**
//**                   The code retains the original command structure
//**                   and is compatible with V 15.12 of the PE0FKO code.
//**                   Thanks Fred!!!
//**
//** 
//**                   Functionality highlights description:
//**
//**                    1) Control of the Si570 Programmable Crystal Oscillator,
//**                       including "smoothtune" for seamless tuning between small
//**					   frequency steps
//**                    2) Variable speed VFO, using a rotary encoder, and 9 short/long term
//**                       memories, using a pushbutton (cycle with short pushes, store with long)
//**					3) Accomodate different Encoder Resolutions (PPT) through a User command
//**                    4) CW Paddle inputs
//**                    5) PTT output
//**                    6) PTT2 output (connected to SWR protect function)
//**                    7) Automatic selection between 4 Bandpass filters,
//**                       with user selectable switchpoints
//**                    8) Automatic selection between 6 Lowpass filters, with user selectable
//**                       switchpoints, controls 3 GPIO output bits.
//**                    9) Measurement of Power forward and Power refleced 
//**                       (uses input from an external Power/SWR bridge)
//**                   10) Hi-SWR protect function, when using an external Power/SWR bridge
//**                       (assert an external signal, for instance to lower the PA output)
//**                   11) PEP indication for power output, shows the highest measured value within a
//**                       user adjustable time window of up to 2 seconds (10 samples per second).
//**                   12) 16x2 LCD display for Frequency readout and Power/SWR metering, using
//**                       a bargraph style display to emulate analog meters.
//**                   13) When switching to TX, Power bargraph display is delayed by 0.5s to enable
//**                       display of frequency change when switching to TX mode, using PowerSDR-IQ   
//**
//**                   Note that some of the items described above, such as the Rotary
//**                   Encoder VFO, the PTT2 output, the Power/SWR bridge and the LCD,
//**                   while not necessary, are optional features provided by the
//**				   firmware.  Further description may be found here:
//**                   http://sites.google.com/site/lofturj/softrock6_3
//**
//**
//** Initial version.: 2009-03-18, Loftur Jonasson, TF3LJ
//**
//**
//** This update:....: 2010-04-19, Version 1.02a & USB Protocol V. 16.2
//**
//*********************************************************************************


#ifndef _MAIN_H_
#define _MAIN_H_ 1

//*****This has been deleted in 15.10  #include <stdbool.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include "usbdrv.h"


#define	VERSION_MAJOR	16
#define	VERSION_MINOR	2


// A few switches to select/deselect features or code segments
#define	INCLUDE_NOT_USED	1				// Compatibility old firmware, USB commands 0x01 - 0x04
#define BARGRAPH_STYLE_1	1				// N8LP LP-100 look alike bargraph
#define ENCODER_FAST_ENABLE	1				// Variable speed Rotary Encoder feature
#define	ENCODER_DIR_REVERSE	0				// Reverse the Encoder Direction (Cost 0, simply reverses direction)
#define TF3LJ_SPECIFIC		0				// Turns on TF3LJ specificly tailored settings


// GPIO defines
#define IO_DDR			DDRB
#define IO_PORT			PORTB
#define IO_PIN			PINB
#define IO_PINB			PINB
#define IO_PIND			PIND
#define IO_DDRB			DDRB
#define IO_PORTB		PORTB
#define IO_DDRD			DDRD
#define IO_PORTD		PORTD
#define	IO_DDRC			DDRC
#define	IO_PORTC		PORTC
#define	IO_BIT_START	PB0					// First I/O line used
#define	IO_BIT_LENGTH	2					// Using 2 I/O line (PB0, PB1)

#define	IO_BIT_MASK		( (1<<IO_BIT_LENGTH)-1 )
#define IO_P1			( IO_BIT_START+0 )
#define IO_P2			( IO_BIT_START+1 )
#define IO_P3			( IO_BIT_START+2 )

#define USERP1 (1 << 1)
#define USERP3 (1 << 3)
#define USERP4 (1 << 4)
#define USERP7 (1 << 7)
#define RXTX USERP1
#define RXTX2 USERP7
// Bits used for CW in reg
#define CWSHORT (1 << 5)
#define CWLONG (1<< 1)
// the actual port bits used for CW input
#define CWKEY1 USERP3
#define CWKEY2 USERP4

#define BIT_SDA			PD6
#define BIT_SCL 		PD5
#define	I2C_DDR			DDRD
#define	I2C_PIN			PIND

#define ADC_P_OUT	0						// Used with adc_read[] variable
#define ADC_P_IN	1						// Used with adc_read[] variable


// Defines for the Si570
#if TF3LJ_SPECIFIC							// TF3LJ specific setting for this item:
#define	DEVICE_XTAL		( 0x72353F7D )		// 114.208 * _2(24)
#else										// Normal setting for this item:
#define	DEVICE_XTAL		( 0x7248F5C2 )		// 114.285 * _2(24)
#endif
#define	DEVICE_I2C		( 0x55 )			// Default Si570 I2C address (can change per device)
#define	INCLUDE_CHECK_DSO_MAX				// Check calulated DSO freq on device max value


//#define LCD_I2C_DISPLAY	1
#define LCD_I2C_ADDRESS		0x28			// Default address for the I2C display
#define LCD_TX_DLY			5				// Display Frequency instead of Power bargraph for a defined period
											// (in units of 100ms) after switch to TX mode.  This is to enable display
											// of frequency change when switching to TX mode, using PowerSDR-IQ
#if TF3LJ_SPECIFIC							// TF3LJ specific setting for this item:
#define LCD_2ND_LINE_RX	"TF3LJ SR6.3+MOBO"	// String displayed in the second line of the LCD during RX
#else										// Normal setting for this item:
#define LCD_2ND_LINE_RX	"      SR6.3+MOBO"	// String displayed in the second line of the LCD during RX
#endif


// Defines for scrambled filter order (cmds 0x18-0x1b)
#define	BP_FLT0		0						// Band Pass filter selection
#define	BP_FLT1		1						// these values are mapped against the result of the
#define	BP_FLT2		2						// filter crossover point comparison  
#define	BP_FLT3		3						// Filter selected by writing value to output port
#define	LP_FLT0		0						// External LPF filter selection
#define	LP_FLT1		1						// these values are mapped against the result of the
#define	LP_FLT2		2						// filter crossover point comparison 
#define	LP_FLT3		3
#define	LP_FLT4		4
#define	LP_FLT5		5


// Defs for ADC routines
#define ADC_P_IN_PIN  3						// ADC MUX Port for PIn measurement
#define ADC_P_OUT_PIN 4						// ADC MUX Port for POut measurement


// Various defines for Power and SWR display and SWR Alarm
#define SWR_TRIGGER			30				// Max SWR threshold (10 x SWR, 30 = 3.0)
#define SWR_PROTECT_TIMER	200				// Timer loop value in increments of 10ms
#define P_MIN_TRIGGER 		49				// Min P out in mW for SWR trigger
#define V_MIN_TRIGGER		4				// Min Vin in 1/4096 Full Scale, for valid SWR measurement
											// (SWR = 1.0 if lower values measured)
#if TF3LJ_SPECIFIC							// TF3LJ specific setting for this item:
#define PWR_CALIBRATE		896				// Power meter calibration value
#else										// Normal setting for this item:
#define PWR_CALIBRATE		1000			// Power meter calibration value
#endif
#define PEP_PERIOD			20				// Time period for PEP measurement (in 100ms)
											// PEP_PERIOD is used with PWR_PEAK_ENVELOPE func
											// under LCD display.
#if TF3LJ_SPECIFIC							// TF3LJ specific setting for this item:
#define PWR_FULL_SCALE		25				// Bargraph Power fullscale in Watts
#else										// Normal setting for this item:
#define PWR_FULL_SCALE		4				// Bargraph Power fullscale in Watts
#endif
#define SWR_FULL_SCALE		4				// Bargraph SWR fullscale: Max SWR = Value + 1 (4=5.0)


// Rotary Encoder definitions
// Configuration of the two input pins for the Rotary Encoder, Phase A and Phase B
#define ENC_A_PORT		PORTB				// PhaseA port register
#define ENC_A_DDR		DDRB				// PhaseA port direction register
#define ENC_A_PORTIN	PINB				// PhaseA port input register
#define ENC_A_PIN		(1 << PB2)			// PhaseA port pin
#define ENC_B_PORT		PORTB				// PhaseB port register
#define ENC_B_DDR		DDRB				// PhaseB port direction register
#define ENC_B_PORTIN	PINB				// PhaseB port input register
#define ENC_B_PIN		(1 << PB3)			// PhaseB port pin

// Definitions for the Pushbutton Encoder functionality
#define ENC_PUSHB_DDR		DDRB
#define ENC_PUSHB_PORT		PORTB
#define	ENC_PUSHB_INPORT	PINB
#define	ENC_PUSHB_PIN		(1 << PB4)
#define ENC_PULSES			64			// Number of resolvable Encoder States per revolution.
											// Note that the pulses per revolution parameter is not consistent
											// for all encoders.  For some Rotary Encoders "pulses per revolution"
											// indicates the number of pulses per revolution for each of the two
											// phase outputs, while for others, this parameter indicates the total
											// number of resolvable states.  In case of the former, the total number
											// of resolvable states is 4 times the specified pulses per revolution.
#define	ENC_INCREMENTS		1000/(.11920929 * ENC_PULSES) // One kHz per revolution of Encoder
											// Frequency increments in units of
											// ~0.11920929 Hz (or 8 ~ 1 Hz)
#define ENC_PUSHB_MIN		1				// Min pushdown for valid push (x 10ms)
#define	ENC_PUSHB_MAX		10				// Min pushdown for memory save (x 10ms)
#define ENC_STORE_DISP		20				// Time to display "Memory Stored" on LCD (x 100ms)

#if ENCODER_FAST_ENABLE						// Variable speed Rotary Encoder feature
// Definitions for Variable Speed Rotary Encoder function
// The below parameters are hopefully more or less generic for any encoder, down to 32 pulses or so,
// but are optimized for a 1024 state Rotary Encoder
#define ENC_FAST_SENSE		96000/ENC_PULSES// Maximum time threshold (in steps of 1/65536 s) per click to enable fast mode
#define ENC_FAST_TRIG		ENC_PULSES/5	// Number of fast clicks to enable fast Mode
											// Make sure this does not exceed uint8_t
#define ENC_FAST_MULTIPLY	100				// Encoder click multiplier during FAST mode (max 128)
#define ENC_FAST_PATIENCE	5				// Time in 1/10th of seconds.  Time to revert to
											// normal encoder mode if no encoder activity
#endif

// Features Selection
#define	LCD_DISPLAY (1 << 0)				// LCD Display feature
#define ROTARY_ENC	(1 << 1)				// Rotary Encoder feature
#if TF3LJ_SPECIFIC							// TF3LJ specific setting for this item:
#define DEF_FEATURES		3				// Which features are default (0 = none of the above)
#else										// Normal setting for this item:
#define DEF_FEATURES		1				// Which features are default (1 = LCD Display, No encoder)
//#define DEF_FEATURES		0				// Which features are default (0 = none of the above)
#endif


//-----------------------------------------------------------------------------------------------
//*** The below are not intended as user tweakables:

// DEFS for all kinds of Flags
extern	volatile uint8_t	Status1;		// Contains various status flags

#define TX_FLAG		(1 << 0)				// Indicates status of TX/RX = TX
#define SWR_ALARM	(1 << 1)				// SWR alarm condition
#define LCD_UPDATE	(1 << 2)				// Signal an LCD update

extern	volatile uint8_t	Status2;		// Contains various status flags

#define	ENC_NEWFREQ	(1 << 0)				// Shaft Enc interrupt routine has a result
#define	ENC_FAST	(1 << 1)				// Encoder FAST mode enabled
#define ENC_STORED	(1 << 2)				// Shaft Enc pushbutton status flag "STORED"
#define	ENC_CHANGE	(1 << 3)				// Encoder debounce flag (used with variable rate enc)
#define ENC_DIR		(1 << 4)				// Encoder Direction Change
//#define ENC_RES		(1 << 6)				// Manual Encoder resolution through cmd 0x36

// Firmware changable USB serial number.
#define INCLUDE_SN		(USB_CFG_DESCR_PROPS_STRING_SERIAL_NUMBER & USB_PROP_IS_RAM)


#define	true			1
#define	false			0

#define	_2(x)			((uint32_t)1<<(x))	// Take power of 2

#define	bit_1(port,bit)	port |= _BV(bit)	// Set bit to one
#define	bit_0(port,bit)	port &= ~_BV(bit)	// Set bit to zero


typedef union {
	uint16_t		w;
	struct {
		uint8_t		b0;
		uint8_t		b1;
	};
} sint16_t;

typedef union {
	uint32_t		dw;
	struct {
		sint16_t	w0;
		sint16_t	w1;
	};
} sint32_t;

typedef union {
	uint8_t			bData[6];
	uint16_t		wData[3];
	struct {
		uint8_t		N1:5;
		uint8_t		HS_DIV:3;
		uint8_t		RFREQ_b4;				// N1[1:0] RFREQ[37:32]
		sint32_t	RFREQ;					// RFREQ[31:0]
	};
} Si570_t;

typedef struct 
{
		uint32_t	FreqXtal;				// crystal frequency[MHz] (8.24bits)
		uint32_t	Freq[10];				// Running frequency[MHz] (11.21bits)
		uint8_t		SwitchFreq;				// Which freq is in use (used with ShaftEncoder)
		uint32_t	FreqSub;				// Freq subtract value[MHz] (11.21bits)
		uint32_t	FreqMul;				// Freq multiply value (11.21bits)
		uint16_t	SmoothTunePPM;			// Max PPM value for the smooth tune
		sint16_t	FilterCrossOver[4];		// Filter cross over points [0..2] (11.5bits)
		sint16_t	TXFilterCrossOver[6];	// 6x Filter cross over points for TX filter
		uint8_t		FilterNumber[4];		// Which Band Pass filter to select at each crossover
		uint8_t		TXFilterNumber[6];		// Which TX Low Pass filter to select at each crossover
		uint32_t	BandSub[4];				// Freq Subtract values [MHz] (11.21bits) for each of
											// the 8 (BPF) Bands
		uint32_t	BandMul[4];				// Freq Multiply values [MHz] (11.21bits) for each of
											// the 8 (BPF) Bands
		uint8_t		ChipCrtlData;			// I2C addres, default 0x55 (85 dec)
		uint8_t		LCD_I2C_addr;			// I2C address for LCD display
		uint8_t		P_Min_Trigger;			// Min P out measurement for SWR trigger
		uint8_t		SWR_Protect_Timer;		// Timer loop value
		uint8_t		SWR_Trigger;			// Max SWR threshold//
		uint16_t	PWR_Calibrate;			// Power meter calibration value
		uint8_t		PWR_fullscale;			// Full Scale setting for Power Output Bargraph
		uint8_t		SWR_fullscale;			// Full Scale setting for SWR Bargraph
		uint8_t		PEP_samples;			// Number of PEP measurement samples for LCD power display
		uint16_t	Resolvable_States;		// Number of Resolvable States per Revolution
		int8_t		LCD_RX_Offset;			// Freq add/subtract value is in kHz
		uint8_t		Features;				// Select which firmware features to make active
		uint8_t		SerialNumber;			// Last char (X) of the USB device Serial Number "TF3LJ-1.X"
} var_t;


extern				var_t	R;				// Variables in Ram
extern	EEMEM 		var_t 	E;				// Default Variables in EEPROM

extern	sint16_t	replyBuf[4];			// USB Reply buffer
extern	Si570_t		Si570_Data;				// Registers 7..12 value for the Si570
register uint8_t	usbRequest asm("r6");	// usbFunctionWrite command
register uint8_t	SI570_OffLine asm("r7");// Si570 loaded, i2c open collector line high

extern	uint8_t		GetRegFromSi570(void);
extern	void		SetFreq(uint32_t freq);
extern	void		DeviceInit(void);


#define	DCO_MIN		4850					// min VCO frequency 4850 MHz
#define DCO_MAX		5670					// max VCO frequency 5670 MHz

extern	void		Si570CmdReg(uint8_t reg, uint8_t data);

extern	uint32_t	FreqSmoothTune;			// The smooth tune center frequency

#define	FilterCrossOverOn	(R.FilterCrossOver[3].b0 != 0)
//extern	void		SetFilter(void);
//extern	uint8_t 	SetFilter(uint32_t);

#define	I2C_KBITRATE	200.0				// I2C Bus speed in Kbs

register uint8_t	I2CErrors asm("r8");
extern	void		I2CSendStart(void);
extern	void		I2CSendStop(void);
extern	void		I2CSendByte(uint8_t b);
extern	void 		I2CSend0(void);
extern	void 		I2CSend1(void);
extern	uint8_t		I2CReceiveByte(void);

#if 0
#   define SWITCH_START(cmd)       switch(cmd){{
#   define SWITCH_CASE(value)      }break; case (value):{
#   define SWITCH_CASE2(v1,v2)     }break; case (v1): case(v2):{
#   define SWITCH_CASE3(v1,v2,v3)  }break; case (v1): case(v2): case(v3):{
#   define SWITCH_CASE6(v1,v2,v3,v4,v5,v6)  }break; case (v1): case(v2): case(v3): case(v4): case(v5): case(v6):{
#   define SWITCH_DEFAULT          }break; default:{
#   define SWITCH_END              }}
#else
#   define SWITCH_START(cmd)       {uchar _cmd = cmd; if(0){
#   define SWITCH_CASE(value)      }else if(_cmd == (value)){
#   define SWITCH_CASE2(v1,v2)     }else if(_cmd == (v1) || _cmd == (v2)){
#   define SWITCH_CASE3(v1,v2,v3)  }else if(_cmd == (v1) || _cmd == (v2) || (_cmd == v3)){
#   define SWITCH_CASE6(v1,v2,v3,v4,v5,v6)  }else if(_cmd == (v1) || _cmd == (v2) || _cmd == (v3) || _cmd == (v4) || _cmd == (v5) || _cmd == (v6)){
#   define SWITCH_CASE7(v1,v2,v3,v4,v5,v6,v7)  }else if(_cmd == (v1) || _cmd == (v2) || _cmd == (v3) || _cmd == (v4) || _cmd == (v5) || _cmd == (v6) || _cmd == (v7)){

#   define SWITCH_DEFAULT          }else{
#   define SWITCH_END              }}
#endif

extern	void 		InitADC(void);
extern	void 		pollADC(void);
extern	uint16_t 	ReadADC(uint8_t mux/*, uint8_t trig*/);
extern	void 		Test_SWR(void);
extern	uint16_t 	measured_Power(uint16_t voltage);

extern	uint16_t	measured_SWR;			// SWR value x 10, in unsigned int format
extern	uint32_t 	adc_read[2];			// last read P_out and P_in values

extern void			lcd_initialize(void);	// Initialize the LCD
extern void			lcd_display(void);		// Display stuff on a LCD 
extern void 		lcd_command(uint8_t);	// Send command to LCD
extern void 		lcd_data(uint8_t);		// Send raw data to LCD


extern void			lcdProgressBar(uint16_t, uint16_t, uint8_t);// Draw a bargraph on LCD
extern void			lcd_bargraph_init(void);// Load the custom bargraph charaters to LCD

extern	void 		shaftEncoderInit(void);	// Initialize the Rotary Encoder
extern	void 		encoder_scan(void);		// Scan the Rotary Encoder

#endif
