//************************************************************************
//**
//** Project......: USB controller firmware for the Softrock 6.3 SDR,
//**                   enhanced with the 9V1AL Motherboard v 3.6, an
//**                   external LPF bank and other essentials to create
//**                   an all singing and all dancing HF SDR amateur
//**                   radio transceiver
//**
//** Platform.....:    ATmega168 @ 16.0 MHz
//**
//**                   Print to a 16x2 I2C driven LCD display
//**
//** Initial version.: 2009-09-08, Loftur Jonasson, TF3LJ
//**                   Check the main.c file
//**
//**                   Last updates, PSDR-IQ LCD offset:
//**                   2010-04-20, Loftur Jonasson, TF3LJ
//**
//************************************************************************

#include	<stdio.h>
#include	"lcd_i2c.h"
#include	"_main.h"
//#include	<rprintf.h>								// AVRLIB (not AVRLibc) functions


//
//-----------------------------------------------------------------------------
//			Display stuff on a 16x2 LCD, including a couple of bargraphs 
//-----------------------------------------------------------------------------
//
void lcd_display(void)
{       	
	char buf[20];								// Print buffer

	static uint8_t	lcd_print_count = 0;		// This value is incremented each time the
												// function is executed, at 10 it is reset to 0 
												// To minimize delay to the execution
												// of other services, the LCD print
												// is cut into several different
												// pieces, where each piece is only executed
												// when the lcd_print_count has the correct value
												// 9 = Display Power
												// 4 = Display SWR
	

	static uint16_t pow_avg[PEP_PERIOD];		//PEP measurement. Highest value in buffer shown

	static uint8_t	tx_display_delay;			// Used to show the frequency display for
												// a moment, after switchover to TX

	lcd_command(LCD_DISP_ON);					// Blinking block cursor OFF

	// Variables used for print formatting, to emulate a floating point printf
	// AVRLIB floating point printf formatting would be very costly in size
	uint16_t swr, swr_hundredths, swr_tenths, pow_tot=0, pow, pow_dw, pow_cw, pow_mw;
	
	if ((lcd_print_count == 9) && (Status1 & TX_FLAG))	// Display Power
	{
		//
		// Prepare Power readout
		//

		pow_tot = measured_Power(adc_read[0]);		// Power in mW (max 65535mW)

		static uint8_t i = 0, j;

		if (R.PEP_samples > PEP_PERIOD) R.PEP_samples = PEP_PERIOD;// Safety measure

		pow_avg[i] = pow_tot;						// Store value in a 10 unit ringbuffer
		i++;
		if ((i == R.PEP_samples) || (i == PEP_PERIOD)) i = 0;// Whichever is smaller

		pow_tot = 0;
		for (j = 0; j < R.PEP_samples; j++)			// Retrieve the largest value out of the measured window
		{
			if (pow_avg[j] > pow_tot) pow_tot = pow_avg[j];
		}
	}
	#if !DEBUG_1LN									// First line of LCD not used for Debug
	//
	//-------------------------------------------------------
	// Display stuff in first line of LCD
	//-------------------------------------------------------
	//
	lcd_gotoxy(0,0);								// First char, first line

	//-------------------------------------------------------
	// Display if new Frequency is stored by Shaft Encoder func
	//-------------------------------------------------------
	if (Status2 & ENC_STORED)
	{
		lcd_puts("Memory Stored   ");
		static int count = 0;						// Display for ENC_STORE_DISP * 100ms
		count++;
		if(count>=ENC_STORE_DISP)
		{
			Status2 &= ~ENC_STORED;					// We're done displaying "Memory Stored"
			Status2 |= ENC_NEWFREQ;					// Signal for an LCD frequency update
			count = 0;
		}
	}

	//-------------------------------------------------------
	// Display stuff while in Transmit "TX" mode
	//
	// Power Output Bargraph
	//-------------------------------------------------------

	if ((Status1 & TX_FLAG)&&(lcd_print_count == 9))	// Display Power
	{
		
		if (tx_display_delay<LCD_TX_DLY) tx_display_delay++;

		if(tx_display_delay==LCD_TX_DLY)				// TX delay timer has matured, display TX stuff
		{
			// progress, maxprogress, len
			lcdProgressBar(pow_tot/100, R.PWR_fullscale*10, 9);

			pow = pow_tot/1000; 						// Watts
			pow_mw = pow_tot - pow * 1000;				// milliWatts
			pow_cw = pow_mw/10;							// centiWatts
			pow_dw = pow_cw/10;							// deciWatts

			if (R.PEP_samples == 1)						// No PEP indication
				lcd_puts("P ");
			else
				lcd_puts("Pp");							// PEP indication

			if (pow<10)									// Less than 10 watts
			{
				sprintf(buf,"%u.%02uW", pow, pow_cw);
				lcd_puts(buf);
			}
			else										// Over 10 watts
			{
				sprintf(buf,"%2u.%01uW", pow, pow_dw);
				lcd_puts(buf);
			}
		}
	}

	//-------------------------------------------------------
	// Display stuff while in Receive "RX" mode
	//
	// Display the Si570 frequency (divided by 4) on the LCD:
	//-------------------------------------------------------
	if ((tx_display_delay<LCD_TX_DLY) || (!(Status1 & TX_FLAG)))
	// This is either RX or TX delay timer has not matured yet
	{
		uint32_t dispFreq;
		dispFreq = (double) R.Freq[0] * 1000000.0 / _2(23);	// R.Freq / 2^21 / 4 = Lo frequency

		if(!(Status1 & TX_FLAG))							// Display a fixed frequency offset during RX only.
		{
			dispFreq = dispFreq + R.LCD_RX_Offset * 1000; 	// 1000.0 * _2(23);
		}

		uint16_t fMHz = dispFreq/1000000;
		uint32_t fHz = dispFreq - fMHz * 1000000;

		sprintf(buf,"%3u.%06luMHz RX", fMHz,fHz);
		lcd_puts(buf);
	}	

	if(!(Status1 & TX_FLAG)) tx_display_delay=0; 	// Clear on RX

	#endif//DEBUG_1LN

	
	#if !DEBUG_2LN									// Second line of LCD not used for Debug
	//
	//-------------------------------------------------------
	// Display stuff in second line of LCD
	//-------------------------------------------------------
	//
	lcd_gotoxy(0,1);								// First char, second line


	//-------------------------------------------
	// Display stuff while in Transmit "TX" mode
	//-------------------------------------------
	if (Status1 & TX_FLAG)
	{
		if (lcd_print_count == 4)							// Display SWR
		{

			//--------------------------------------------
			// Display SWR
			//--------------------------------------------

			// Prepare SWR readout
			swr = measured_SWR/100;					// SWR
			swr_hundredths = measured_SWR - swr*100;// sub decimal point, 1/100 accuracy
			swr_tenths = swr_hundredths/10;			// sub decimal point, 1/10 accuracy

			// progress, maxprogress, len
			lcdProgressBar(measured_SWR - 100, R.SWR_fullscale*100, 9);

			lcd_puts("SWR");							// Print string from flash rom

			if(swr<10)									// For SWR under 10:1
			{
				sprintf(buf,"%u.", swr);				// Print the super decimal portion of the SWR
				lcd_puts(buf);
	
				if(Status1 & SWR_ALARM)					// SWR Alarm flag set
				{
					lcd_data(swr_tenths+0x30);			// Print the sub-decimal value of the SWR, single digit precision
														// (a lean byte saving way to print a single decimal)
		
					lcd_data('A');						// A in SWRA indication
					lcd_command(LCD_DISP_ON_BLINK);		// Blinking block cursor ON
				}
				else
				{
					sprintf(buf,"%02u", swr_hundredths);// Print the sub-decimal value of the SWR, double digit precision
					lcd_puts(buf);
				}
			}
			else										// SWR is 10 or higher:
			{
				if(swr >= 100)							// SWR more than 99
					lcd_puts("-- ");
				else
				{
					sprintf(buf,"%2u.", swr);			// Print the super decimal portion of the SWR
					lcd_puts(buf);
				}
				if(Status1 & SWR_ALARM)					// SWR Alarm flag set
				{
					lcd_data('A');						// A in SWRA indication
					lcd_command(LCD_DISP_ON_BLINK);		// Blinking block cursor ON
				}
				else
				{
					lcd_data(swr_tenths+0x30);			// Print the sub-decimal value of the SWR, single digit precision
				}										// (a lean byte saving way to print a single decimal)
			}
		
			lcd_gotoxy(15,1);							// In case blinking cursor, put it on "A" in "SWR Alarm"
		}
	}
	//-------------------------------------------
	// Display stuff when in Receive ("RX") mode
	//-------------------------------------------
	else
	{
		lcd_puts(LCD_2ND_LINE_RX);

		for (uint8_t j = 0; j < R.PEP_samples; j++)			// Clear PEP storage
		{
			pow_avg[j]=0;
		}
	}


	#endif//DEBUG_2LN

	lcd_print_count++;
	if (lcd_print_count >= 10) lcd_print_count = 0;
}	

