//************************************************************************
//**
//** Project......: USB controller firmware for the Softrock 6.3 SDR,
//**                   enhanced with the 9V1AL Motherboard v 3.6, an
//**                   external LPF bank and other essentials to create
//**                   an all singing and all dancing HF SDR amateur
//**                   radio transceiver
//**
//** Platform.....:    ATmega168 @ 16.0 MHz
//**
//**                   Calculate SWR and provide SWR Alarm functionality
//**                   Calculate Power output
//**
//** Initial version.: 2009-09-23, Loftur Jonasson, TF3LJ
//**                   Check the main.c file
//**
//************************************************************************

#include "_main.h"


/* ------------------------------------------------------------------------- */
/* ------------------ SWR calculations and SWR protect---------------------- */
/* ------------------------------------------------------------------------- */
void Test_SWR(void)
{
static uint8_t second_pass=0, timer=0;

	//-------------------------------------------------------------
	// Calculate SWR
	//-------------------------------------------------------------
	if (adc_read[ADC_P_IN] < V_MIN_TRIGGER)	// Too little for valid measurement, SWR = 1.0
		measured_SWR = 100;
				 
	else if (adc_read[ADC_P_IN] >= adc_read[ADC_P_OUT])	// Quick check for an invalid result 
		measured_SWR = 9990; 				// Infinite SWR gives a reading of 99.9
	else 									//	Standard SWR formula multiplied by 100
		measured_SWR = 
			(100 * (adc_read[ADC_P_OUT]+adc_read[ADC_P_IN]))
			/
			(adc_read[ADC_P_OUT]-adc_read[ADC_P_IN]);
	if (measured_SWR > 9990) measured_SWR = 9990;	


	//-------------------------------------------------------------
	// SWR Alarm function
	// If PTT is keyed, key the PTT2 line according to SWR status
	//-------------------------------------------------------------
	// On measured Power output and high SWR, force clear RXTX2 and seed timer

	// If PTT is keyed, key the PTT2 line according to SWR status
	if (PORTD & RXTX)
	{
		// Compare power measured (in mW) with min Trigger value
		// On measured Power output and high SWR, force clear RXTX2 and seed timer
		if ((measured_Power(adc_read[ADC_P_OUT]) > R.P_Min_Trigger) 	
			&& (measured_SWR > 10*R.SWR_Trigger)) 
		{
			if (!second_pass)	// First time, set flag, no action yet
				second_pass++;
			else				// There have been two or more consecutive measurements
								// with high SWR, take action
			{
				PORTD = PORTD & ~RXTX2;
				Status1 |= SWR_ALARM;
				timer = R.SWR_Protect_Timer;
			}
		}
		// If SWR OK and timer has been zeroed, set the PTT2 line
		else 
		{
			if (timer == 0)
			{
				PORTD = PORTD | RXTX2;	// Set PTT2 line
				Status1 &= ~SWR_ALARM;
				second_pass = 0;		// clear flag
			}
			else
				timer--;
		} 
	}
	else PORTD = PORTD & ~RXTX2;		// Clear PTT2 line
}



/* ------------------------------------------------------------------------- */
/* ------- Read Power and SWR meter, update Trigger values in EEPROM ------- */
/* ------------------------------------------------------------------------- */
uint16_t ReadADC(uint8_t mux/*, uint8_t trig*/)
{
uint16_t x = 0;

/*	// If value field > 0, then update EEPROM settings
	if (trig > 0)
	{
		switch (mux) {
			case 0:		// Min P out measurement for SWR trigger
				R.P_Min_Trigger = trig;
				eeprom_write_block(&trig, &E.P_Min_Trigger, sizeof (E.P_Min_Trigger));
				break;
	
			case 1:		// Timer loop value
				R.SWR_Protect_Timer = trig;
				eeprom_write_block(&trig, &E.SWR_Protect_Timer, sizeof (E.SWR_Protect_Timer));
				break;
			case 2:		// Max SWR threshold
				R.SWR_Trigger = trig;
				eeprom_write_block(&trig, &E.SWR_Trigger, sizeof (E.SWR_Trigger));
				break;
		}
	}
*/
	// Read Power out,Power in or calculated SWR
	if (mux < 2) 
		x = adc_read[mux];  
	else 
		x = measured_SWR;
	return x;
}


//
//-----------------------------------------------------------------------------------------
// 				Convert AD reading into "Measured Power in milliWatts"
//-----------------------------------------------------------------------------------------
//

// A simplified integer aritmetic version, still with decent accuracy
// (the return value overflows above 65W max)
// (comparison Ref 11604 bytes)

uint16_t measured_Power(uint16_t voltage)
{
	// All standard stuff
	// normalise the measured value from the VSWR bridge
	// Reference voltage is 5V, 
	// diode offset ~ .15V
	// R.PWR_Calibrate = Power meter calibration value
	uint32_t measured_P; 
	

	// The below not used if opamp/diode-comp bridge
	//if (voltage > 0) voltage = voltage/0x10 + 82;				// If no input voltage, then we do not add offset voltage
																// as this would otherwise result in a bogus minimum power
																// reading
	voltage = voltage/0x10;										// voltage is a MSB adjusted 12 bit value, therefore 
																// dividing by 10 does not lose any information
																// 4096 = 5V,
																// 82 = 100mV, compensating for schottky diode loss
	// Formula roughly adjusted for the ratio in the SWR bridge
	measured_P = (uint32_t)voltage * R.PWR_Calibrate/77;
	measured_P = (measured_P*measured_P)/50000;
	return (uint16_t) measured_P;								// Return power in mW
}

